test_that("geom_vline_year handles basic functionality", {
  test_dates <- data.frame(
    date = as.Date("2023-12-01") + 0:60
  )
  
  # Test default year break (January 1)
  p <- ggplot(test_dates, aes(x = date)) +
    geom_bar() +
    geom_vline_year()
  
  expect_no_error(p)
  expect_s3_class(p, "ggplot")
})

test_that("geom_vline_year handles datetime data", {
  test_datetime <- data.frame(
    datetime = as.POSIXct("2023-12-15") + seq(0, 86400 * 30, by = 3600)
  )
  
  p <- ggplot(test_datetime, aes(x = datetime)) +
    geom_bar() +
    geom_vline_year()
  expect_no_error(p)
  expect_s3_class(p, "ggplot")
})

test_that("geom_hline_year works with flipped coordinates", {
  test_dates <- data.frame(
    date = as.Date("2023-12-01") + 0:60
  )
  
  p <- ggplot(test_dates, aes(y = date)) +
    geom_bar() +
    geom_hline_year()
  expect_no_error(p)
  expect_s3_class(p, "ggplot")
})


test_that("calc_visible_years handles date ranges correctly", {
  # Test basic date range spanning multiple years
  range <- as_date(c("2023-06-01", "2025-06-01"))
  years <- calc_visible_years(as.numeric(range), is_date = TRUE)
  expect_equal(years, as.numeric(as_date(c("2024-01-01", "2025-01-01"))))
  
  # Test date range within single year
  range <- as_date(c("2023-03-01", "2023-09-01"))
  years <- calc_visible_years(as.numeric(range), is_date = TRUE)
  expect_equal(length(years), 0) # No year breaks visible
  
  # Test custom year break (fiscal year starting April 1)
  range <- as_date(c("2023-01-01", "2024-12-31"))
  years <- calc_visible_years(as.numeric(range), is_date = TRUE, year_break = "04-01")
  expect_equal(years, as.numeric(as_date(c("2023-04-01", "2024-04-01"))))
})

test_that("calc_visible_years handles datetime ranges correctly", {
  # Test datetime range spanning multiple years
  range <- as_datetime(c("2023-12-31 18:00:00", "2024-01-01 06:00:00"))
  years <- calc_visible_years(as.numeric(range), is_date = FALSE)
  expect_equal(years, as.numeric(as_datetime("2024-01-01 00:00:00")))
  
  # Test datetime range within single year
  range <- as_datetime(c("2023-06-01 00:00:00", "2023-12-30 23:59:59"))
  years <- calc_visible_years(as.numeric(range), is_date = FALSE)
  expect_equal(length(years), 0) # No year breaks visible
})

test_that("calc_visible_years handles edge cases", {
  # Test range exactly at year boundaries, years are inclusive at borders
  range <- as_date(c("2023-01-01", "2024-01-01"))
  years <- calc_visible_years(as.numeric(range), is_date = TRUE)
  expect_equal(years, as.numeric(as_date(c("2023-01-01", "2024-01-01"))))
  
  # Test very short range around new year
  range <- as_date(c("2023-12-31", "2024-01-01"))
  years <- calc_visible_years(as.numeric(range), is_date = TRUE)
  expect_equal(years, as.numeric(as_date("2024-01-01")))
})