% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggseqrfplot.R
\name{ggseqrfplot}
\alias{ggseqrfplot}
\title{Relative Frequency Sequence Plot}
\usage{
ggseqrfplot(
  seqdata,
  k = floor(nrow(seqdata)/10),
  diss,
  sortv = NULL,
  border = FALSE,
  ylab = NULL,
  yaxis = TRUE,
  box.color = NULL,
  box.fill = NULL,
  box.alpha = NULL,
  which.plot = "both",
  quality = TRUE
)
}
\arguments{
\item{seqdata}{State sequence object (class \code{stslist}) created with the \code{\link[TraMineR:seqdef]{TraMineR::seqdef}} function.}

\item{k}{integer specifying the number of frequency groups}

\item{diss}{pairwise dissimilarities between sequences in \code{seqdata} (see \code{TraMineRextras::seqdist})}

\item{sortv}{optional sorting variable that may be used to compute the frequency groups. If  \code{NULL} (default), an MDS is used. Ties are randomly ordered.}

\item{border}{if \code{TRUE} bars of index plot are plotted with black outline; default is \code{FALSE} (also accepts \code{NULL})}

\item{ylab}{character string specifying title of y-axis. If \code{NULL} axis title is "Frequency group"}

\item{yaxis}{Controls if a y-axis is plotted. When set as \code{TRUE}, index of frequency groups is displayed.}

\item{box.color}{specifies color of boxplot borders; default is "black}

\item{box.fill}{specifies fill color of boxplots; default is "white"}

\item{box.alpha}{specifies alpha value of boxplot fill color; default is 1}

\item{which.plot}{character string specifying which components of relative frequency sequence plot should be displayed.
Default is \code{"both"}. If set to \code{"medoids"} only the index plot of medoids is shown. If \code{"diss.to.med"} only
the box plots of the group-specific distances to the medoids are shown.}

\item{quality}{specifies if representation quality is shown as figure caption; default is \code{TRUE}}
}
\value{
A relative frequency sequence plot using \code{\link[ggplot2]{ggplot}}.
}
\description{
Function for rendering sequence index plots with \code{\link[ggplot2]{ggplot2}} instead of base
R's \code{\link[base]{plot}} function that is used by \code{\link[TraMineRextras:seqplot.rf]{TraMineRextras::seqplot.rf}}.
Note that \code{ggseqrfplot} uses \code{\link[patchwork]{patchwork}} to combine the different components of the plot.
For further details on relative frequency sequence plots see documentation of \code{\link[TraMineRextras:seqplot.rf]{TraMineRextras::seqplot.rf}}.
}
\details{
Note that an identical function call might produce different results
if \code{sortv} has ties, because the sequences are sorted randomly within
each set of ties (see \code{\link[base]{rank}}; \code{ties.method="random"})
}
\examples{
## From TraMineRextras::seqplot.rf
library(TraMineR)
library(TraMineRextras)
library(patchwork)
library(ggplot2)

## Defining a sequence object with the data in columns 10 to 25
## (family status from age 15 to 30) in the biofam data set
data(biofam)
biofam.lab <- c("Parent", "Left", "Married", "Left+Marr",
  "Child", "Left+Child", "Left+Marr+Child", "Divorced")

## Here, we use only 100 cases selected such that all elements
## of the alphabet be present.
## (More cases and a larger k would be necessary to get a meaningful example.)
biofam.seq <- seqdef(biofam[501:600, ], 10:25, labels = biofam.lab)
diss <- seqdist(biofam.seq, method = "LCS")


## Using 12 groups and default MDS sorting
## ... with TraMineRextras::seqplot.rf
seqplot.rf(biofam.seq, diss = diss, k = 12)

## ... with ggseqrfplot
ggseqrfplot(biofam.seq, diss = diss, k = 12)


## With a user specified sorting variable
## Here time spent in parental home
parentTime <- seqistatd(biofam.seq)[, 1]

## ... with TraMineRextras::seqplot.rf
set.seed(123)
seqplot.rf(biofam.seq, diss = diss, k = 12,
  sortv = parentTime, main = "Sorted by parent time")

## ... with ggseqrfplot
set.seed(123)
ggseqrfplot(biofam.seq, diss = diss, k = 12, sortv = parentTime) +
  plot_annotation(title = "Sorted by parent time",
     theme = theme(plot.title = element_text(hjust = 0.5, size = 18)))
}
\author{
Marcel Raab
}
