#' Issue a geoparser query
#'
#' The function calls the \code{geoparser.io} API which identifies places
#' mentioned in the input text (in English), disambiguates those places, and
#' returns data about the places found in the text.
#'
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#'
#' @param text_input A vector of strings, where each size must be smaller than 8 KB. See
#'  \code{nchar(text_input, type = "bytes")}.
#' @param key Your \code{geoparser.io} key.
#'
#' @details To get an API key, you need to register at
#' \url{https://geoparser.io/pricing.html}.
#' With an hobbyist account, you can make up to 1,000 calls a month to the API.
#' For ease of use, save your API key as an environment variable as described
#' at \url{https://stat545-ubc.github.io/bit003_api-key-env-var.html}.
#'
#' The package will conveniently look for your API key using
#' \code{Sys.getenv("GEOPARSER_KEY")} so if your API key is an environment
#' variable called "\code{GEOPARSER_KEY}" you don't need to input it manually.
#'
#' \code{Geoparser.io} works best on complete sentences in English. If you
#' have a very short text, such as a partial address like
#' "\code{Auckland New Zealand}," you probably want to use a geocoder tool
#' instead of a geoparser. In R, you can use the
#' \href{https://cran.r-project.org/package=opencage}{opencage package} for
#' geocoding!
#'
#' @return A list of 2 \code{data.frame}s (\code{dplyr::tbl_df}s). The first
#' \code{data.frame} is called \code{properties} and contains:
#' \itemize{
#' \item \code{apiVersion} the version of the API currently in use
#' \item \code{source} the source of authority for the results
#' \item \code{id} the unique id of the query
#' \item \code{text_md5} MD5 hash of the text that was sent to the API.
#' }
#' The second \code{data.frame} contains the results and is called results.
#' If there are no results it is totally empty, otherwise:
#' \itemize{
#' \item \code{country} \href{https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2}{ISO-3166 2-letter country code}
#' for the country in which this place is located or \code{NULL} for features
#' outside any sovereign territory.
#' \item \code{confidence} A confidence score produced by the place
#' name disambiguation algorithm. \emph{Currently returns a placeholder value;
#' subject to change}.
#' \item \code{name} Best name for the specified location, with a
#' preference for official/short name forms (e.g., "\code{New York}" over
#' "\code{NYC}," and "\code{California}" over "\code{State of California}"),
#' which may be different from exactly what appears in the text.
#' \item \code{admin1} A code representing the state/province-level
#' administrative division containing this place. (From \code{GeoNames.org}:
#' \emph{"Most adm1 are FIPS codes. ISO codes are used for US, CH, BE and ME.
#' UK and Greece are using an additional level between country and FIPS code.
#' The code '\code{00}' stands for general features where no specific adm1 code
#' is defined.}")
#' \item \code{type} A text description of the geographic feature
#' type — \emph{see \url{http://www.geonames.org/} for a complete list. Subject to
#' change}.
#' \item \code{geometry.type} Type of the geographical feature, e.g.
#' "\code{Point}".
#' \item \code{longitude} Longitude.
#' \item \code{latitude} Latitude.
#' \item \code{reference1} Start (index of the first character in the place
#' reference) --  \emph{each reference to this place name found in the input
#' text is on one distinct line}.
#' \item \code{reference2} End (index of the first character after the place
#' reference) --  \emph{each reference to this place name found in the input
#' text is on one distinct line}.
#' \item \code{text_md5} MD5 hash of the text that was sent to the API.
#' }
#' @references \href{https://geoparser.io/docs.html}{\code{Geoparser.io}
#' documentation}
#' @export
#'
#' @examples
#' \dontrun{
#' geoparser_q(text_input = "I was born in Vannes but I live in Barcelona.")
#' geoparser_q(text_input = c("I was born in Vannes but I live in Barcelona.",
#' "France is the most beautiful place in the world."))
#' }
geoparser_q <- function(text_input,
                        key = geoparser_key()){

  # check arguments
  geoparser_query_check(text_input, key)

  result <- lapply(text_input, total, key = key)


  list(results = do.call("rbind", lapply(result, "[[", "results")),
       properties = do.call("rbind", lapply(result, "[[", "properties")))

}
