#' Visualize observed densities
#'
#' @description `vis_obs_dens()` visualizes actual and predicted counts 
#' generated by [get_obs_dens()].
#'
#' @param actual_data actual data in the form of `hyperframe$column`.
#' @param dens_1 im object that represents densities
#' @param dens_2 density 2 (if any). By default, `NA`.
#' @param dens_3 density 3 (if any). By default, `NA`.
#' @param color_actual color for actual data. By default, `"darkgrey"`.
#' @param color_dens_1 color for actual data. By default, `"#D55E00"`.
#' @param color_dens_2 color for actual data. By default, `"#0072B2"`.
#' @param color_dens_3 color for actual data. By default, `"#009E73"`.
#' @param time_unit x-axis label of the output
#' 
#' @returns list of dataframe and two ggplot objects for comparison of densities and residuals

vis_obs_dens <- function(actual_data,
                         dens_1, dens_2 = NA, dens_3 = NA,
                         color_actual = "darkgrey",
                         color_dens_1 = "#D55E00", 
                         color_dens_2 = "#0072B2", 
                         color_dens_3 = "#009E73",
                         time_unit) {
  
  # Obtain the actual counts 
  actual_counts <- unlist(purrr::map(actual_data, function(x) x$n))
  
  if(is.na(dens_2)[1]){
    
    # Obtain the predicted counts
    predicted_counts <- dens_1$estimated_counts
    
    # Construct dataset
    plot_data <- data.frame(time = c(1:length(actual_counts)), 
                            actual_counts = actual_counts, 
                            predicted_counts = predicted_counts)
    
    plot_compare <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_line(aes(x = time, y = actual_counts), color = color_actual, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts), color = color_dens_1, size = 0.6) +
      theme_bw() + labs(title = "Actual vs. Predicted Counts", x = time_unit, y = "Count") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    plot_residual <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed") +
      ggplot2::geom_line(aes(x = time, y = predicted_counts - actual_counts), color = color_dens_1, size = 0.6) +
      theme_bw() + labs(title = "Residual Plot", x = time_unit, y = "Predicted - Actual Counts") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    return(list(plot_data = plot_data, plot_compare = plot_compare, plot_residual = plot_residual))
    
  } else if (is.na(dens_3)[1]) {
    
    # Obtain the predicted counts
    predicted_counts <- dens_1$estimated_counts
    predicted_counts_2 <- dens_2$estimated_counts
    
    # Construct dataset
    plot_data <- data.frame(time = c(1:length(actual_counts)), 
                            actual_counts = actual_counts, 
                            predicted_counts = predicted_counts,
                            predicted_counts_2 = predicted_counts_2)
    
    plot_compare <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_line(aes(x = time, y = actual_counts), color = color_actual, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts), color = color_dens_1, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_2), color = color_dens_2, size = 0.6) +
      theme_bw() + labs(title = "Actual vs. Predicted Counts", x = time_unit, y = "Count") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    plot_residual <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed") +
      ggplot2::geom_line(aes(x = time, y = predicted_counts - actual_counts), color = color_dens_1, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_2 - actual_counts), color = color_dens_2, size = 0.6) +
      theme_bw() + labs(title = "Residual Plot", x = time_unit, y = "Predicted - Actual Counts") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    return(list(plot_data = plot_data, plot_compare = plot_compare, plot_residual = plot_residual))
    
  } else {
    
    # Obtain the predicted counts
    predicted_counts <- dens_1$estimated_counts
    predicted_counts_2 <- dens_2$estimated_counts
    predicted_counts_3 <- dens_3$estimated_counts
    
    # Construct dataset
    plot_data <- data.frame(time = c(1:length(actual_counts)), 
                            actual_counts = actual_counts, 
                            predicted_counts = predicted_counts,
                            predicted_counts_2 = predicted_counts_2,
                            predicted_counts_3 = predicted_counts_3)
    
    plot_compare <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_line(aes(x = time, y = actual_counts), color = color_actual, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts), color = color_dens_1, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_2), color = color_dens_2, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_3), color = color_dens_3, size = 0.6) +
      theme_bw() + labs(title = "Actual vs. Predicted Counts", x = time_unit, y = "Count") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    plot_residual <- plot_data %>%
      ggplot2::ggplot() +
      ggplot2::geom_hline(yintercept = 0, linetype = "dashed") +
      ggplot2::geom_line(aes(x = time, y = predicted_counts - actual_counts), color = color_dens_1, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_2 - actual_counts), color = color_dens_2, size = 0.6) +
      ggplot2::geom_line(aes(x = time, y = predicted_counts_3 - actual_counts), color = color_dens_3, size = 0.6) +
      theme_bw() + labs(title = "Residual Plot", x = time_unit, y = "Predicted - Actual Counts") +
      theme(plot.title = element_text(hjust = 0.5, face = "bold"))
    
    return(list(plot_data = plot_data, plot_compare = plot_compare, plot_residual = plot_residual))
    
  }
  
}