% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdal_helpers.R
\name{addFilesInZip}
\alias{addFilesInZip}
\title{Create/append to a potentially Seek-Optimized ZIP file (SOZip)}
\usage{
addFilesInZip(
  zip_file,
  add_files,
  overwrite = FALSE,
  full_paths = TRUE,
  sozip_enabled = NULL,
  sozip_chunk_size = NULL,
  sozip_min_file_size = NULL,
  num_threads = NULL,
  content_type = NULL,
  quiet = FALSE
)
}
\arguments{
\item{zip_file}{Filename of the ZIP file. Will be created if it does not
exist or if \code{overwrite = TRUE}. Otherwise will append to an existing file.}

\item{add_files}{Character vector of one or more input filenames to add.}

\item{overwrite}{Logical scalar. Overwrite the target zip file if it already
exists.}

\item{full_paths}{Logical scalar. By default, the full path will be stored
(relative to the current directory). \code{FALSE} to store just the name of a
saved file (drop the path).}

\item{sozip_enabled}{String. Whether to generate a SOZip index for the file.
One of \code{"AUTO"} (the default), \code{"YES"} or \code{"NO"} (see Details).}

\item{sozip_chunk_size}{The chunk size for a seek-optimized file.
Defaults to 32768 bytes. The value is specified in bytes, or K and M
suffix can be used respectively to specify a value in kilo-bytes or
mega-bytes. Will be coerced to string.}

\item{sozip_min_file_size}{The minimum file size to decide if a file
should be seek-optimized, in \code{sozip_enabled="AUTO"} mode. Defaults to
1 MB byte. The value is specified in bytes, or K, M or G suffix can be used
respectively to specify a value in kilo-bytes, mega-bytes or giga-bytes.
Will be coerced to string.}

\item{num_threads}{Number of threads used for SOZip generation. Defaults to
\code{"ALL_CPUS"} or specify an integer value (coerced to string).}

\item{content_type}{String Content-Type value for the file. This is stored
as a key-value pair in the extra field extension 'KV' (0x564b) dedicated to
storing key-value pair metadata.}

\item{quiet}{Logical scalar. \code{TRUE} for quiet mode, no progress messages
emitted. Defaults to \code{FALSE}.}
}
\value{
Logical indicating success (invisible \code{TRUE}).
An error is raised if the operation fails.
}
\description{
\code{addFilesInZip()} will create new or open existing ZIP file, and
add one or more compressed files potentially using the seek optimization
extension. This function is basically a wrapper for \code{CPLAddFileInZip()}
in the GDAL Common Portability Library, but optionally creates a new ZIP
file first (with \code{CPLCreateZip()}). It provides a subset of functionality
in the GDAL \code{sozip} command-line utility
(\url{https://gdal.org/en/stable/programs/sozip.html}). Requires GDAL >= 3.7.
}
\details{
A Seek-Optimized ZIP file (SOZip) contains one or more compressed files
organized and annotated such that a SOZip-aware reader can perform very
fast random access within the .zip file
(see \url{https://github.com/sozip/sozip-spec}).
Large compressed files can be accessed directly from SOZip without prior
decompression. The .zip file is otherwise fully backward compatible.

If \code{sozip_enabled="AUTO"} (the default), a file is seek-optimized only if
its size is above the values of \code{sozip_min_file_size} (default 1 MB) and
\code{sozip_chunk_size} (default \code{32768}).
In \code{"YES"} mode, all input files will be seek-optimized. In \code{"NO"} mode, no
input files will be seek-optimized. The default can be changed with the
\code{CPL_SOZIP_ENABLED} configuration option.
}
\note{
The \code{GDAL_NUM_THREADS} configuration option can be set to \code{ALL_CPUS} or an
integer value to specify the number of threads to use for SOZip-compressed
files (see \code{\link[=set_config_option]{set_config_option()}}).

SOZip can be validated with:

\if{html}{\out{<div class="sourceCode">}}\preformatted{vsi_get_file_metadata(zip_file, domain="ZIP")
}\if{html}{\out{</div>}}

where \code{zip_file} uses the /vsizip/ prefix.
}
\examples{
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
zip_file <- file.path(tempdir(), "storml_lcp.zip")

# Requires GDAL >= 3.7
if (gdal_version_num() >= gdal_compute_version(3, 7, 0)) {
  addFilesInZip(zip_file, lcp_file, full_paths = FALSE,
                sozip_enabled = "YES", num_threads = 1)

  print("Files in zip archive:")
  print(unzip(zip_file, list = TRUE))

  # Open with GDAL using Virtual File System handler '/vsizip/'
  # see: https://gdal.org/en/stable/user/virtual_file_systems.html#vsizip-zip-archives
  lcp_in_zip <- file.path("/vsizip", zip_file, "storm_lake.lcp")
  print("SOZip metadata:")
  print(vsi_get_file_metadata(lcp_in_zip, domain = "ZIP"))

  ds <- new(GDALRaster, lcp_in_zip)
  ds$info()
  ds$close()
  \dontshow{vsi_unlink(zip_file)}
}
}
\seealso{
\code{\link[=vsi_get_file_metadata]{vsi_get_file_metadata()}}
}
