% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fuzzy_numerical_operations.R
\name{fsr_numerical_operations}
\alias{fsr_numerical_operations}
\alias{spa_avg_degree}
\alias{spa_ncomp}
\alias{spa_area}
\alias{spa_perimeter}
\alias{spa_length}
\title{Compute fuzzy numerical operations}
\usage{
spa_avg_degree(pgo)

spa_ncomp(pgo)

spa_area(pgo)

spa_perimeter(pgo)

spa_length(pgo)
}
\arguments{
\item{pgo}{A \code{pgeometry} object of the type \code{PLATEAULINE}, \code{PLATEAUCOMPOSITION}, or \code{PLATEAUCOLLECTION}. It throws a warning if a different type is given.}
}
\value{
A numerical value.
}
\description{
Fuzzy numerical operations are implemented by spatial plateau numerical operations, which
extract geometric measurements from spatial plateau objects,
such as the area of a plateau region object and the length of a plateau line object.
}
\details{
These functions calculate numerical properties from spatial plateau objects (i.e., \code{pgeometry} objects).
Some of them are \emph{type-independent}. This means that the parameter can be a \code{pgeometry} object of any type.
The type-independent functions are:
\itemize{
\item \code{spa_avg_degree()} calculates the average membership degree of a spatial plateau object.
\item \code{spa_ncomp()} returns the number of components of a spatial plateau object.
}

The remaining functions are \emph{type-dependent}. This means that the parameter have to be of a specific type.
The type-dependent functions are:
\itemize{
\item \code{spa_area()} computes the area of a plateau region, plateau composition, or plateau collection object.
\item \code{spa_perimeter()} computes the perimeter of a plateau region, plateau composition, or plateau collection.
\item \code{spa_length()} computes the length of a plateau line, plateau composition, or plateau collection object.
}

For the aforementioned functions, if the input has the incorrect data type, it throws a warning message and returns 0.
}
\examples{
# Point components
pcp1 <- create_component("POINT(0 0)", 0.3)
pcp2 <- create_component("MULTIPOINT((2 2), (2 4), (2 0))", 0.5)
pcp3 <- create_component("MULTIPOINT((1 1), (3 1), (1 3), (3 3))", 0.9)
pcp4 <- create_component("MULTIPOINT((1 2), (2 1), (3 2))", 1)
pcp5 <- create_component("MULTIPOINT((0 0.5), (2 3))", 0.7)
pcp6 <- create_component("MULTIPOINT((0 1), (3 3.5))", 0.85)
pcp7 <- create_component("MULTIPOINT((1 0), (4 2))", 0.4)
# Line components
lcp1 <- create_component("LINESTRING(0 0, 1 1.5)", 0.2)
lcp2 <- create_component("LINESTRING(1 3, 1 2, 2 0.5)", 0.5)
lcp3 <- create_component("LINESTRING(2 1.2, 3 1.6, 4 4)", 0.7)
lcp4 <- create_component("LINESTRING(1 1.5, 2 1.2)", 1.0)
lcp5 <- create_component("LINESTRING(-1 1, 2 2)", 0.9)
# Polygon components
rcp1 <- create_component("POLYGON((0 0, 1 4, 2 2, 0 0))", 0.4)
rcp2 <- create_component("POLYGON((2 0.5, 4 1, 4 0, 2 0.5))", 0.8)

# Creating spatial plateau objects
ppoint <- create_pgeometry(list(pcp1, pcp2, pcp3, pcp4, pcp5), "PLATEAUPOINT")
pline <- create_pgeometry(list(lcp1, lcp2, lcp3), "PLATEAULINE")
pregion <- create_pgeometry(list(rcp1, rcp2), "PLATEAUREGION")
pcomp <- create_pgeometry(list(pcp6, pcp7, lcp4, lcp5), "PLATEAUCOMPOSITION")
pcol <- create_pgeometry(list(ppoint, pline, pregion, pcomp), "PLATEAUCOLLECTION")

spa_avg_degree(ppoint)
spa_avg_degree(pline)
spa_avg_degree(pregion)
spa_avg_degree(pcomp)
spa_avg_degree(pcol)

spa_ncomp(ppoint)
spa_ncomp(pline)
spa_ncomp(pregion)
spa_ncomp(pcomp)
spa_ncomp(pcol)

spa_area(pregion)
spa_area(pcomp)
spa_area(pcol)

spa_perimeter(pregion)
spa_perimeter(pcomp)
spa_perimeter(pcol)

spa_length(pline)
spa_length(pcomp)
spa_length(pcol)
}
\references{
\href{https://onlinelibrary.wiley.com/doi/10.1111/tgis.13044}{Carniel, A. C.; Venâncio, P. V. A. B; Schneider, M. fsr: An R package for fuzzy spatial data handling. Transactions in GIS, vol. 27, no. 3, pp. 900-927, 2023.}

Underlying concepts and formal definitions of some spatial plateau numerical operations are introduced in:
\itemize{
\item \href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
}
