% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MPPsims.R
\name{MPPsims}
\alias{MPPsims}
\title{Missing person power simulations}
\usage{
MPPsims(
  reference,
  missing = "MP",
  selections,
  ep = TRUE,
  ip = TRUE,
  addBaseline = TRUE,
  nProfiles = 1,
  lrSims = 1,
  thresholdIP = NULL,
  disableMutations = NA,
  numCores = 1,
  seed = NULL,
  verbose = TRUE
)
}
\arguments{
\item{reference}{A connected \code{ped} object, or a list of pedigrees. In the
latter case, the list must have the same length as \code{selections}.}

\item{missing}{The ID label of the missing pedigree member.}

\item{selections}{A list of pedigree member subsets. In the special case that
all subsets consist of a single individual, \code{selections} can be given as a
simple vector.}

\item{ep}{A logical: Estimate the exclusion power? (Default: TRUE)}

\item{ip}{A logical: Estimate the inclusion power? (Default: TRUE)}

\item{addBaseline}{A logical. If TRUE (default) an \emph{empty} selection, named
"Baseline", is added as the first element of \code{selection}.}

\item{nProfiles}{The number of profile simulations for each selection.}

\item{lrSims, thresholdIP}{Parameters passed onto \code{\link[=missingPersonIP]{missingPersonIP()}}.}

\item{disableMutations}{This parameter determines how mutation models are
treated. Possible values are as follows:
\itemize{
\item \code{NA} (the default): Mutations are disabled only for those markers whose
known genotypes are consistent with \code{reference}. This is determined by
temporarily removing all mutation models and checking which markers have
nonzero likelihood.
\item \code{TRUE}: Mutations are disabled for all markers. This will result in an
error if any markers are inconsistent with \code{reference}.
\item \code{FALSE}: No action is done to disable mutations.
\item A vector containing the names or indices of those markers for which
mutations should be disabled.
}}

\item{numCores}{The number of cores used for parallelisation, by default 1.}

\item{seed}{A numeric seed for the random number generator (optional)}

\item{verbose}{A logical.}
}
\value{
An object of class "MPPsim", which is basically a list with one entry
for each element of \code{selections}. Each entry has elements \code{ep} and \code{ip},
each of which is a list of length \code{nProfiles}.

The output object has various attributes reflecting the input. Note that
\code{reference} and \code{selection} may differ slightly from the original input,
since they may be modified during the function run. (For instance, a
"Baseline" entry is added to \code{selection} if \code{addBaseline} is TRUE.) The
crucial point is that the output attributes correspond exactly to the
output data.
\itemize{
\item \code{reference} (always a list, of the same length as the \code{selections}
attribute
\item \code{selections}
\item \code{nProfiles},\code{lrSims},\code{thresholdIP},\code{seed} (as in the input)
\item \code{totalTime} (the total time used)
}
}
\description{
Estimate the exclusion/inclusion power for various selections of available
individuals.
}
\examples{
x = nuclearPed(fa = "Gf", mo = "Gm", children = c("Uncle", "Mother"), sex = 1:2)
x = addChildren(x, fa = "Father", mo = "Mother", nch = 3, sex = c(1,2,1),
                id = c("S1", "S2", "MP"))
x = addSon(x, "Father", id = "HS")

# Brother S1 is already genotyped with a marker with 4 alleles
m = marker(x, S1 = 1:2, alleles = 1:4)
x = setMarkers(x, m)

# Alternatives for additional genotyping
sel = list("Father", "S2", "HS", c("Gm", "Uncle"))

plot(x, marker = 1, hatched = sel)

# Simulate
simData = MPPsims(x, selections = sel, nProfiles = 2, lrSims = 2)

# Power plot
powerPlot(simData, type = 3)

\donttest{
### With  mutations
# Create inconsistent marker
m2 = m
genotype(m2, "Father") = 3
x = setMarkers(x, list(m, m2))

# Set mutation models for both
mutmod(x, 1:2) = list("equal", rate = 0.1)

# By default mutations are disabled for consistent markers
# MPPsims(x, selections = "Father", addBaseline = FALSE)

# Don't disable anything
# MPPsims(x, selections = "Father", addBaseline = FALSE,
#       disableMutations = FALSE)


# Disable all mutation models. SHOULD GIVE ERROR FOR SECOND MARKER
# MPPsims(x, selections = "Father", addBaseline = FALSE,
#         disableMutations = TRUE)

}

}
