% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eigenvalue.R
\name{eigenvalue}
\alias{eigenvalue}
\alias{fviz_eig}
\alias{fviz_screeplot}
\alias{get_eig}
\alias{get_eigenvalue}
\title{Extract and visualize the eigenvalues/variances of dimensions}
\usage{
get_eig(X)

get_eigenvalue(X)

fviz_eig(X, choice = c("variance", "eigenvalue"), geom = c("bar", "line"),
  barfill = "steelblue", barcolor = "steelblue", linecolor = "black",
  ncp = 10, addlabels = FALSE, hjust = 0, ...)

fviz_screeplot(...)
}
\arguments{
\item{X}{an object of class PCA, CA, MCA, MFA and HMFA [FactoMineR]; prcomp 
and princomp [stats]; dudi, pca, coa and acm [ade4]; ca and mjca [ca 
package].}

\item{choice}{a text specifying the data to be plotted. Allowed values are 
"variance" or "eigenvalue".}

\item{geom}{a text specifying the geometry to be used for the graph. Allowed 
values are "bar" for barplot, "line" for lineplot or c("bar", "line") to use
both types.}

\item{barfill}{fill color for bar plot.}

\item{barcolor}{outline color for bar plot.}

\item{linecolor}{color for line plot (when geom contains "line").}

\item{ncp}{a numeric value specifying the number of dimensions to be shown.}

\item{addlabels}{logical value. If TRUE, labels are added at the top of bars 
or points showing the information retained by each dimension.}

\item{hjust}{horizontal adjustment of the labels.}

\item{...}{optional arguments to be passed to the functions geom_bar(), 
geom_line(), geom_text() or fviz_eig().}
}
\value{
\itemize{ \item{get_eig() (or get_eigenvalue()): returns a data.frame 
 containing 3 columns: the eigenvalues, the percentage of variance and  the 
 cumulative percentage of variance retained by each dimension.} 
 \item{fviz_eig() (or fviz_screeplot()): returns a ggplot2} }
}
\description{
Eigenvalues correspond to the amount of the variation explained 
 by each principal component (PC). Read more: 
 \href{http://www.sthda.com/english/wiki/factominer-and-factoextra-principal-component-analysis-visualization-r-software-and-data-mining}{Principal
  Component Analysis}
 
 \itemize{ \item{get_eig(): Extract the eigenvalues/variances of the 
 principal dimensions} \item{fviz_eig(): Plot the eigenvalues/variances 
 against the number of dimensions} \item{get_eigenvalue(): an alias of 
 get_eig()} \item{fviz_screeplot(): an alias of fviz_eig()} }
 
 These functions support the results of Principal Component Analysis (PCA), 
 Correspondence Analysis (CA), Multiple Correspondence Analysis (MCA), 
 Multiple Factor Analysis (MFA) and Hierarchical Multiple Factor Analysis 
 (HMFA) functions.
}
\examples{
# Principal Component Analysis
# ++++++++++++++++++++++++++
data(iris)
res.pca <- prcomp(iris[, -5],  scale = TRUE)

# Extract eigenvalues/variances
get_eig(res.pca)

# Default plot
fviz_eig(res.pca)

# Customize the plot
   # - Add labels
   # - Change line color, bar fill and color. 
   # - Change axis limits and themes
   
p <- fviz_eig(res.pca, addlabels = TRUE, hjust = -0.3,
           linecolor = "#FC4E07", 
           barfill="white", barcolor ="darkblue")+ 
     ylim(0, 85)+ # y axis limits
     theme_minimal() # themes: http://www.sthda.com/english/wiki/ggplot2-themes
print(p)

# Change plot title and axis labels
p + labs(title = "Variances - PCA",
        x = "Principal Components", y = "\% of variances")

  
# Scree plot - Eigenvalues
fviz_eig(res.pca, choice = "eigenvalue", addlabels=TRUE)

# Use only bar  or line plot: geom = "bar" or geom = "line"
fviz_eig(res.pca, geom="line")
 
        
# Correspondence Analysis
# +++++++++++++++++++++++++++++++++
library(FactoMineR)
data(housetasks)
res.ca <- CA(housetasks, graph = FALSE)
get_eig(res.ca)
fviz_eig(res.ca, linecolor = "#FC4E07",
   barcolor = "#00AFBB", barfill = "#00AFBB")+
   theme_minimal()

# Multiple Correspondence Analysis
# +++++++++++++++++++++++++++++++++
library(FactoMineR)
data(poison)
res.mca <- MCA(poison, quanti.sup = 1:2, 
              quali.sup = 3:4, graph=FALSE)
get_eig(res.mca)
fviz_eig(res.mca, linecolor = "#FC4E07",
   barcolor = "#2E9FDF", barfill = "#2E9FDF")+
   theme_minimal()
 
 
# Multiple Factor Analysis
# +++++++++++++++++++++++++++++++++
library(FactoMineR)
data(wine)
res.mfa <- MFA(wine, group=c(2,5,3,10,9,2), type=c("n",rep("s",5)),
               ncp=5, name.group=c("orig","olf","vis","olfag","gust","ens"),
               num.group.sup=c(1,6), graph=FALSE)
get_eig(res.mfa)
fviz_eig(res.mfa, linecolor = "#FC4E07",
   barcolor = "#E7B800", barfill = "#E7B800")+
   theme_minimal()


}
\author{
Alboukadel Kassambara \email{alboukadel.kassambara@gmail.com}
}
\references{
http://www.sthda.com
}
\seealso{
\code{\link{fviz_pca}}, \code{\link{fviz_ca}},
 \code{\link{fviz_mca}}, \code{\link{fviz_mfa}}, \code{\link{fviz_hmfa}}
}

