% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esreg.R
\name{esreg}
\alias{esreg}
\alias{esreg.formula}
\alias{esreg.default}
\title{Joint Quantile and Expected Shortfall Regression}
\usage{
esreg(...)

\method{esreg}{formula}(formula, data = parent.frame(), alpha, g1 = 2L,
  g2 = 1L, early_stopping = 10, ...)

\method{esreg}{default}(xq, xe, y, alpha, g1 = 2L, g2 = 1L,
  early_stopping = 10, ...)
}
\arguments{
\item{...}{Further arguments (does not apply here)}

\item{formula}{Formula: y ~ x1 + x2 ... | x1 + x2 ...
where the first part after the response variable specifies the quantile equation
and the second the expected shortfall part. If only one set of regressors is
provided it is used for both model specifications.}

\item{data}{data.frame that holds the variables}

\item{alpha}{Probability level}

\item{g1}{1, 2 (see \link{G1_fun}, \link{G1_prime_fun}), defaults to 1}

\item{g2}{1, 2, 3, 4, 5 (see \link{G2_curly_fun}, \link{G2_fun}, \link{G2_prime_fun}). defaults to 2}

\item{early_stopping}{Stop the iterated local search if there is no improvement in early_stopping steps.}

\item{xq}{Explanatory variables for the quantile regression equation}

\item{xe}{Explanatory variables for the expected shortfall regression equation}

\item{y}{Response vector}
}
\value{
An esreg object
}
\description{
Estimates a joint linear regression model for the pair (VaR, ES):
\deqn{Q_\alpha(Y | Xq) = Xq'\beta_q}
\deqn{ES_\alpha(Y | Xe) = Xe'\beta_e}
}
\examples{
# Simulate data (DGP-(2) in the linked paper)
set.seed(0)
x <- rchisq(1000, df=1)
y <- -x + (1 + 0.5 * x) * rnorm(1000)

# True quantile and expected shortfall regression parameters (for alpha=0.025)
alpha=0.025
true_pars <- c(-1.959964, -1.979982, -2.337803, -2.168901)

# Estimate the model using the standard settings
fit <- esreg(y ~ x, alpha=alpha)

# Compare the different variance-covariance estimators
cov1 <- vcov(object=fit, sparsity="iid", cond_var="ind")
cov2 <- vcov(object=fit, sparsity="nid", cond_var="scl_N")
cov3 <- vcov(object=fit, sparsity="nid", cond_var="scl_sp")

print("Comparison of the variance-covariance estimators")
print(rbind(Truth=true_pars,
            Estimate=coef(fit),
            SE_iid_ind=sqrt(diag(cov1)),
            SE_nid_N=sqrt(diag(cov2)),
            SE_nid_sp=sqrt(diag(cov3))))

# Compares estimates using different G2 functions
fit1 <- esreg(y ~ x, alpha=alpha, g2=1)
fit2 <- esreg(y ~ x, alpha=alpha, g2=2)
fit3 <- esreg(y ~ x, alpha=alpha, g2=3)
fit4 <- esreg(y ~ x, alpha=alpha, g2=4)
fit5 <- esreg(y ~ x, alpha=alpha, g2=5)
fits <- sapply(list(fit1, fit2, fit3, fit4, fit5), coef)
colnames(fits) <- sapply(1:5, function(i) esreg:::.G_function_names(1, i)[2])
print("Comparison of the five G2 functions")
print(rbind(Truth=true_pars, t(fits)))

# Usage of different covariates
x <- rchisq(1000, df=1)
noise <- rnorm(1000)
y <- -x + (1 + 0.5 * x) * rnorm(1000)
fit <- esreg(y ~ x | x + noise, alpha=0.025)
print("Using different covariates for VaR and ES")
print(summary(fit))

}
\references{
\href{https://arxiv.org/abs/1704.02213}{A Joint Quantile and Expected Shortfall Regression Framework}
}
\seealso{
\code{\link{vcov.esreg}} for covariance estimation
}
