% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/signal_motion.R
\name{signal_motion}
\alias{signal_motion}
\title{Calculate particle motion parameters}
\usage{
signal_motion(data, time, dt, window, step, order = "xyz")
}
\arguments{
\item{data}{\code{List} of eseis objects or \code{matrix}, seismic
components to be processed. If \code{data} is a matrix, the components 
must be organised as columns.}

\item{time}{\code{POSIXct} vector, time vector corresponding to the 
seismic signal components. If omitted, a synthetic time vector will 
be generated. If omitted, the sampling period (\code{dt}) must be 
provided or is taken from the first eseis object in the \code{data} list.}

\item{dt}{\code{Numeric} value, sampling period. Only needed if 
\code{time} is omitted or if \code{data} is no \code{eseis} object.}

\item{window}{\code{Numeric} value, time window length (given as 
number of samples) used to calculate the particle motion parameters. 
If value is even, it will be set to the next smaller odd value. If 
omitted, the window size is set to 1 percent of the time series length by
default.}

\item{step}{\code{Numeric} value, step size (given as number of samples), 
by which the window is shifted over the data set. If omitted, the step 
size is set to 50 percent of the window size by default.}

\item{order}{\code{Character} value, order of the seismic components. 
Description must contain the letters \code{"x"},\code{"y"} and
\code{"z"} in the order according to the input data set. Default is 
\code{"xyz"} (EW-NS-vertical).}
}
\value{
A List object with rectilinearity (\code{rectilinearity}),  
angularity (\code{polarity}), azimuth (\code{azimuth}) and incidence 
(\code{incidence}), as well as the corresponding time vector for 
these values.
}
\description{
The function calculates from a data set of three seismic 
components of the same signal the following particle motion 
paramters using a moving window approach: horizontal-vertical 
eigenvalue ratio, azimuth and inclination.
}
\details{
The function code is loosely based on the function GAZI() from 
the package RSEIS with removal of unnecessary content and updated 
or rewritten loop functionality. In its current form, it also uses 
additional workflows from obspy.signal.polarisation, specifically 
following the Flinn (1965) approach. It windows the input signals,
calculates the covariance matrix and performs a singular values 
decomposition to use the eigen values and vectors to determine the 
ratios corresponding to the output values rectilinearity, angularity,
azimuth and incidence. 

Note that the names of the output objects as well as the calculation 
routine have changed from the earlier version (V. 0.6.0), to increase 
computational efficiency and fix a bug in the windowing implementation.
}
\examples{
## load example data set
data(earthquake)

## filter seismic signals
s <- eseis::signal_filter(data = s, 
                          dt = 1/200, 
                          f = c(1, 3))

## convert list of signal vectors to column-wise matrix
s <- do.call(cbind, s)

## calculate particle motion parameters
pm <- signal_motion(data = s, 
                    dt = 1 / 200,
                    window = 500, 
                    step = 250)
                    
## plot function output
par_original <- par(no.readonly = TRUE)
par(mfcol = c(2, 2))

plot(pm$time, pm$rect, type = "b")
plot(pm$time, pm$plan, type = "b")
plot(pm$time, pm$azimuth, type = "b")
plot(pm$time, pm$incidence, type = "b")

par(par_original)

}
\author{
Michael Dietze
}
\keyword{eseis}
