% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/job.R
\docType{class}
\name{EplusJob}
\alias{EplusJob}
\alias{eplus_job}
\title{Run EnergyPlus Simulation and Collect Outputs}
\usage{
eplus_job(idf, epw)
}
\arguments{
\item{idf}{A path to an local EnergyPlus IDF file or an \code{Idf} object.}

\item{epw}{A path to an local EnergyPlus EPW file or an \code{Epw} object. \code{epw}
can also be \code{NULL} which will force design-day-only simulation when
\code{\link[=EplusJob]{$run()}} method is called. Note this needs at least one
\code{Sizing:DesignDay} object exists in the \link{Idf}.}
}
\value{
An \code{EplusJob} object.
}
\description{
\code{EplusJob} class wraps the EnergyPlus command line interface and provides
methods to extract simulation outputs.

\code{eplus_job()} takes an IDF and EPW as input, and returns an \code{EplusJob} object
for running EnergyPlus simulation and collecting outputs.
}
\details{
eplusr uses the EnergyPlus SQL output for extracting simulation outputs.

\code{EplusJob} has provide some wrappers that do SQL query to get report data
results, i.e. results from \code{Output:Variable} and \verb{Output:Meter*}. But for
\code{Output:Table} results, you have to be familiar with the structure of the
EnergyPlus SQL results, especially for table \emph{"TabularDataWithStrings"}. For
details, please see \emph{"2.20 eplusout.sql"}, especially \emph{"2.20.4.4 TabularData
Table"} in EnergyPlus \emph{"Output Details and Examples"} documentation. An
object in \code{Output:SQLite} with \verb{Option Type} value of \code{SimpleAndTabular} will
be automatically created if it does not exists, to ensure that the output
collection functionality works successfully.

In order to make sure \code{.rdd} (Report Data Dictionary) and \code{.mdd} (Meter Data
Dictionary) files are created during simulation, an object in
\code{Output:VariableDictionary} class with \verb{Key Field} value being \code{IDF} will be
automatically created if it does not exists.
}
\examples{

## ------------------------------------------------
## Method `EplusJob$new`
## ------------------------------------------------

\dontrun{
if (is_avail_eplus("8.8")) {
    name_idf <- "1ZoneUncontrolled.idf"
    name_epw <-  "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    path_idf <- path_eplus_example("8.8", name_idf)
    path_epw <- path_eplus_weather("8.8", name_epw)

    # create from local files
    job <- eplus_job(path_idf, path_epw)

    # create from an Idf and an Epw object
    job <- eplus_job(read_idf(path_idf), read_epw(path_epw))
}
}

## ------------------------------------------------
## Method `EplusJob$version`
## ------------------------------------------------

\dontrun{
job$version()
}


## ------------------------------------------------
## Method `EplusJob$path`
## ------------------------------------------------

\dontrun{
job$path()
job$path("idf")
job$path("epw")
}


## ------------------------------------------------
## Method `EplusJob$run`
## ------------------------------------------------

\dontrun{
# only run design day
job$run(NULL)

# specify output directory
job$run(dir = tempdir())

# run in the background
job$run(wait = TRUE)
# see job status
job$status()

# force to kill background job before running the new one
job$run(force = TRUE)

# do not show anything in the console
job$run(echo = FALSE)

# copy external files used in the model to simulation output directory
job$run(copy_external = TRUE)
}


## ------------------------------------------------
## Method `EplusJob$kill`
## ------------------------------------------------

\dontrun{
job$kill()
}


## ------------------------------------------------
## Method `EplusJob$status`
## ------------------------------------------------

\dontrun{
job$status()
}


## ------------------------------------------------
## Method `EplusJob$errors`
## ------------------------------------------------

\dontrun{
job$errors()

# show all information
job$errors(info = TRUE)
}


## ------------------------------------------------
## Method `EplusJob$output_dir`
## ------------------------------------------------

\dontrun{
job$output_dir()

# Below will open output directory
# job$output_dir(open = TRUE)
}


## ------------------------------------------------
## Method `EplusJob$list_files`
## ------------------------------------------------

\dontrun{
# list all files in the output directory
job$list_files(simplify = TRUE)

# get a full list of all possible inputs and outputs even though they
# may not exist for current simulation
job$list_files()

# return the full paths instead of just file names
job$locate_output(full = TRUE)
}


## ------------------------------------------------
## Method `EplusJob$locate_output`
## ------------------------------------------------

\dontrun{
# get the file path of the error file
job$locate_output(".err", strict = FALSE)

# can use to detect if certain output file exists
job$locate_output(".expidf", strict = TRUE)
}


## ------------------------------------------------
## Method `EplusJob$read_rdd`
## ------------------------------------------------

\dontrun{
job$read_rdd()
}


## ------------------------------------------------
## Method `EplusJob$read_mdd`
## ------------------------------------------------

\dontrun{
job$read_mdd()
}


## ------------------------------------------------
## Method `EplusJob$list_table`
## ------------------------------------------------

\dontrun{
job$list_table()
}


## ------------------------------------------------
## Method `EplusJob$read_table`
## ------------------------------------------------

\dontrun{
# read a specific table
job$read_table("Zones")
}


## ------------------------------------------------
## Method `EplusJob$report_data_dict`
## ------------------------------------------------

\dontrun{
job$report_data_dict()
}


## ------------------------------------------------
## Method `EplusJob$report_data`
## ------------------------------------------------

\dontrun{
# read all report data
job$report_data()

# specify output variables using report data dictionary
dict <- job$report_data_dict()
job$report_data(dict[units == "C"])

# specify output variables using 'key_value' and 'name'
job$report_data("environment", "site outdoor air drybulb temperature")

# explicitly specify year value and time zone
job$report_data(dict[1], year = 2020, tz = "Etc/GMT+8")

# explicitly specify case name
job$report_data(dict[1], case = "example")

# get all possible columns
job$report_data(dict[1], all = TRUE)

# return in a format that is similar as EnergyPlus CSV output
job$report_data(dict[1], wide = TRUE)

# return in a format that is similar as EnergyPlus CSV output with
# extra columns
job$report_data(dict[1], wide = TRUE, all = TRUE)

# only get data at the working hour on the first Monday
job$report_data(dict[1], hour = 8:18, day_type = "monday", simulation_days = 1:7)

# only get specified run period data
job$read_table("EnvironmentPeriods") # possible environment name
job$report_data(dict[1], environment_name = "San Francisco Intl Ap CA USA TMY3 WMO#=724940")
# can also be done using 'environment_period_index' column
job$report_data(dict[1], all = TRUE)[environment_period_index == 3L]
}


## ------------------------------------------------
## Method `EplusJob$tabular_data`
## ------------------------------------------------

\dontrun{
# read all tabular data
job$tabular_data()

# explicitly specify data you want
str(job$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy"
))

# get tabular data in wide format and coerce numeric values
str(job$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy",
    wide = TRUE, string_value = FALSE
))
}


## ------------------------------------------------
## Method `EplusJob$print`
## ------------------------------------------------

\dontrun{
job$print()
}

}
\seealso{
\link{ParametricJob} class for EnergyPlus parametric simulations.

\code{\link[=param_job]{param_job()}} for creating an EnergyPlus parametric job.
}
\author{
Hongyuan Jia
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-EplusJob-new}{\code{EplusJob$new()}}
\item \href{#method-EplusJob-version}{\code{EplusJob$version()}}
\item \href{#method-EplusJob-path}{\code{EplusJob$path()}}
\item \href{#method-EplusJob-run}{\code{EplusJob$run()}}
\item \href{#method-EplusJob-kill}{\code{EplusJob$kill()}}
\item \href{#method-EplusJob-status}{\code{EplusJob$status()}}
\item \href{#method-EplusJob-errors}{\code{EplusJob$errors()}}
\item \href{#method-EplusJob-output_dir}{\code{EplusJob$output_dir()}}
\item \href{#method-EplusJob-list_files}{\code{EplusJob$list_files()}}
\item \href{#method-EplusJob-locate_output}{\code{EplusJob$locate_output()}}
\item \href{#method-EplusJob-read_rdd}{\code{EplusJob$read_rdd()}}
\item \href{#method-EplusJob-read_mdd}{\code{EplusJob$read_mdd()}}
\item \href{#method-EplusJob-list_table}{\code{EplusJob$list_table()}}
\item \href{#method-EplusJob-read_table}{\code{EplusJob$read_table()}}
\item \href{#method-EplusJob-report_data_dict}{\code{EplusJob$report_data_dict()}}
\item \href{#method-EplusJob-report_data}{\code{EplusJob$report_data()}}
\item \href{#method-EplusJob-tabular_data}{\code{EplusJob$tabular_data()}}
\item \href{#method-EplusJob-print}{\code{EplusJob$print()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-new"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-new}{}}}
\subsection{Method \code{new()}}{
Create an \code{EplusJob} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$new(idf, epw)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{idf}}{Path to an local EnergyPlus IDF file or an \link{Idf} object.}

\item{\code{epw}}{Path to an local EnergyPlus EPW file or an \link{Epw} object.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An \code{EplusJob} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
if (is_avail_eplus("8.8")) {
    name_idf <- "1ZoneUncontrolled.idf"
    name_epw <-  "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    path_idf <- path_eplus_example("8.8", name_idf)
    path_epw <- path_eplus_weather("8.8", name_epw)

    # create from local files
    job <- eplus_job(path_idf, path_epw)

    # create from an Idf and an Epw object
    job <- eplus_job(read_idf(path_idf), read_epw(path_epw))
}
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-version"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-version}{}}}
\subsection{Method \code{version()}}{
Get the version of IDF in current job
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$version()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$version()} returns the version of IDF that current \code{EplusJob} uses.
}

\subsection{Returns}{
A \code{\link[base:numeric_version]{base::numeric_version()}} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$version()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-path"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-path}{}}}
\subsection{Method \code{path()}}{
Get the paths of file that current \code{EpwSql} uses
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$path(type = c("all", "idf", "epw"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{If \code{"all"}, both the \link{Idf} path and \link{Epw} path are
returned. If \code{"idf"}, only IDF path is returned. If \code{"epw"},
only EPW path is returned. If \code{epw} is \code{NULL}, \code{NA} is
returned for EPW path. Default: \code{"all"}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$path()} returns the path of IDF or EPW of current job.
}

\subsection{Returns}{
A character vector.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$path()
job$path("idf")
job$path("epw")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-run"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-run}{}}}
\subsection{Method \code{run()}}{
Run simulationA
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$run(
  epw,
  dir = NULL,
  wait = TRUE,
  force = FALSE,
  echo = wait,
  copy_external = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{epw}}{A path to an \code{.epw} file or an \link{Epw} object. \code{epw} can
also be \code{NULL} which will force design-day-only simulation.
Note this needs EnergyPlus v8.3 and later, and at least one
\code{Sizing:DesignDay} object exists in the \code{Idf}. If not given,
the \code{epw} input used when creating this \code{EplusJob} object will
be used.}

\item{\code{dir}}{The directory to save the simulation results. If \code{NULL},
the input \code{idf} folder will be used. Default: \code{NULL}.}

\item{\code{wait}}{If \code{TRUE}, R will hang on and wait for the simulation to
complete. EnergyPlus standard output (stdout) and error
(stderr) is printed to R console. If \code{FALSE}, simulation will
be run in a background process.  Default: \code{TRUE}.}

\item{\code{force}}{Only applicable when the last job runs with \code{wait}
equals to \code{FALSE} and is still running. If \code{TRUE}, current
running job is forced to stop and a new one will start.
Default: \code{FALSE}.}

\item{\code{echo}}{Only applicable when \code{wait} is \code{TRUE}. Whether to show
standard output and error from EnergyPlus. Default: same as
\code{wait}.}

\item{\code{copy_external}}{If \code{TRUE}, the external files that current \code{Idf}
object depends on will also be copied into the simulation
output directory. The values of file paths in the Idf will be
changed automatically. This ensures that the output directory
will have all files needed for the model to run. Default is
\code{FALSE}.}

\item{\code{readvars}}{If \code{TRUE}, the \code{ReadVarESO} post-processor will run
to generate CSV files from the ESO output. Since those CSV
files are never used when extracting simulation data in eplusr,
setting it to \code{FALSE} can speed up the simulation if there are
hundreds of output variables or meters. Default: \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$run()} runs the simulation using input IDF and EPW file. If \code{wait}
is \code{FALSE}, the job is run in the background. You can get updated job
status by just
\href{../../eplusr/html/EplusJob.html#method-print}{printing}
the \code{EplusJob} object.

Parameter \code{epw} can be used to reset the EPW file to use for
simulation. If not given, the \code{epw} input used when creating
this \code{EplusJob} object will be used.
}

\subsection{Returns}{
The \code{EplusJob} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# only run design day
job$run(NULL)

# specify output directory
job$run(dir = tempdir())

# run in the background
job$run(wait = TRUE)
# see job status
job$status()

# force to kill background job before running the new one
job$run(force = TRUE)

# do not show anything in the console
job$run(echo = FALSE)

# copy external files used in the model to simulation output directory
job$run(copy_external = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-kill"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-kill}{}}}
\subsection{Method \code{kill()}}{
Kill current running job
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$kill()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$kill()} kills the background EnergyPlus process if possible. It
only works when simulation runs in non-waiting mode.
}

\subsection{Returns}{
A single logical value of \code{TRUE} or \code{FALSE}, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$kill()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-status"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-status}{}}}
\subsection{Method \code{status()}}{
Get the job status
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$status()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$status()} returns a named list of values that indicates the status of the
job:
\itemize{
\item \code{run_before}: \code{TRUE} if the job has been run before. \code{FALSE} otherwise.
\item \code{alive}: \code{TRUE} if the simulation is still running in the background.
\code{FALSE} otherwise.
\item \code{terminated}: \code{TRUE} if the simulation was terminated during last
simulation. \code{FALSE} otherwise. \code{NA} if the job has not been run yet.
\item \code{successful}: \code{TRUE} if last simulation ended successfully. \code{FALSE}
otherwise. \code{NA} if the job has not been run yet.
\item \code{changed_after}: \code{TRUE} if the IDF file has been changed since last
simulation. \code{FALSE} otherwise. \code{NA} if the job has not been run yet.
}
}

\subsection{Returns}{
A named list of 5 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$status()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-errors"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-errors}{}}}
\subsection{Method \code{errors()}}{
Read simulation errors
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$errors(info = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{info}}{If \code{FALSE}, only warnings and errors are printed.
Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
$errors() returns an \link[=read_err]{ErrFile} object which contains all
contents of the simulation error file (\code{.err}). If \code{info} is \code{FALSE},
only warnings and errors are printed.
}

\subsection{Returns}{
An \link[=read_err]{ErrFile} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$errors()

# show all information
job$errors(info = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-output_dir"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-output_dir}{}}}
\subsection{Method \code{output_dir()}}{
Get simulation output directory
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$output_dir(open = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{open}}{If \code{TRUE}, the output directory will be opened.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$output_dir()} returns the output directory of simulation results.
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$output_dir()

# Below will open output directory
# job$output_dir(open = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-list_files"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-list_files}{}}}
\subsection{Method \code{list_files()}}{
List all output files in current simulation
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$list_files(simplify = FALSE, full = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{simplify}}{If \code{TRUE}, a character vector of EnergyPlus input
and output file names in the output directory is given. If \code{FALSE}, a
full named list of all possible input and output types is given. \code{NA}
is returned if no input or output files are found for that type.
Default: \code{FALSE}.}

\item{\code{full}}{If \code{TRUE}, the full file paths in the output directory
are returned. Otherwise, only the file names are returned. Default:
\code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$list_files()} returns all input and output files for current
EnergyPlus simulation.

Description of all possible outputs from EnergyPlus can be found in
EnergyPlus documentation "Output Details and Examples".

Below gives a brief summary on the meaning of elements in the
returned list.\tabular{lll}{
   # \tab Element \tab Description \cr
   1 \tab \code{ads} \tab EnergyPlus AirflowNetwork related output \cr
   2 \tab \code{audit} \tab EnergyPlus inputs echo \cr
   3 \tab \code{bnd} \tab EnergyPlus branch node details \cr
   4 \tab \code{bsmt_audit} \tab Basement input Echo \cr
   5 \tab \code{bsmt_csv} \tab Basement CSV output \cr
   6 \tab \code{bsmt_idf} \tab Basement IDF output \cr
   7 \tab \code{bsmt_out} \tab Basement Output \cr
   8 \tab \code{cbor} \tab Energyplus CBOR binary output introduced since v9.5 \cr
   9 \tab \code{dbg} \tab Energyplus debug output \cr
   10 \tab \code{delight} \tab EnergyPlus DElight simulation inputs and outputs \cr
   11 \tab \code{dfs} \tab EnergyPlus daylighting factor for exterior windows \cr
   12 \tab \code{dxf} \tab EnergyPlus surface drawing output \cr
   13 \tab \code{edd} \tab EnergyPlus EMS report \cr
   14 \tab \code{eio} \tab EnergyPlus standard and optional reports \cr
   15 \tab \code{end} \tab EnergyPlus simulation status in one line \cr
   16 \tab \code{epjson} \tab EnergyPlus epJSON input converted from IDF \cr
   17 \tab \code{epmdet} \tab EPMacro inputs echo \cr
   18 \tab \code{epmidf} \tab EPMacro IDF output \cr
   19 \tab \code{epw} \tab EnergyPlus Weather File input \cr
   20 \tab \code{err} \tab EnergyPlus error summarry \cr
   21 \tab \code{eso} \tab EnergyPlus standard output \cr
   22 \tab \code{experr} \tab ExpandObjects error summary \cr
   23 \tab \code{expidf} \tab ExpandObjects IDF output \cr
   24 \tab \code{glhe} \tab EnergyPlus ground heat exchange file \cr
   25 \tab \code{idf} \tab EnergyPlus IDF input \cr
   26 \tab \code{imf} \tab EPMacro IMF input \cr
   27 \tab \code{iperr} \tab convertESOMTR error summary \cr
   28 \tab \code{ipeso} \tab convertESOMTR standard output in IP units \cr
   29 \tab \code{ipmtr} \tab convertESOMTR meter output in IP units \cr
   30 \tab \code{json} \tab EnergyPlus JSON time series output introduced since v9.5 \cr
   31 \tab \code{log} \tab EnergyPlus log output \cr
   32 \tab \code{map} \tab EnergyPlus daylighting intensity map output \cr
   33 \tab \code{mdd} \tab EnergyPlus meter list \cr
   34 \tab \code{meter} \tab EnergyPlus meter CSV output \cr
   35 \tab \code{msgpack} \tab EnergyPlus MessagePack binary output introduced since v9.5 \cr
   36 \tab \code{mtd} \tab EnergyPlus meter details \cr
   37 \tab \code{mtr} \tab EnergyPlus meter output \cr
   38 \tab \code{perflog} \tab EnergyPlus log for `PerformancePrecisionTradeoffs \cr
   39 \tab \code{rdd} \tab EnergyPlus report variable names \cr
   40 \tab \code{rvaudit} \tab ReadVarsESO input echo \cr
   41 \tab \code{sci} \tab EnergyPlus cost benefit calculation information \cr
   42 \tab \code{screen} \tab EnergyPlus window scrren transmittance map output \cr
   43 \tab \code{shading} \tab EnergyPlus surface shading CSV output \cr
   44 \tab \code{shd} \tab EnergyPlus surface shading combination report \cr
   45 \tab \code{slab_ger} \tab Slab error summary \cr
   46 \tab \code{slab_gtp} \tab Slab ground temperature output \cr
   47 \tab \code{slab_out} \tab Slab IDF output \cr
   48 \tab \code{sln} \tab EnergyPlus \verb{Output:Surfaces:List, Lines} output \cr
   49 \tab \code{sqlite} \tab EnergyPlus SQLite output \cr
   50 \tab \code{sqlite_err} \tab EnergyPlus SQLite error summary \cr
   51 \tab \code{ssz} \tab EnergyPlus system sizing outputs in CSV, TAB or TXT format \cr
   52 \tab \code{svg} \tab HVAC-Diagram HVAC diagram output \cr
   53 \tab \code{table} \tab EnergyPlus tabular outputs in CSV, TAB, TXT, HTM, or XML format \cr
   54 \tab \code{variable} \tab EnergyPlus report variable CSV output \cr
   55 \tab \code{wrl} \tab EnergyPlus \verb{Output:Surfaces:List, VRML} output \cr
   56 \tab \code{zsz} \tab EnergyPlus system sizing outputs in CSV, TAB or TXT format \cr
   57 \tab \code{resource} \tab External file resources used for the simulation, e.g. \code{Schedule:File} \cr
}
}

\subsection{Returns}{
If \code{FALSE}, a character vector. Otherwise, a named list.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# list all files in the output directory
job$list_files(simplify = TRUE)

# get a full list of all possible inputs and outputs even though they
# may not exist for current simulation
job$list_files()

# return the full paths instead of just file names
job$locate_output(full = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-locate_output"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-locate_output}{}}}
\subsection{Method \code{locate_output()}}{
Get path of output file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$locate_output(suffix = ".err", strict = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{suffix}}{A string that indicates the file extension of
simulation output. Default: \code{".err"}.}

\item{\code{strict}}{If \code{TRUE}, it will check if the simulation was
terminated, is still running or the file exists or not.
Default: \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$locate_output()} returns the path of a single output file specified
by file suffix.
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get the file path of the error file
job$locate_output(".err", strict = FALSE)

# can use to detect if certain output file exists
job$locate_output(".expidf", strict = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-read_rdd"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-read_rdd}{}}}
\subsection{Method \code{read_rdd()}}{
Read Report Data Dictionary (RDD) file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$read_rdd()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$read_rdd()} return the core data of Report Data Dictionary (RDD)
file. For details, please see \code{\link[=read_rdd]{read_rdd()}}.
}

\subsection{Returns}{
An \link[=read_rdd]{RddFile} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$read_rdd()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-read_mdd"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-read_mdd}{}}}
\subsection{Method \code{read_mdd()}}{
Read Report Data Dictionary (RDD) file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$read_mdd()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$read_mdd()} return the core data of Meter Data Dictionary (MDD)
file. For details, please see \code{\link[=read_mdd]{read_mdd()}}.
}

\subsection{Returns}{
An \link[=read_mdd]{MddFile} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$read_mdd()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-list_table"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-list_table}{}}}
\subsection{Method \code{list_table()}}{
List all table names in EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$list_table()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$list_table()} returns all available table and view names in the
EnergyPlus SQLite file.
}

\subsection{Returns}{
A character vector
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$list_table()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-read_table"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-read_table}{}}}
\subsection{Method \code{read_table()}}{
Read a single table from EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$read_table(name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{A single string specifying the name of table to read.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$read_table()} takes a valid table \code{name} of those from
\href{../../eplusr/html/EplusJob.html#method-list_table}{\code{$list_table()}}
and returns that table data in a \code{\link[data.table:data.table]{data.table::data.table()}} format.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read a specific table
job$read_table("Zones")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-report_data_dict"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-report_data_dict}{}}}
\subsection{Method \code{report_data_dict()}}{
Read report data dictionary from EnergyPlus SQL output
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$report_data_dict()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$report_data_dict()} returns a \code{\link[data.table:data.table]{data.table::data.table()}} which
contains all information about report data.

For details on the meaning of each columns, please see "2.20.2.1
ReportDataDictionary Table" in EnergyPlus "Output Details and
Examples" documentation.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} of 10 columns:
\itemize{
\item \code{report_data_dictionary_index}: The integer used to link the
dictionary data to the variable data. Mainly useful when joining
different tables
\item \code{is_meter}: Whether report data is a meter data. Possible values:
\code{0} and \code{1}
\item \code{timestep_type}: Type of data timestep. Possible values: \code{Zone} and
\verb{HVAC System}
\item \code{key_value}: Key name of the data
\item \code{name}: Actual report data name
\item \code{reporting_frequency}: Data reporting frequency
\item \code{schedule_name}: Name the the schedule that controls reporting
frequency.
\item \code{units}: The data units
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$report_data_dict()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-report_data"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-report_data}{}}}
\subsection{Method \code{report_data()}}{
Read report data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$report_data(
  key_value = NULL,
  name = NULL,
  year = NULL,
  tz = "UTC",
  case = "auto",
  all = FALSE,
  wide = FALSE,
  period = NULL,
  month = NULL,
  day = NULL,
  hour = NULL,
  minute = NULL,
  interval = NULL,
  simulation_days = NULL,
  day_type = NULL,
  environment_name = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key_value}}{A character vector to identify key values of the
data. If \code{NULL}, all keys of that variable will be returned.
\code{key_value} can also be data.frame that contains \code{key_value}
and \code{name} columns. In this case, \code{name} argument in
\verb{$report_data()} is ignored. All available \code{key_value} for
current simulation output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-report_data_dict}{\code{$report_data_dict()}}.
Default: \code{NULL}.}

\item{\code{name}}{A character vector to identify names of the data. If
\code{NULL}, all names of that variable will be returned. If
\code{key_value} is a data.frame, \code{name} is ignored. All available
\code{name} for current simulation output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-report_data_dict}{\code{$report_data_dict()}}.
Default: \code{NULL}.}

\item{\code{year}}{Year of the date time in column \code{datetime}. If \code{NULL}, it
will calculate a year value that meets the start day of week
restriction for each environment. Default: \code{NULL}.}

\item{\code{tz}}{Time zone of date time in column \code{datetime}. Default:
\code{"UTC"}.}

\item{\code{case}}{If not \code{NULL}, a character column will be added indicates
the case of this simulation. If \code{"auto"}, the name of the IDF
file without extension is used.}

\item{\code{all}}{If \code{TRUE}, extra columns are also included in the returned
\code{\link[data.table:data.table]{data.table::data.table()}}.}

\item{\code{wide}}{If \code{TRUE}, the output is formatted in the same way as
standard EnergyPlus csv output file.}

\item{\code{period}}{A Date or POSIXt vector used to specify which time
period to return. The year value does not matter and only
month, day, hour and minute value will be used when
subsetting. If \code{NULL}, all time period of data is returned.
Default: \code{NULL}.}

\item{\code{month, day, hour, minute}}{Each is an integer vector for month,
day, hour, minute subsetting of \code{datetime} column when
querying on the SQL database. If \code{NULL}, no subsetting is
performed on those components. All possible \code{month}, \code{day},
\code{hour} and \code{minute} can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.
Default: \code{NULL}.}

\item{\code{interval}}{An integer vector used to specify which interval
length of report to extract. If \code{NULL}, all interval will be
used. Default: \code{NULL}.}

\item{\code{simulation_days}}{An integer vector to specify which simulation
day data to extract. Note that this number resets after warmup
and at the beginning of an environment period. All possible
\code{simulation_days} can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.
If \code{NULL}, all simulation days will be used. Default: \code{NULL}.}

\item{\code{day_type}}{A character vector to specify which day type of data
to extract. All possible day types are: \code{Sunday}, \code{Monday},
\code{Tuesday}, \code{Wednesday}, \code{Thursday}, \code{Friday}, \code{Saturday},
\code{Holiday}, \code{SummerDesignDay}, \code{WinterDesignDay}, \code{CustomDay1},
and \code{CustomDay2}. All possible values for current simulation
output can be obtained using
\href{../../eplusr/html/EplusSql.html#method-read_table}{\code{$read_table("Time")}}.}

\item{\code{environment_name}}{A character vector to specify which
environment data to extract. If \code{NULL}, all environment data
are returned. Default: \code{NULL}. All possible
\code{environment_name} for current simulation output can be
obtained using:

\if{html}{\out{<div class="sourceCode">}}\preformatted{$read_table("EnvironmentPeriods")
}\if{html}{\out{</div>}}}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$report_data()} extracts the report data in a
\code{\link[data.table:data.table]{data.table::data.table()}} using key values, variable names and other
specifications.

\verb{$report_data()} can also directly take all or subset output from
\verb{$report_data_dict()} as input, and extract all data specified.

The returned column numbers varies depending on \code{all} argument.
\itemize{
\item \code{all} is \code{FALSE}, the returned \code{\link[data.table:data.table]{data.table::data.table()}} has 6 columns:
\itemize{
\item \code{case}: Simulation case specified using \code{case} argument
\item \code{datetime}: The date time of simulation result
\item \code{key_value}: Key name of the data
\item \code{name}: Actual report data name
\item \code{units}: The data units
\item \code{value}: The data value
}
\item \code{all} is \code{TRUE}, besides columns described above, extra columns are also
included:
\itemize{
\item \code{month}: The month of reported date time
\item \code{day}: The day of month of reported date time
\item \code{hour}: The hour of reported date time
\item \code{minute}: The minute of reported date time
\item \code{dst}: Daylight saving time indicator. Possible values: \code{0} and \code{1}
\item \code{interval}: Length of reporting interval
\item \code{simulation_days}: Day of simulation
\item \code{day_type}: The type of day, e.g. \code{Monday}, \code{Tuesday} and etc.
\item \code{environment_period_index}: The indices of environment.
\item \code{environment_name}: A text string identifying the environment.
\item \code{is_meter}: Whether report data is a meter data. Possible values: \code{0} and
\code{1}
\item \code{type}: Nature of data type with respect to state. Possible values: \code{Sum}
and \code{Avg}
\item \code{index_group}: The report group, e.g. \code{Zone}, \code{System}
\item \code{timestep_type}: Type of data timestep. Possible values: \code{Zone} and \verb{HVAC System}
\item \code{reporting_frequency}: The reporting frequency of the variable, e.g.
\verb{HVAC System Timestep}, \verb{Zone Timestep}.
\item \code{schedule_name}: Name of the the schedule that controls reporting
frequency.
}
}

With the \code{datetime} column, it is quite straightforward to apply time-series
analysis on the simulation output. However, another painful thing is that
every simulation run period has its own \verb{Day of Week for Start Day}. Randomly
setting the \code{year} may result in a date time series that does not have
the same start day of week as specified in the RunPeriod objects.

eplusr provides a simple solution for this. By setting \code{year} to \code{NULL},
which is the default behavior, eplusr will calculate a year value (from
year 2017 backwards) for each run period that compliances with the start
day of week restriction.

It is worth noting that EnergyPlus uses 24-hour clock system where 24 is only
used to denote midnight at the end of a calendar day. In EnergyPlus output,
"00:24:00" with a time interval being 15 mins represents a time period from
"00:23:45" to "00:24:00", and similarly "00:15:00" represents a time period
from "00:24:00" to "00:15:00" of the next day. This means that if current day
is Friday, day of week rule applied in schedule time period "00:23:45" to
"00:24:00" (presented as "00:24:00" in the output) is also Friday, but not
Saturday. However, if you try to get the day of week of time "00:24:00" in R,
you will get Saturday, but not Friday. This introduces inconsistency and may
cause problems when doing data analysis considering day of week value.

With \code{wide} equals \code{TRUE}, \verb{$report_data()} will format the simulation output
in the same way as standard EnergyPlus csv output file. Sometimes this can be
useful as there may be existing tools/workflows that depend on this format.
When both \code{wide} and \code{all} are \code{TRUE}, columns of runperiod environment names
and date time components are also returned, including:
\verb{environment_period_index", "environment_name}, \code{simulation_days},
\code{datetime}, \code{month}, \code{day}, \code{hour}, \code{minute}, \code{day_type}.

For convenience, input character arguments matching in
\verb{$report_data()} are \strong{case-insensitive}.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read all report data
job$report_data()

# specify output variables using report data dictionary
dict <- job$report_data_dict()
job$report_data(dict[units == "C"])

# specify output variables using 'key_value' and 'name'
job$report_data("environment", "site outdoor air drybulb temperature")

# explicitly specify year value and time zone
job$report_data(dict[1], year = 2020, tz = "Etc/GMT+8")

# explicitly specify case name
job$report_data(dict[1], case = "example")

# get all possible columns
job$report_data(dict[1], all = TRUE)

# return in a format that is similar as EnergyPlus CSV output
job$report_data(dict[1], wide = TRUE)

# return in a format that is similar as EnergyPlus CSV output with
# extra columns
job$report_data(dict[1], wide = TRUE, all = TRUE)

# only get data at the working hour on the first Monday
job$report_data(dict[1], hour = 8:18, day_type = "monday", simulation_days = 1:7)

# only get specified run period data
job$read_table("EnvironmentPeriods") # possible environment name
job$report_data(dict[1], environment_name = "San Francisco Intl Ap CA USA TMY3 WMO#=724940")
# can also be done using 'environment_period_index' column
job$report_data(dict[1], all = TRUE)[environment_period_index == 3L]
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-tabular_data"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-tabular_data}{}}}
\subsection{Method \code{tabular_data()}}{
Read tabular data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$tabular_data(
  report_name = NULL,
  report_for = NULL,
  table_name = NULL,
  column_name = NULL,
  row_name = NULL,
  wide = FALSE,
  string_value = !wide
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{report_name, report_for, table_name, column_name, row_name}}{Each is
a character vector for subsetting when querying the SQL
database.  For the meaning of each argument, please see the
description above.}

\item{\code{wide}}{If \code{TRUE}, each table will be converted into the similar
format as it is shown in EnergyPlus HTML output file. Default:
\code{FALSE}.}

\item{\code{string_value}}{Only applicable when \code{wide} is \code{TRUE}. If
\code{string_value} is \code{FALSE}, instead of keeping all values as
characters, values in possible numeric columns are converted
into numbers. Default: the opposite of \code{wide}. Possible
numeric columns indicate column that:
\itemize{
\item columns that have associated units
\item columns that contents numbers
}}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$tabular_data()} extracts the tabular data in a
\code{\link[data.table:data.table]{data.table::data.table()}} using report, table, column and row name
specifications. The returned \code{\link[data.table:data.table]{data.table::data.table()}} has
9 columns:
\itemize{
\item \code{case}: Simulation case specified using \code{case} argument
\item \code{index}: Tabular data index
\item \code{report_name}: The name of the report that the record belongs to
\item \code{report_for}: The \code{For} text that is associated with the record
\item \code{table_name}: The name of the table that the record belongs to
\item \code{column_name}: The name of the column that the record belongs to
\item \code{row_name}: The name of the row that the record belongs to
\item \code{units}: The units of the record
\item \code{value}: The value of the record \strong{in string format} by default
}

For convenience, input character arguments matching in
\verb{$tabular_data()} are \strong{case-insensitive}.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} with 8 columns (when \code{wide} is
\code{FALSE}) or a named list of \code{\link[data.table:data.table]{data.table::data.table()}}s where the
names are the combination of \code{report_name}, \code{report_for} and
\code{table_name}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read all tabular data
job$tabular_data()

# explicitly specify data you want
str(job$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy"
))

# get tabular data in wide format and coerce numeric values
str(job$tabular_data(
    report_name = "AnnualBuildingUtilityPerformanceSummary",
    table_name = "Site and Source Energy",
    column_name = "Total Energy",
    row_name = "Total Site Energy",
    wide = TRUE, string_value = FALSE
))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EplusJob-print"></a>}}
\if{latex}{\out{\hypertarget{method-EplusJob-print}{}}}
\subsection{Method \code{print()}}{
Print \code{EplusSql} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EplusJob$print()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$print()} shows the core information of this \code{EplusJob} object,
including the path of model and weather, the version and path of
EnergyPlus used to run simulations, and the simulation job status.

\verb{$print()} is quite useful to get the simulation status, especially
when \code{wait} is \code{FALSE} in \verb{$run()}. The job status will be updated
and printed whenever \verb{$print()} is called.
}

\subsection{Returns}{
The \code{EplusSql} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
job$print()
}

}
\if{html}{\out{</div>}}

}

}
}
