% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aim.R
\name{rodeo.aim}
\alias{rodeo.aim}
\title{Regularised Ordinary Differential Equation Optimisation (RODEO) initialised via Adaptive Integral Matching}
\usage{
\method{rodeo}{aim}(x, adjusts = c("lambda"), trace = FALSE, ...)
}
\arguments{
\item{x}{\code{aim}-object created via \code{\link{aim}}-function.}

\item{adjusts}{Character vector holding names of what quantities to adjust during algorithm. Possible quantities are: \code{"lambda"}, \code{"scales"} and \code{"weights"}.}

\item{trace}{Logical indicating if status messages should be printed during \code{rodeo}.}

\item{...}{Additional arguments passed to \code{rodeo}.}
}
\value{
An object with S3 class "rodeo":
\item{o}{Original \code{ode}-object.}
\item{op}{Original \code{opt}-object with default values for \code{lambda_min_ratio}, \code{lambda} and (if needed) \code{a} inserted, if these were originally \code{NULL}.}
\item{params}{Parameter estimates, stored as list of sparse column format matrices, "dgCMatrix" (or a list of those if multiple initialisations). Rows represent coordinates and columns represent the \code{lambda} value.}
\item{x0s}{Initial state estimates stored in a matrix (or array). Rows represent coordinates, columns represent the \code{lambda} value and (if multiple initialisations) slices represent initialisations.}
\item{dfs}{A matrix (or array, if multiple initialisations) of degrees of freedom. Row represents a parameter (the first is always the initial state parameter), columns represent lambda, slices represent initialisation, if multiple are provided.}
\item{codes}{A matrix (or array) of convergence codes organised as \code{dfs}.
\describe{
\item{0:}{The convergence criteria is met (see details in \code{\link{opt}}). Current estimate is probably a local minimum.}
\item{1:}{Backtracking in the last iteration yields no numerical improvement, but no unsual behavior observed. Current estimate is probably a local minimum. However, if \code{exact_gradient} = FALSE in the \code{reg}-object in the \code{ode}-object, changing this may improve the code. Alternatively one can adjust backtracking via \code{backtrack_max} and \code{tau_min} in \code{reg} objects in \code{ode} object.}
\item{2:}{The optimisation procedure exceeded maximal number of steps (\code{step_max} in \code{reg} objects).}
\item{3:}{The last gradient was unsually large. Either the tolerances in \code{reg} objects are off or the ODE systems is very sensitive and runs over long time spans. In the latter case, initialisation(s) may have inappropriate zeros (change initialisation and/or make sure they start at smaller lambda value).}
\item{4:}{The numeric ODE solver exceeded maximal number of steps. Check if supplied initial states were out of bounds, if not increase \code{step_max} (or \code{tol}) in \code{reg}-objects in \code{ode}-object.}
}
}
\item{steps}{A matrix (or array) holding number of steps used in optimisation procedure. Organised as \code{dfs}.}
\item{losses}{A vector (or matrix) of unpenalised losses at optimum for each lambda value (stored row-wise if  multiple are provided).}
\item{penalties}{A matrix (or array) of penalties for each parameter, organised as \code{dfs}.}
\item{jerr}{A matrix (or array) of summary codes (for internal debugging), organised as \code{dfs}.}
}
\description{
Fit the parameters for an ODE model with data sampled across different contexts.
}
\details{
The adapted quantities (\code{scales}, \code{weights}, \code{penalty_factor}) of \code{x} (returned by \code{\link{aim}}) are fed to the exact estimator \code{rodeo}. This estimator then traverses the \code{lambda} sequence in reverse order initialised in the last estimates from \code{aim}.

If desired, the quantities \code{lambda}, \code{scales} and \code{weights} are adjusted as in \code{aim}.
}
\examples{
set.seed(123)
# Example: Power Law Kinetics
A <- matrix(c(1, 0, 1,
  1, 1, 0), byrow = TRUE, nrow = 2)
p <- plk(A)
x0 <- c(10, 4, 1)
theta <- matrix(c(0, -0.25,
  0.75, 0,
  0, -0.1), byrow = TRUE, nrow = 3)
Time <- seq(0, 1, by = .025)

# Simulate data
y <- numsolve(p, Time, x0, theta)
y[, -1] <- y[, -1] + matrix(rnorm(prod(dim(y[, -1])), sd = .25), nrow = nrow(y))

# Estimation via aim
a <- aim(p, opt(y, nlambda = 10))
a$params$theta

# Supply to rodeo
rod <- rodeo(a)
rod$params$theta

# Compare with true parameter on column vector form
matrix(theta, ncol = 1)


# Example: include data from an intervened system
# where the first complex in A is inhibited
contexts <- cbind(1, c(0, 0, 0, 1, 1, 1))
y2 <- numsolve(p, Time, x0 + 1, theta * contexts[, 2])
y2[, -1] <- y2[, -1] + matrix(rnorm(prod(dim(y2[, -1])), sd = .25), nrow = nrow(y2))

# Estimation via aim
a <- aim(plk(A, r = reg(contexts = contexts)), opt(rbind(y, y2), nlambda = 10))
a$params$theta

# Supply to rodeo
rod <- rodeo(a)
rod$params$theta


}
\seealso{
rodeo, aim, rodeo.ode
}
