#' Calculate statistics about nests
#' @title Calculte statistics about nests
#' @author Marc Girondot
#' @return Return or the total likelihood or a list with $metric and $summary depending on out parameter
#' @param NestsResult A NestsResult file generated by searchR
#' @param parameters A set of parameters if result is not provided.
#' @param SexualisationTRN A set of parameters used to model thermal reaction norm during TSP
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if result is not provided, or replace the one in NestsResult. Use SE=NA to remove SE from NestResult
#' @param temperatures Timeseries of temperatures formatted using formatNests(). Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings
#' @param weight Weights of the different nests to estimate likelihood
#' @param stopattest TRUE or FALSE. If TRUE, the model stops when proxy of size reached the mean test size.
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be displayed. Only one series can be displayed at a time.
#' @param TSP.borders The limits of TSP in stages. See embryo.stages parameter.
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided to estimate TSP borders. See note.
#' @param replicate.CI Number of replicates to estimate CI. If 1, no CI is estimated.
#' @param fill Number of minutes between two records. Create new one if they do not exist. NULL does not change the time of temperature recordings.
#' @param out Can take the values of likelihood or metric or summary. metric and summary can be combined: c("metric", "summary")
#' @param metric.end.incubation The expected metric at the end of incubation. Used to calibrate TSP size. If NULL, take the maximum Mean of the test parameter. If NA, use the actual final size. Can be a vector and is recycled if necessary.
#' @param progress If FALSE, the progress bar is not shown (useful for using with sweave or knitr)
#' @param warnings if FALSE, does not show warnings
#' @description This function calculates many statistics about nests.\cr
#' The embryo.stages is a list with stage numbers and relative size as compared to final size at the beginning of the stage.\cr
#' For example embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33) indicates that the stages 21 begins at the relative size of 8.4/39.33.\cr
#' Series can be indicated as the name of the series, their numbers or series or sucession of TRUE or FALSE. "all" indicates that all series must be analyzed.\cr
#' The likelihood object is just the total likelihood of the data in the model.\cr
#' The summary object is a data.frame composed of these elements:
#' \itemize{
#'   \item \code{TimeWeighted.temperature.mean} Average temperature weigthed by the time at each temperature
#'   \item \code{TimeWeighted.temperature.se} Standard error for the average temperature weigthed by the time at each temperature
#'   \item \code{TSP.TimeWeighted.temperature.mean} Average temperature during the TSP weigthed by the time at each temperature
#'   \item \code{TSP.TimeWeighted.temperature.se} Standard error for the average temperature during the TSP weigthed by the time at each temperature
#'   \item \code{TSP.MassWeighted.temperature.mean} Average temperature during the TSP weigthed by the mass increase at each temperature
#'   \item \code{TSP.MassWeighted.temperature.se} Standard error for the average temperature during the TSP weigthed by the mass increase at each temperature
#'   \item \code{TSP.STRNWeighted.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{TSP.STRNWeighted.temperature.se} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{TSP.MassWeighted.STRNWeighted.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{TSP.MassWeighted.STRNWeighted.temperature.se} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{TSP.length.mean} Average length of the TSP
#'   \item \code{TSP.length.se} Standard error for length of the TSP
#'   \item \code{TSP.begin.mean} Average time of the beginning of the TSP
#'   \item \code{TSP.begin.se} Standard error for the beginning of the TSP
#'   \item \code{TSP.end.mean} Average time of the endd of the TSP
#'   \item \code{TSP.end.se} Standard error for the end of the TSP
#'   \item \code{Incubation.length.mean} Average number of days for the incubation length
#'   \item \code{Incubation.length.se} Standard error for number of days for the incubation length
#'   \item \code{Middlethird.begin.mean} Average day at which the middle third incubation begins
#'   \item \code{Middlethird.begin.se} Standard error for day at which the middle third incubation begins
#'   \item \code{Middlethird.end.mean} Average day at which the middle third incubation ends
#'   \item \code{Middlethird.end.se} Standard error at which the middle third incubation ends
#'   \item \code{Middlethird.length.mean} Average days for second third of incubation
#'   \item \code{Middlethird.length.se} Standard error for second third of incubation
#'   \item \code{MiddleThird.TimeWeighted.temperature.mean} Average temperature during the middle third of incubation
#'   \item \code{MiddleThird.TimeWeighted.temperature.se} Standard error for the temperature during the middle third of incubation
#'   \item \code{MiddleThird.MassWeighted.temperature.mean} Average temperature during the middle third of incubation weight by embryo growth
#'   \item \code{MiddleThird.MassWeighted.temperature.se} Standard error for temperature during the middle third of incubation weight by embryo growth
#' }
#' The metric object is a list composed of data.frames which have the following attributes for each element of the list:
#' \itemize{
#'   \item \code{time.begin.tsp} Average time at the beginning of TSP
#'   \item \code{time.end.tsp} Average time at the end of TSP
#'   \item \code{time.begin.middlethird} Average time at the beginning of the middle third of incubation
#'   \item \code{time.begin.middlethird} Average time at the end of the middle third of incubation
#'   \item \code{time.begin.tsp.se} Standard error for the time at the beginning of TSP
#'   \item \code{time.end.tsp.se} Standard error for the time at the end of TSP
#'   \item \code{time.begin.middlethird.se} Standard error for the time at the beginning of the middle third of incubation
#'   \item \code{time.end.middlethird.se} Standard error for the time at the end of the middle third of incubation
#'   \item \code{stopattest} NA if stopattest was false, TRUE if at least one incubation series was longer than hatchling size and FALSE at contrary
#' }
#' If you indicate new set of temperatures, you must probably also indicate new test values.\cr
#' Note: three species have predefined embryo stages. embryo.stages parameter can take the values:\cr
#' \itemize{
#'   \item \code{Caretta caretta.SCL}
#'   \item \code{Chelonia mydas.SCL}
#'   \item \code{Emys orbicularis.SCL}
#'   \item \code{Emys orbicularis.mass}
#'   }
#' The progress bar is based on replicates and timeseries.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' summary.nests <- info.nests(resultNest_4p, out="summary", replicate.CI=20, 
#'   SE=c(DHA=17.7357362231266, DHH=55.2002231419364, 
#'   T12H=0.850237694629938, Rho25=8.47868153136681))
#'   # Result is in summary.nests$summary
#' infoall <- info.nests(resultNest_4p)
#'   # Result is a value
#' infoall.df <- info.nests(resultNest_4p, out="metric", replicate.CI=20, 
#'   SE=c(DHA=17.7357362231266, DHH=55.2002231419364, 
#'   T12H=0.850237694629938, Rho25=8.47868153136681)))
#' # Result is in summary.nests$metric
#' infoall.both <- info.nests(resultNest_4p, out=c("metric", "summary"), replicate.CI=20, 
#'   SE=c(DHA=17.7357362231266, DHH=55.2002231419364, 
#'   T12H=0.850237694629938, Rho25=8.47868153136681)))
#' # Result is in summary.nests$summary and in summary.nests$metric
#' }
#' @export


  info.nests <- function(NestsResult=NULL, parameters=NULL, fixed.parameters=NULL, 
                         SE=NULL, temperatures=NULL, derivate=NULL, 
                         test=NULL, stopattest=FALSE, M0=NULL, series="all",
                         TSP.borders=NULL, embryo.stages="Caretta caretta.SCL",
                         replicate.CI=1, weight=NULL, out="Likelihood", fill=NULL, 
                         SexualisationTRN=NULL, metric.end.incubation=NULL,
                         progress=TRUE, warnings=TRUE) {

    # NestsResult=NULL; parameters=NULL; fixed.parameters=NULL; SE=NULL; temperatures=NULL; derivate=NULL; test=NULL; stopattest=FALSE; M0=NULL; series="all"; TSP.borders=NULL; embryo.stages="Caretta caretta.SCL"; replicate.CI=1; weight=NULL; out="Likelihood"; fill=NULL; SexualisationTRN=NULL; metric.end.incubation=NULL; progress=TRUE
    # NestsResult=resultNest_4p
    out <- tolower(out)
    TSP.list <- embryogrowth::TSP.list
    
    if (class(NestsResult)=="NestsResult") {
      # temperatures est un objet Nests
      if (is.null(temperatures)) temperatures <- NestsResult$data
      if (is.null(derivate)) derivate <- NestsResult$derivate
      if (is.null(weight)) weight <- NestsResult$weight
      if (is.null(test)) test <- NestsResult$test
      if (is.null(M0)) M0 <- NestsResult$M0
      if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
      if (is.null(SE)) SE <- NestsResult$SE
      if (is.null(parameters)) parameters <- NestsResult$par
    }

    # 16/9/2015 pour retirer un SE si il existe dans NestsResult
    if (!is.null(SE[1])) if (all(is.na(SE[])))  SE <- NULL
    
    if (is.null(SE) | out[1]=="likelihood") replicate.CI <- 1
    
    if (replicate.CI!=1) if (all(is.na(SE))) replicate.CI <- 1
    
    if (is.numeric(NestsResult) & is.null(parameters)) parameters <- NestsResult
    
    NBTs <- temperatures[["IndiceT"]][3]
    if (is.null(series[1]) | (series[1]=="all")) series <- rep(TRUE, NBTs)
    
    name <- names(temperatures[1:NBTs])
    
    if (is.null(test)) {
      # si tous sont pareils, je reprends les memes
      # Correction d'un bug, rajout de [[1]] dans result$test["Mean"][[1]][1] 30/7/2012
      if (all(NestsResult$test["Mean"]==NestsResult$test["Mean"][[1]][1]) & all(NestsResult$test["SD"]==NestsResult$test["SD"][[1]][1])) {
        test <- c(Mean=NestsResult$test["Mean"][[1]][1], SD=NestsResult$test["SD"][[1]][1])
      } else {	
        stop("The size at hatching (test parameter) must be provided.")
      }
    }
    
    
    if (class(test) == "numeric") {
      test <- data.frame(Mean=rep(test["Mean"], length(name)), SD=rep(test["SD"], length(name)), row.names=name)
    }
    
    if (is.null(metric.end.incubation)) {
      metric.end.incubation <- rep(max(test[, "Mean"]), NBTs)
      names(metric.end.incubation) <- name
    }
    # if (is.na(metric.end.incubation)) metric.end.incubation <- 1
    
    if (length(metric.end.incubation) != NBTs | is.null(names(metric.end.incubation))) {
     metric.end.incubation <- rep(metric.end.incubation, NBTs)[1:NBTs]
     names(metric.end.incubation) <- name
    }
    
    if (class(embryo.stages)=="character") {
      estages <- TSP.list[[gsub(" ", "_", embryo.stages)]]
      if (is.null(estages)) {
        stop("The TSP for ", embryo.stages, " does not exist")
      } else {
        embryo.stages <- estages[, "metric"]
        names(embryo.stages) <- estages[, "stages"]
        TSP.borders <- c(attributes(estages)$TSP.begin.stages, attributes(estages)$TSP.end.stages)
      }
    }
    
    # La taille au dbut et fin de TSP devrait dependre de la taille finale
    # A mditer
    # C'est resolu le 16/9/2015
    size.begin.TSP <- unname(embryo.stages[as.character(TSP.borders[1])])
    size.end.TSP <- unname(embryo.stages[as.character(TSP.borders[2])])
    
    # dans x j'ai les paramtres  ajuster
    # Il faut que je rajoute les fixes - 16/7/2012
    x <- c(parameters, fixed.parameters)
    logicTransition <- (is.na(x["transition_P"]) | is.na(x["transition_S"]))
    # je gere les se
    df_random <- data.frame(fake=rep(NA, replicate.CI))

    if (is.null(SE)) {
      SE[] <- 0
    }

    SEp <- x
    SEp[] <- SE[match(names(SEp), names(SE))]
    SEp[is.na(SEp)] <- 0
    
    s <- seq_along(x)
    names(s) <- names(x)
    df_random <- as.data.frame(lapply(s, function(indx) c(x[indx], rnorm(replicate.CI-1, x[indx], SEp[names(x)[indx]]))))

    
    temperatures_ec <- temperatures[1:NBTs][series]
    xx <- seq_along(temperatures_ec)
    names(xx) <- names(temperatures_ec)
    
    # prepare le df pour les resultats
    temperatures_ec <- mclapply(xx, function (xxx) {
#      temperatures_ec <- lapply(xx, function (xxx) {
        # xxx <- 1
    df <- as.data.frame(temperatures_ec[[xxx]])
    colnames(df) <- c("Time", "TempC", "TempK", "R", "SCL", "IndiceK")
    if (!is.null(fill)) {
      newt <- seq(from=fill, to=tail(df$Time, n=1L), by=fill)
      aretirer <- -which(!is.na(match(newt, df$Time)))
      if (!identical(aretirer, integer(0))) newt <- newt[aretirer]
      df <- rbind(df, 
                  data.frame(Time=newt, TempC=NA, 
                             TempK=NA, R=NA, SCL=NA, IndiceK=NA)) 
      df <- df[order(df$Time),]
    }
    
    df <- cbind(df, DeltaT=c(diff(df$Time), 0))

    ind <- which(!is.na(df$TempC))      # get positions of nonmissing values
    if(is.na(df$TempC[1]))             # if it begins with a missing, add the 
      ind <- c(1,ind)        # first position to the indices
    df$TempC <- rep(df$TempC[ind], times = diff(   # repeat the values at these indices
      c(ind, length(df$TempC) + 1) ))
    df$TempK <- rep(df$TempK[ind], times = diff(   # repeat the values at these indices
      c(ind, length(df$TempK) + 1) ))
    df$IndiceK <- rep(df$IndiceK[ind], times = diff(   # repeat the values at these indices
      c(ind, length(df$IndiceK) + 1) ))
    
    df[1, "SCL"] <- M0
    return(df)
    })
    
    returntotal.summary <- list()
    returntotal.likelihood <- list()
    returntotal.metric <- list()
    
    if (progress) pb<-txtProgressBar(min=0, max=replicate.CI, style=3)
    
    
    for(sp in 1:replicate.CI) {
      if (progress) setTxtProgressBar(pb, sp)
# sp <- 1      
# if (sp==48) browser()
#     print(sp)

      x <- as.numeric(df_random[sp, ])
      names(x) <- gsub("^X", "", colnames(df_random))
      if (!logicTransition) {
        transition_P <- x["transition_P"]
        transition_S <- x["transition_S"]
      }
      
      # Ce sont toutes les temperatures de tous les nids
      tempK <- as.numeric(temperatures[["Temperatures"]])
      rlist <- getFromNamespace(".SSM", ns="embryogrowth")(tempK, x)
      r <- rlist[[1]]
      r_L <- rlist[[2]]
      
      
      if (!is.na(x["K"])) {
        Kval <- unname(x["K"])
      } else {
        Kval <- NULL
      }
      
       AnalyseTraces <- mclapply(xx, function (xxx) {
#      AnalyseTraces <- lapply(xx, function (xxx) {
        # xxx <- 1
 #       print(xxx)
        df <- temperatures_ec[[xxx]]
        namets <- names(temperatures_ec)[xxx]
        
        meanSCL <- as.numeric(test[namets, "Mean"])
        sdSCL <- as.numeric(test[namets, "SD"])
        
        if (is.na(meanSCL) | is.na(sdSCL)) stop("Check the test parameter. The size for time series does not exist.")
        
        if (!is.na(x["rK"])) {
          Kval <- unname(x["rK"]*meanSCL)
        }

        tmin <- df[, "Time"]

        ldt <- which(!is.na(df[, "TempC"]))
        IndK <- df[, "IndiceK"]
        
        if (!stopattest) {
          if (logicTransition) {  
            for (i in 1:(length(ldt)-1)) {
              y <- df[ldt[i], "SCL"]
              range <- c(ldt[i]:ldt[i+1])
              timesunique <- tmin[range]
              a <- r[IndK[ldt[i]]]
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              y <- as.numeric(tail(out1[,2], n=1))
              df[c(ldt[i]:(ldt[i+1]-1)), "R"] <- a
              df[range, "SCL"] <- out1[,2]
            }
          } else {
            for (i in 1:(length(tmin)-1)) {
              y <- df[i, "SCL"]
              timesunique <- c(tmin[i], tmin[i+1])
              transition <- 1/(1+exp(transition_S*(y-transition_P)))
              if (!is.na(IndK[i])) tk <- IndK[i]
              a <- r[tk]*transition+r_L[tk]*(1-transition)
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              y <- as.numeric(tail(out1[,2], n=1))
              df[i, "R"] <- a
              df[i+1, "SCL"] <- y
            }
          }
          # je suis en stopattest
        } else {
          if (logicTransition) {  
            for (i in 1:(length(ldt)-1)) {
              y <- df[ldt[i], "SCL"]
              range <- c(ldt[i]:ldt[i+1])
              timesunique <- tmin[range]
              a <- r[IndK[ldt[i]]]
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              y <- as.numeric(tail(out1[,2], n=1))
              df[c(ldt[i]:(ldt[i+1]-1)), "R"] <- a
              df[range, "SCL"] <- out1[,2]
              if (y>meanSCL) {
                df <- df[1:ldt[i+1], ]
                df <- df[1:which(df[, "SCL"] >meanSCL)[1],]
                y <- tail(df[, "SCL"], n=1)
                break
              }
            }
          } else {
            for (i in 1:(length(tmin)-1)) {
              y <- df[i, "SCL"]
              timesunique <- c(tmin[i], tmin[i+1])
              transition <- 1/(1+exp(transition_S*(y-transition_P)))
              if (!is.na(IndK[i])) tk <- IndK[i]
              a <- r[tk]*transition+r_L[tk]*(1-transition)
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              y <- as.numeric(tail(out1[,2], n=1))
              df[i, "R"] <- a
              df[i+1, "SCL"] <- y
              if (y>meanSCL) {
                df <- df[1:(i+1), ]
                df <- df[1:which(df[, "SCL"] >meanSCL)[1],]
                y <- tail(df[, "SCL"], n=1)
                break
              }
            }
          }
        }
        
        
        #  names(L) <- namets
        if (out[1]=="likelihood") {
          # dans y j'ai une valeur de taille finale
          # print(-dnorm(y, mean=meanSCL, sd=sdSCL, log=TRUE))
          return(-dnorm(y, mean=meanSCL, sd=sdSCL, log=TRUE))
        }
        
        
 #       if (out=="metric" | out=="summary") { 
          # si metric ou summary
          # les valeurs sont fausses pour transition
        
        # je suis sois en metric soit en summary

###############################
#     indice.begin.tsp        #
###############################

        # 16/9/2015 Mme si je suis en stopattest, je prends la valeur finale si trop court        
#       if (stopattest) {
#          meanSCL_ec <- meanSCL
#        } else {
#          meanSCL_ec <- tail(df[,"SCL"], n=1)
#        }
        
        meanSCL_ec <- y
        
        
        
        if (stopattest & meanSCL_ec <= meanSCL) {
          # warning(paste("Serie", xxx, "for replicate", sp, "stops before the mean hatchling metric"))
          retWrong <- FALSE
          attributes(retWrong)$error <- paste("Serie", xxx, "for replicate", sp, "stops before the mean hatchling metric")
          return(retWrong)
        }
        
        
        if (is.na(metric.end.incubation[namets])) {
          size.begin.TSP_ec <- size.begin.TSP*meanSCL_ec
          size.end.TSP_ec <- size.end.TSP*meanSCL_ec
        } else {
          size.begin.TSP_ec <- size.begin.TSP*metric.end.incubation[namets]
          size.end.TSP_ec <- size.end.TSP*metric.end.incubation[namets]
        }
        
        df_ec <- df
        
        if (meanSCL_ec < size.begin.TSP_ec) {
          time.begin.TSP <- NA
          indice.begin.tsp <- NA
        } else {

        repeat {
          indice.begin.tsp <- which(df_ec[,"SCL"]>(size.begin.TSP_ec))[1]-1
          if (is.na(indice.begin.tsp)) break
          # la vrai TSP commence entre indice.begin.tsp et indice.begin.tsp+1
          timei1 <- df_ec[indice.begin.tsp,"Time"]+df_ec[indice.begin.tsp,"DeltaT"]*1/3
          timei2 <- df_ec[indice.begin.tsp,"Time"]+df_ec[indice.begin.tsp,"DeltaT"]*2/3
          timesunique <- c(df_ec[indice.begin.tsp, "Time"], timei1, timei2, df_ec[indice.begin.tsp+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df_ec[indice.begin.tsp, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df_ec[indice.begin.tsp, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          df_ec <- data.frame(Time=timesunique, R=a, SCL=dfpol[,"SCL"])
          df_ec <- cbind(df_ec, DeltaT=c(diff(df_ec[, "Time"]), 0))
          if (df_ec[1, "DeltaT"]<1) break
        }
        indice.begin.tsp <- which(df_ec[,"SCL"]>(size.begin.TSP_ec))[1]-1
        if (is.na(indice.begin.tsp)) indice.begin.tsp <- 4
        time.begin.TSP <- df_ec[indice.begin.tsp,"Time"]
        indice.begin.tsp <- which(df[,"SCL"]>(size.begin.TSP_ec))[1]-1
        }

###############################
#       indice.end.tsp        #
###############################
          
        df_ec <- df
        
        if (meanSCL_ec < size.end.TSP_ec) {
          time.end.TSP <- NA
          indice.end.tsp <- NA
        } else {
        
        repeat {
          indice.end.tsp <- which(df_ec[,"SCL"]>(size.end.TSP_ec))[1]-1
          if (is.na(indice.end.tsp)) break
          # la vrai TSP fini entre indice.end.tsp et indice.end.tsp+1
          timei1 <- df_ec[indice.end.tsp,"Time"]+df_ec[indice.end.tsp,"DeltaT"]*1/3
          timei2 <- df_ec[indice.end.tsp,"Time"]+df_ec[indice.end.tsp,"DeltaT"]*2/3
          timesunique <- c(df_ec[indice.end.tsp, "Time"], timei1, timei2, df_ec[indice.end.tsp+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df_ec[indice.end.tsp, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df_ec[indice.end.tsp, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          df_ec <- data.frame(Time=timesunique, R=a, SCL=dfpol[,"SCL"])
          df_ec <- cbind(df_ec, DeltaT=c(diff(df_ec[, "Time"]), 0))
          if (df_ec[1, "DeltaT"]<1) break
        }
        indice.end.tsp <- which(df_ec[,"SCL"]>(size.end.TSP_ec))[1]-1
        if (is.na(indice.end.tsp)) indice.end.tsp <- 4
        time.end.TSP <- df_ec[indice.end.tsp,"Time"]
        indice.end.tsp <- which(df[,"SCL"]>(size.end.TSP_ec))[1]-1
        }

          
          # C'est necessaire pour estimer les temperatures moyennes
          # on doit les garder
        
        if (any(out=="summary")) {
          
          if (!is.na(indice.end.tsp) & !is.na(indice.begin.tsp)) {
            
          if (indice.begin.tsp!=indice.end.tsp)  {
          
          df <- rbind(df[1:indice.begin.tsp,], 
                      c(Time=time.begin.TSP, 
                        TempC=df[indice.begin.tsp,"TempC"], 
                        TempK=df[indice.begin.tsp,"TempK"], 
                        R=df[indice.begin.tsp,"R"], 
                        SCL=size.begin.TSP_ec, 
                        IndiceK=df[indice.begin.tsp,"IndiceK"], 
                        DeltaT=NA),
                      df[(indice.begin.tsp+1):indice.end.tsp,],
                      c(Time=time.end.TSP, 
                        TempC=df[indice.end.tsp,"TempC"], 
                        TempK=df[indice.end.tsp,"TempK"], 
                        R=df[indice.end.tsp,"R"], 
                        SCL=size.end.TSP_ec, 
                        IndiceK=df[indice.end.tsp,"IndiceK"], 
                        DeltaT=NA),
                      df[(indice.end.tsp+1):nrow(df),])
          } else {
            df <- rbind(df[1:indice.begin.tsp,], 
                        c(Time=time.begin.TSP, 
                          TempC=df[indice.begin.tsp,"TempC"], 
                          TempK=df[indice.begin.tsp,"TempK"], 
                          R=df[indice.begin.tsp,"R"], 
                          SCL=size.begin.TSP_ec, 
                          IndiceK=df[indice.begin.tsp,"IndiceK"], 
                          DeltaT=NA),
                        c(Time=time.end.TSP, 
                          TempC=df[indice.end.tsp,"TempC"], 
                          TempK=df[indice.end.tsp,"TempK"], 
                          R=df[indice.end.tsp,"R"], 
                          SCL=size.end.TSP_ec, 
                          IndiceK=df[indice.end.tsp,"IndiceK"], 
                          DeltaT=NA),
                        df[(indice.end.tsp+1):nrow(df),])
          }
          
          df[, "DeltaT"] <- c(diff(df[, "Time"]), 0)
        }

        }

###############################
#   indice.fin.incubation     #
###############################
          # si je suis en stopattest, il faut que je calcule la vraie valeur de fin
          if (stopattest) {
            
            if (tail(df[,"SCL"], n=1L)>meanSCL) {
              attSAT <- TRUE
            df_ec <- df
            
            repeat {
            indice.fin.incubation <- which(df_ec[,"SCL"]>(meanSCL))[1]-1
            if (is.na(indice.fin.incubation)) break
            timei1 <- df_ec[indice.fin.incubation,"Time"]+df_ec[indice.fin.incubation,"DeltaT"]*1/3
            timei2 <- df_ec[indice.fin.incubation,"Time"]+df_ec[indice.fin.incubation,"DeltaT"]*2/3
            timesunique <- c(df_ec[indice.fin.incubation, "Time"], timei1, timei2, df_ec[indice.fin.incubation+1, "Time"])
            # si je suis en transition, je dois recalculer R
            a <- df_ec[indice.fin.incubation, "R"]
            param <- c(alpha=unname(a), K=Kval)
            y <- df_ec[indice.fin.incubation, "SCL"]
            names(y) <- "SCL"
            dfpol <- lsoda(y, timesunique, derivate, param)
            df_ec <- data.frame(Time=timesunique, R=a, SCL=dfpol[,"SCL"])
            df_ec <- cbind(df_ec, DeltaT=c(diff(df_ec[, "Time"]), 0))
            if (df_ec[1, "DeltaT"]<1) break
            }
            
            indice.end.incubation <- which(df_ec[,"SCL"]>(meanSCL))[1]-1
            if (is.na(indice.end.incubation)) indice.end.incubation <- 4
            time.end.incubation <- df_ec[indice.end.incubation,"Time"]
            indice.end.incubation <- which(df[,"SCL"]>(meanSCL))[1]-1
            
            # Maintenant je tronque df de faon  ne garder que jusqu' time.end.incubation
            # garde
            df <- df[df[, "SCL"]<meanSCL,]
            df <- rbind(df, c(Time=time.end.incubation, 
                              TempC=tail(df$TempC, n=1L), tempK=tail(df$TempK, n=1L), 
                              R=NA, SCL=meanSCL, 
                              IndiceK=tail(df$IndiceK, n=1L), DeltaT=0))
            df[, "DeltaT"] <- c(diff(df[, "Time"]), 0)
      } else {attSAT <- FALSE}
          } else {attSAT <- NA}

###################################
#    indice.begin.middlethird     #
###################################
          
          time.begin.middlethird <- tail(df$Time, n=1L)/3
          indice.begin.middlethird <- which(df[,"Time"]>time.begin.middlethird)[1]-1
          # Je calcul la taille
          timesunique <- c(df[indice.begin.middlethird, "Time"], time.begin.middlethird, df[indice.begin.middlethird+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df[indice.begin.middlethird, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df[indice.begin.middlethird, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          size.begin.middlethird <- dfpol[2, "SCL"]

#################################
#    indice.end.middlethird     #
#################################
          
          time.end.middlethird <- tail(df$Time, n=1L)*2/3
          indice.end.middlethird <- which(df[,"Time"]>time.end.middlethird)[1]-1
          # Je calcul la taille
          timesunique <- c(df[indice.end.middlethird, "Time"], time.end.middlethird, df[indice.end.middlethird+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df[indice.end.middlethird, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df[indice.end.middlethird, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          size.end.middlethird <- dfpol[2, "SCL"]
  
          
          if (any(out=="summary")) {
          if (indice.begin.middlethird!=indice.end.middlethird) {
            
          df <- rbind(df[1:indice.begin.middlethird,], 
                      c(Time=time.begin.middlethird, 
                        TempC=df[indice.begin.middlethird,"TempC"], 
                        TempK=df[indice.begin.middlethird,"TempK"], 
                        R=df[indice.begin.middlethird,"R"], 
                        SCL=size.begin.middlethird, 
                        IndiceK=df[indice.begin.middlethird,"IndiceK"], 
                        DeltaT=NA),
                      df[(indice.begin.middlethird+1):indice.end.middlethird,],
                      c(Time=time.end.middlethird, 
                        TempC=df[indice.end.middlethird,"TempC"], 
                        TempK=df[indice.end.middlethird,"TempK"], 
                        R=df[indice.end.middlethird,"R"], 
                        SCL=size.end.middlethird, 
                        IndiceK=df[indice.end.middlethird,"IndiceK"], 
                        DeltaT=NA),
                      df[(indice.end.middlethird+1):nrow(df),])
          } else {
            df <- rbind(df[1:indice.begin.middlethird,], 
                        c(Time=time.begin.middlethird, 
                          TempC=df[indice.begin.middlethird,"TempC"], 
                          TempK=df[indice.begin.middlethird,"TempK"], 
                          R=df[indice.begin.middlethird,"R"], 
                          SCL=size.begin.middlethird, 
                          IndiceK=df[indice.begin.middlethird,"IndiceK"], 
                          DeltaT=NA),
                        c(Time=time.end.middlethird, 
                          TempC=df[indice.end.middlethird,"TempC"], 
                          TempK=df[indice.end.middlethird,"TempK"], 
                          R=df[indice.end.middlethird,"R"], 
                          SCL=size.end.middlethird, 
                          IndiceK=df[indice.end.middlethird,"IndiceK"], 
                          DeltaT=NA),
                        df[(indice.end.middlethird+1):nrow(df),])
          }

          df[,"DeltaT"] <- c(diff(df[,"Time"]), 0)
          }
          
#######################
# fin des indices     #
#######################                        
          
          rownames(df) <- 1:nrow(df)
          if (any(out=="summary")) {
           attributes(df)$indice.end.tsp <- which(df[, "Time"]==time.end.TSP)[1]
           attributes(df)$indice.begin.tsp <- which(df[, "Time"]==time.begin.TSP)[1]
           attributes(df)$indice.begin.middlethird <- which(df[, "Time"]==time.begin.middlethird)[1]
           attributes(df)$indice.end.middlethird <- which(df[, "Time"]==time.end.middlethird)[1]
          }
          attributes(df)$metric.begin.tsp <- size.begin.TSP_ec
          attributes(df)$metric.end.tsp <-  size.end.TSP_ec
          attributes(df)$test.mean <- meanSCL
          attributes(df)$test.sd <- sdSCL
          attributes(df)$time.end.tsp <- time.end.TSP
          attributes(df)$time.begin.tsp <- time.begin.TSP
          attributes(df)$metric.end.incubation <- metric.end.incubation[namets]
          attributes(df)$time.begin.middlethird <- time.begin.middlethird
          attributes(df)$time.end.middlethird <- time.end.middlethird
          attributes(df)$stopattest <- attSAT
          return(df)
          # fin de si metric ou summary
#        }
        # fin du mcapply
      })
      
      if (any(out=="likelihood")) {
        L <- unlist(AnalyseTraces)
        names(L) <- names(AnalyseTraces)
        # dans L j'ai un vecteur avec le nom
        # il faut que j'applique le weight
        if (!is.null(weight))	L <- L*weight[match(names(L), names(weight))]
        returntotal.likelihood <- (sum(L))
      }
      
      if (any(out=="metric")) {
        returntotal.metric <- c(returntotal.metric, list(AnalyseTraces))
      }
      
       if (any(out=="summary")) {
        
        if (any(unlist(lapply(AnalyseTraces, function(x) {attributes(x) <- NULL; identical(x = x, y=FALSE)})))) {
          if (warnings) warning("replicate ", sp, ": Something goes wrong")
          summarydf <- data.frame(
            TimeWeighted.temperature.mean=NA,
            TSP.TimeWeighted.temperature.mean=NA,
            TSP.MassWeighted.temperature.mean=NA,
            TSP.STRNWeighted.temperature.mean=NA,
            TSP.MassWeighted.STRNWeighted.temperature.mean=NA,
            TSP.length.mean=NA,
            TSP.begin.mean=NA,
            TSP.end.mean=NA,
            Incubation.length.mean=NA,
            Middlethird.length.mean=NA,
            Middlethird.begin.mean=NA,
            Middlethird.end.mean=NA,
            MiddleThird.TimeWeighted.temperature.mean=NA,
            MiddleThird.MassWeighted.temperature.mean=NA
          )
          returntotal.summary <- c(returntotal.summary, list(summarydf))
        } else {
        
        TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, function(x) sum(x[, "TempC"]*x[, "DeltaT"])/sum(x[, "DeltaT"])))
        TSP.TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                           function(x) {
                                                             if (!is.na(attributes(x)$indice.begin.tsp) & !is.na(attributes(x)$indice.end.tsp)) {
                                                             x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), ]
                                                             sum(x2[, "TempC"]*x2[, "DeltaT"])/sum(x2[, "DeltaT"])
                                                             } else {NA}
                                                           }))
        TSP.MassWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                           function(x) {
                                                             if (!is.na(attributes(x)$indice.begin.tsp) & !is.na(attributes(x)$indice.end.tsp)) {
                                                             x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), "TempC"]
                                                             dSCL <- c(diff(x[, "SCL"]), 0)[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1)]
                                                             sum(x2*dSCL)/sum(dSCL)
                                                             } else {NA}
                                                           }))
        if (is.null(SexualisationTRN)) {
          TSP.STRNWeighted.temperature.mean <- rep(NA, length(AnalyseTraces))
          TSP.MassWeighted.STRNWeighted.temperature.mean <- rep(NA, length(AnalyseTraces))
        } else {
          TSP.STRNWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                             function(x) {
                                                               if (!is.na(attributes(x)$indice.begin.tsp) & !is.na(attributes(x)$indice.end.tsp)) {
                                                                 # dans x[, "TempC"] j'ai les temperatures en Celsius
                                                               x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), ]
                                                               
                                                               STRN_model <- getFromNamespace(".SSM", ns="embryogrowth")(x2[, "TempK"], SexualisationTRN)[[1]]
                                                               sum(x2[, "TempC"]*x2[, "DeltaT"]*STRN_model)/sum(x2[, "DeltaT"]*STRN_model)
                                                             } else {NA}
                                                             }))
          
          TSP.MassWeighted.STRNWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                                          function(x) {
                                                                            if (!is.na(attributes(x)$indice.begin.tsp) & !is.na(attributes(x)$indice.end.tsp)) {
                                                                              # dans x[, "TempC"] j'ai les temperatures en Celsius
                                                                            x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), ]
                                                                            
                                                                            STRN_model <- getFromNamespace(".SSM", ns="embryogrowth")(x2[, "TempK"], SexualisationTRN)[[1]]
                                                                            dSCL <- c(diff(x[, "SCL"]), 0)[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1)]
                                                                            sum(x2[, "TempC"]*dSCL*STRN_model)/sum(dSCL*STRN_model)
                                                                            } else {NA}
                                                                          }))
        }
        
        TSP.length.mean <- unlist(lapply(AnalyseTraces, 
                                         function(x) {
                                           if (!is.na(attributes(x)$indice.begin.tsp) & !is.na(attributes(x)$indice.end.tsp)) {
                                           t1 <- attributes(x)$time.begin.tsp
                                           t2 <- attributes(x)$time.end.tsp
                                           return(t2-t1)
                                           } else {NA}
                                         }))
        TSP.begin.mean <- unlist(lapply(AnalyseTraces, 
                                        function(x) {
                                          return(attributes(x)$time.begin.tsp)
                                        }))
        TSP.end.mean <- unlist(lapply(AnalyseTraces, 
                                      function(x) {
                                        return(attributes(x)$time.end.tsp)
                                      }))
        Middlethird.length.mean <- unlist(lapply(AnalyseTraces, 
                                         function(x) {
                                           t1 <- attributes(x)$time.begin.middlethird
                                           t2 <- attributes(x)$time.end.middlethird
                                           return(t2-t1)
                                         }))
        Middlethird.begin.mean <- unlist(lapply(AnalyseTraces, 
                                                   function(x) {
                                                     return(attributes(x)$time.begin.middlethird)
                                                   }))
        Middlethird.end.mean <- unlist(lapply(AnalyseTraces, 
                                                function(x) {
                                                  return(attributes(x)$time.end.middlethird)
                                                }))
        Incubation.length.mean <- unlist(lapply(AnalyseTraces, 
                                                function(x) {
                                                  return(x[nrow(x), "Time"])
                                                }))

        MiddleThird.TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                           function(x) {
                                                             if (!is.na(attributes(x)$indice.begin.middlethird) & !is.na(attributes(x)$indice.end.middlethird)) {
                                                             x2 <- x[(attributes(x)$indice.begin.middlethird):(attributes(x)$indice.end.middlethird-1), ]
                                                             sum(x2[, "TempC"]*x2[, "DeltaT"])/sum(x2[, "DeltaT"])
                                                             } else {NA}
                                                           }))
        
        MiddleThird.MassWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                                   function(x) {
                                                                     if (!is.na(attributes(x)$indice.begin.middlethird) & !is.na(attributes(x)$indice.end.middlethird)) {
                                                                       x2 <- x[(attributes(x)$indice.begin.middlethird):(attributes(x)$indice.end.middlethird-1), "TempC"]
                                                                       dSCL <- c(diff(x[, "SCL"]), 0)[(attributes(x)$indice.begin.middlethird):(attributes(x)$indice.end.middlethird-1)]
                                                                       sum(x2*dSCL)/sum(dSCL)
                                                                     } else {NA}
                                                                   }))
        
        summarydf <- data.frame(
          TimeWeighted.temperature.mean=TimeWeighted.temperature.mean,
          TSP.TimeWeighted.temperature.mean=TSP.TimeWeighted.temperature.mean,
          TSP.MassWeighted.temperature.mean=TSP.MassWeighted.temperature.mean,
          TSP.STRNWeighted.temperature.mean=TSP.STRNWeighted.temperature.mean,
          TSP.MassWeighted.STRNWeighted.temperature.mean=TSP.MassWeighted.STRNWeighted.temperature.mean,
          TSP.length.mean=TSP.length.mean,
          TSP.begin.mean=TSP.begin.mean,
          TSP.end.mean=TSP.end.mean,
          Incubation.length.mean=Incubation.length.mean,
          Middlethird.length.mean=Middlethird.length.mean,
          Middlethird.begin.mean=Middlethird.begin.mean,
          Middlethird.end.mean=Middlethird.end.mean,
          MiddleThird.TimeWeighted.temperature.mean=MiddleThird.TimeWeighted.temperature.mean,
          MiddleThird.MassWeighted.temperature.mean=MiddleThird.MassWeighted.temperature.mean
        )
        returntotal.summary <- c(returntotal.summary, list(summarydf))
        
        }
      }
      
      # fin de la boucle des replicats
    }
    
    if (any(out=="likelihood")) return(returntotal.likelihood)
    
    ret.summary <- NULL
    ret.metric <- NULL
    
    if (any(out=="metric")) {
      
      # Il faut d'abord que je retire les temps crees pour le debut et la fin de la TSP
      # En fait le plus simple serait que je ne les mette pas si 
      # je suis en metric, seulement en summary
      
      # je sors la longeur max de la serie
      lmax <- sapply(returntotal.metric, function(x) sapply(x, function(y) dim(y)[1]))
      if (class(lmax)=="matrix") lmax <- lmax[,1]
      lmax <- aggregate(lmax, by=list(names(lmax)), max)
      rownames(lmax) <- lmax[,1]
      
      # l1 <- length(meanTotal[[j]])
      # l2 <- length(meanTotal_i[[j]])
      
      # D'abord je sors les valeurs pour le premier de la serie des randoms
      
      meanTotal <- lapply(returntotal.metric[[1]], function(x) x[,"SCL"])
      
      meanTotal <- lapply(lmax[,1], function(x) c(meanTotal[[x]], rep(0, lmax[x, 2]-length(meanTotal[[x]]))))
      names(meanTotal) <- lmax[,1]
      nbdif0 <- lapply(meanTotal, function(x) ifelse(x != 0, 1, 0))
      
      meanTotal2 <- lapply(returntotal.metric[[1]], function(x) x[,"SCL"]^2)
      meanTotal2 <- lapply(lmax[,1], function(x) c(meanTotal2[[x]], rep(0, lmax[x, 2]-length(meanTotal2[[x]]))))
      meanRTotal <- lapply(returntotal.metric[[1]], function(x) x[,"R"])
      meanRTotal <- lapply(lmax[,1], function(x) c(meanRTotal[[x]], rep(0, lmax[x, 2]-length(meanRTotal[[x]]))))
      meanRTotal2 <- lapply(returntotal.metric[[1]], function(x) x[,"R"]^2)
      meanRTotal2 <- lapply(lmax[,1], function(x) c(meanRTotal2[[x]], rep(0, lmax[x, 2]-length(meanRTotal2[[x]]))))
      
      att <- lapply(returntotal.metric[[1]], function(x) c(
        attributes(x)[["time.begin.tsp"]], 
        attributes(x)[["time.end.tsp"]], 
        attributes(x)[["time.begin.middlethird"]],
        attributes(x)[["time.end.middlethird"]],
        attributes(x)[["metric.begin.tsp"]],
        attributes(x)[["metric.end.tsp"]], 
        attributes(x)[["test.mean"]], 
        attributes(x)[["test.sd"]], 
        attributes(x)[["metric.end.incubation"]]
      ))
      att2 <- lapply(returntotal.metric[[1]], function(x) c(
        attributes(x)[["time.begin.tsp"]]^2, 
        attributes(x)[["time.end.tsp"]]^2, 
        attributes(x)[["time.begin.middlethird"]]^2,
        attributes(x)[["time.end.middlethird"]]^2,
        attributes(x)[["metric.begin.tsp"]]^2,
        attributes(x)[["metric.end.tsp"]]^2, 
        attributes(x)[["test.mean"]]^2, 
        attributes(x)[["test.sd"]]^2, 
        attributes(x)[["metric.end.incubation"]]^2
      ))
      att_sat <- lapply(returntotal.metric[[1]], function(x) c(
        attributes(x)[["stopattest"]]
      ))
      
      if (length(returntotal.metric) != 1) {
        for (i in 2:length(returntotal.metric)) {
          meanTotal_i <- lapply(returntotal.metric[[i]], function(x) x[,"SCL"])
          meanTotal_i <- lapply(lmax[,1], function(x) c(meanTotal_i[[x]], rep(0, lmax[x, 2]-length(meanTotal_i[[x]]))))
          
          meanTotal2_i <- lapply(returntotal.metric[[i]], function(x) x[,"SCL"]^2)
          meanTotal2_i <- lapply(lmax[,1], function(x) c(meanTotal2_i[[x]], rep(0, lmax[x, 2]-length(meanTotal2_i[[x]]))))

          meanRTotal_i <- lapply(returntotal.metric[[i]], function(x) x[,"R"])
          meanRTotal_i <- lapply(lmax[,1], function(x) c(meanRTotal_i[[x]], rep(0, lmax[x, 2]-length(meanRTotal_i[[x]]))))
          
          meanRTotal2_i <- lapply(returntotal.metric[[i]], function(x) x[,"R"]^2)
          meanRTotal2_i <- lapply(lmax[,1], function(x) c(meanRTotal2_i[[x]], rep(0, lmax[x, 2]-length(meanRTotal2_i[[x]]))))

          att_i <- lapply(returntotal.metric[[i]], function(x) c(
            attributes(x)[["time.begin.tsp"]], 
            attributes(x)[["time.end.tsp"]], 
            attributes(x)[["time.begin.middlethird"]],
            attributes(x)[["time.end.middlethird"]],
            attributes(x)[["metric.begin.tsp"]],
            attributes(x)[["metric.end.tsp"]], 
            attributes(x)[["test.mean"]], 
            attributes(x)[["test.sd"]], 
            attributes(x)[["metric.end.incubation"]]
          ))
          att2_i <- lapply(returntotal.metric[[i]], function(x) c(
            attributes(x)[["time.begin.tsp"]]^2, 
            attributes(x)[["time.end.tsp"]]^2, 
            attributes(x)[["time.begin.middlethird"]]^2,
            attributes(x)[["time.end.middlethird"]]^2,
            attributes(x)[["metric.begin.tsp"]]^2,
            attributes(x)[["metric.end.tsp"]]^2, 
            attributes(x)[["test.mean"]]^2, 
            attributes(x)[["test.sd"]]^2, 
            attributes(x)[["metric.end.incubation"]]^2
          ))
          att_sat_i <- lapply(returntotal.metric[[i]], function(x) c(
            attributes(x)[["stopattest"]]
          ))   
          
          for(j in 1:length(meanTotal_i)) {
            meanTotal[[j]] <- meanTotal[[j]]+meanTotal_i[[j]]
            meanTotal2[[j]] <- meanTotal2[[j]]+meanTotal2_i[[j]]
            meanRTotal[[j]] <- meanRTotal[[j]]+meanRTotal_i[[j]]
            meanRTotal2[[j]] <- meanRTotal2[[j]]+meanRTotal2_i[[j]]
            
            nbdif1 <- lapply(meanTotal_i, function(x) ifelse(x != 0, 1, 0))
            
            nbdif0 <- lapply(seq_along(nbdif0), function(x) nbdif0[[x]]+nbdif1[[x]])
            
            att[[j]] <- att[[j]]+att_i[[j]]
            att2[[j]] <- att2[[j]]+att2_i[[j]]
            att_sat[[j]] <- att_sat[[j]] | att_sat_i[[j]]
            
          }
        }
      }
      
      
      ret <- returntotal.metric[[1]]
      
      # sum(a^2)-length(a)*mean(a)^2)/(length(a)-1)
      for(j in 1:length(meanTotal)) {
        
        se_i <- ifelse(nbdif0[[j]]>1, 
                       (meanTotal2[[j]]-nbdif0[[j]]*(meanTotal[[j]]/nbdif0[[j]])^2)/(nbdif0[[j]]-1), 
                       NA)
        se_i <- ifelse(is.na(se_i), NA, sqrt(ifelse(se_i<=0, 0,se_i)))
        seR_i <- ifelse(nbdif0[[j]]>1, 
                        (meanRTotal2[[j]]-nbdif0[[j]]*(meanRTotal[[j]]/nbdif0[[j]])^2)/(nbdif0[[j]]-1),
                        NA)
        seR_i <- ifelse(is.na(seR_i), NA, sqrt(ifelse(seR_i<=0, 0,seR_i)))
        
        intret <- ret[[j]]
        if (lmax[j, 2]!=dim(intret)[1]) {
        missinglines <- rep(NA, lmax[j, 2]-dim(intret)[1])
        intret <- rbind(intret, data.frame(Time=missinglines, TempC=missinglines, TempK=missinglines,
                   R=missinglines, SCL=missinglines, IndiceK=missinglines, 
                   DeltaT=missinglines))
        }
        
        
        ret[[j]] <- cbind(intret, mean.SCL=meanTotal[[j]]/nbdif0[[j]], 
                          mean.R=meanRTotal[[j]]/nbdif0[[j]], 
                          se.SCL=se_i, se.R=seR_i)
        
        # je fais les attributs
        mean_att_i <- att[[j]]/replicate.CI
        se_att_i <- (att2[[j]]-replicate.CI*mean_att_i^2)/(replicate.CI-1)
        se_att_i <- sqrt(ifelse(se_att_i<=0, 0, se_att_i))
        attributes(ret[[j]])$time.begin.tsp <- mean_att_i[1]
        attributes(ret[[j]])$time.end.tsp <- mean_att_i[2]
        attributes(ret[[j]])$time.begin.middlethird <- mean_att_i[3]
        attributes(ret[[j]])$time.end.middlethird <- mean_att_i[4]
        attributes(ret[[j]])$metric.begin.tsp <- mean_att_i[5]
        attributes(ret[[j]])$metric.end.tsp <- mean_att_i[6]
        attributes(ret[[j]])$test.mean <- mean_att_i[7]
        attributes(ret[[j]])$test.sd <- mean_att_i[8]
        attributes(ret[[j]])$metric.end.incubation <- mean_att_i[9]
        attributes(ret[[j]])$time.begin.tsp.se <- se_att_i[1]
        attributes(ret[[j]])$time.end.tsp.se <- se_att_i[2]
        attributes(ret[[j]])$time.begin.middlethird.se <- se_att_i[3]
        attributes(ret[[j]])$time.end.middlethird.se <- se_att_i[4]
        attributes(ret[[j]])$stopattest <- att_sat[[j]]
        
        
      }
      ret.metric <- ret
    }
    
    
    if (any(out=="summary")) {
      
      
      if (replicate.CI != 1) {
        meanTotal2 <- returntotal.summary[[1]]^2
        meanTotal <- returntotal.summary[[1]]
        
        cpt <- 1
        
        for (i in 2:length(returntotal.summary)) {
          if (any(!is.na(returntotal.summary[[i]]))) {
          meanTotal <- meanTotal+returntotal.summary[[i]]
          meanTotal2 <- meanTotal2+returntotal.summary[[i]]^2
          cpt <- cpt+1
          }
        }
        
        meanTotal <- meanTotal/cpt
        df_se <- (meanTotal2-cpt*meanTotal^2)/(cpt-1)
        df_se <- sqrt(ifelse(df_se<1E-9, 0,df_se))
        
        
        colnames(df_se) <- paste0(gsub("(.+)\\.mean", "\\1", colnames(df_se)), ".se")
        ret.summary <- cbind(meanTotal, df_se)
      } else {
        
        df_se <- as.matrix(returntotal.summary[[1]])
        df_se[, ] <- NA
        df_se <- as.data.frame(df_se)
        colnames(df_se) <- paste0(gsub("(.+)\\.mean", "\\1", colnames(df_se)), ".se")
        
        ret.summary <- cbind(returntotal.summary[[1]], df_se)
      }
    }
    return(list(metric=ret.metric, summary=ret.summary))
  }
