#' MovingIncubation simulate incubation of a nest with the beginning varying day by day
#' @title Simulate incubation of a nest with the beginning of incubation varying
#' @author Marc Girondot
#' @return A dataframe with informations about thermosensitive period length and incubation length day by day of incubation
#' @param result A result file generated by searchR
#' @param times Timeseries of times
#' @param temperatures Timeseries of temperatures
#' @param temperatures.end.incubation Timeseries of temperatures at the end of incubation
#' @param average.incubation.duration The average time to complete incubation
#' @param skip Number of data to skip between two runs
#' @param min.length Minimum length of possible incubation
#' @param max.length Maximum length of possible incubation
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if not present in result is not provided (not used still)
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings as a vector ie test=c(Mean=xx, SD=yy)
#' @param M0 Measure of hatchling size proxi at laying date
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided to estimate TSP length
#' @param replicate.CI Number of randomizations to estimate CI (not used still)
#' @param progress If TRUE, progression bar is shown (desactivate for sweave or knitr)
#' @description Simulate incubation of a nest with the beginning varying day by day\cr
#' Temperatures must be in a data.frame with one column (Time) being the time and the second the temperatures (Temperatures). Do not use FormatNests() for this dataframe.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' ti <- seq(from=0, to=(60*24*100), by=60)
#' temp <- rnorm(length(ti), 29, 5)
#' out <- MovingIncubation(result=resultNest_4p, temperatures=temp,
#' times=ti)
#' }
#' @export


MovingIncubation <-
function(result=stop("A result file from searchR() must be provided"), 
	temperatures=stop("A timeseries of temperature must be provided"),
	times=stop("A timeseries of time for each temperature must be provided"),
	temperatures.end.incubation=temperatures,
	average.incubation.duration=60*1440,
	skip = 1, parameters=NULL, fixed.parameters=NULL, SE=NULL, 
	derivate=NULL, test=NULL, M0=NULL, TSP.borders=c(21, 26), 
			embryo.stages="Caretta caretta", 
				replicate.CI=1, min.length=40*1440, max.length=150*1440, progress=TRUE) {

# skip = 1; parameters=NULL; fixed.parameters=NULL; SE=NULL; derivate=NULL; test=NULL; M0=NULL
# TSP.borders=c(21, 26); embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33)
# replicate.CI=100; min.length=40*1440; max.length=150*1440; progress=TRUE;average.incubation.duration=60*1440
# temperatures.end.incubation=temperatures

# maintenant il n'est plus possible qu'il n'y ait pas de result
# ni de temperatures
#  

if (embryo.stages[1]=="Caretta caretta") 
	embryo.stages <- list(number=21:30, 
			size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33)
	
if (is.null(test)) {
# si tous sont pareils, je reprends les memes
# Correction d'un bug, rajout de [[1]] dans result$test["Mean"][[1]][1] 30/7/2012
	if (all(result$test["Mean"]==result$test["Mean"][[1]][1]) & all(result$test["SD"]==result$test["SD"][[1]][1])) {
		test <- c(Mean=result$test["Mean"][[1]][1], SD=result$test["SD"][[1]][1])
	} else {	
		print("Error: the size at hatching must be provided.")
		return()
	}
}

if ((length(times)!=length(temperatures)) | (length(temperatures.end.incubation)!=length(temperatures))) {
		print("Error: length of times and temperatures must be the same")
		return()
	}


if (is.null(derivate)) {
	derivate <- result$derivate
}

if (is.null(M0)) {
	M0 <- result$M0
}

if (is.null(fixed.parameters)) {
	fixed.parameters <- result$fixed.parameters
}

if (is.null(parameters)) {
  parameters <- result$par
}
	
parssm <- c(parameters, fixed.parameters)

# je peux indiquer des SE en plus de ceux de result
if (is.null(SE)) {
	res <- result$SE
} else {
	res <- SE
}

if(all(is.na(res))) replicate.CI <- 1

mean.ts <- test["Mean"]
sd.ts <- test["SD"]

stages <- data.frame(stade=embryo.stages$number, metric=(embryo.stages$size)*mean.ts)

begin.TSP <- stages[stages[,"stade"]==TSP.borders[1], "metric"]
end.TSP <- stages[stages[,"stade"]==TSP.borders[2], "metric"]

incubationL <- NULL
incubationE <- NULL
incubationTSP  <- NULL
incubationTIME <- NULL
incubationTEMPERATURE <- NULL
SDincubationL <- NULL
SDincubationE <- NULL
SDincubationTSP  <- NULL
SDincubationTIME <- NULL
SDincubationTEMPERATURE <- NULL

# on suppose que les temperatures sont dans un data.frame avec une colonne Time et une colonne Temperatures
# 26/9/2013: Plus simple, je cree un vecteur de temps
# et donc j'ai les temperatures dans un vecteur

anK <- NULL
if (!is.na(parssm["rK"])) {
	anK <- parssm[["rK"]]*mean.ts
}
if (!is.na(parssm["K"])) {
	anK <- parssm[["K"]]
}

nbtp <- length(temperatures)

if (progress) pb<-txtProgressBar(min=0, max=(nbtp-2)*replicate.CI, style=3)

for (temp in seq(from=1, to=nbtp-2, by=skip)) {

rep.incubationL <- NULL
rep.incubationE <- NULL
rep.incubationTSP <- NULL
rep.incubationTIME <- NULL
rep.incubationTEMPERATURE <- NULL

for (rep in 1:replicate.CI) {

  if (progress) 	setTxtProgressBar(pb, (temp-1)*(replicate.CI)+rep)

  pencours <- NULL
  for(p in 1:length(parameters)) {
  	nm <- names(parameters)[p]
  	if (is.na(res[nm])) {
  		pencours <- c(pencours, parameters[p])
  	} else {
  		pencours <- c(pencours, rnorm(1, parameters[p], res[nm]))
  	}
  }
  names(pencours) <- names(parameters)
  
  parssm <- c(pencours, fixed.parameters)


	dt <- floor(as.numeric(times[temp:nbtp]-times[temp]))
	dt <- dt[dt<max.length]
	
	if (max(dt)>=min.length) {
	
	tempnid <- temperatures[temp:(temp+length(dt)-1)]
	tempnid2 <- temperatures.end.incubation[temp:(temp+length(dt)-1)]
	# et dans dt j ai le temps
	tempencours <- tempnid + (tempnid2-tempnid)*(dt/average.incubation.duration)
	templevels <- as.factor(tempencours)
	valeur.r.list <- .SSM(as.numeric(levels(templevels))+273.15, parssm)
	v <- data.frame(temp=levels(templevels), valeur.r=valeur.r.list[[1]], 
	                valeur.r_L=valeur.r.list[[2]])
  
	df <- data.frame(Time=dt, temp=tempencours)

	formated <- FormatNests(df)
	nids <- formated[[1]]
	meanduree <- NA
	y <- M0
	serie.taille <- M0
	k <- dim(nids)[1]-1
  
	transition_P <- parssm["transition_P"][[1]]
	transition_S <- parssm["transition_S"][[1]]
	
  begin.TSP.ec <- NA
  end.TSP.ec <- NA

  for (i in 1:k) {
#    print(i)
    if (is.na(transition_S) | is.na(transition_P)) {
      transition <- 1
    } else {
      transition <- 1/(1+exp(transition_S*(y-transition_P)))
    }
		timesunique <- c(nids[i,"Time"], nids[i+1,"Time"])
		a <- v$valeur.r[v$temp==nids[i,2]]*transition+v$valeur.r_L[v$temp==nids[i,2]]*(1-transition)
#		parms <- c(alpha=as.numeric(valeur.r[as.numeric(names(valeur.r))==nids[i,2]]), K=as.numeric(anK))
		parms <- c(alpha=a, K=as.numeric(anK))
		out1 <- lsoda(y, timesunique, derivate, parms)
		y <- as.numeric(out1[2,2])
    serie.taille <- c(serie.taille, y)
    if (is.na(begin.TSP.ec) & (y>=begin.TSP)) {
      begin.TSP.ec <- nids[i+1,"Time"]
      indice.x1 <- i+1
    }
    if (is.na(end.TSP.ec) & (y>=end.TSP)) {
      end.TSP.ec <- nids[i+1,"Time"]
      indice.x2 <- i
#      indice.x2 <<- indice.x2
#      indice.x1 <<- indice.x1
#      serie.taille <<- serie.taille
#      nids <<- nids
      weighted.mean.temp.TSP <- sum(diff(serie.taille[indice.x1:indice.x2])*
                                nids[indice.x1:(indice.x2-1), "Temperatures C"])/
                                sum(diff(serie.taille[indice.x1:indice.x2]))
    }
		if (y>=	mean.ts) {
			meanduree <- nids[i+1,"Time"]
			break
		}
	}
# fin du test pour savoir si l'incubation est suffisamment longue
	} else {
	  meanduree <- NA
    y <- NA
	  weighted.mean.temp.TSP <- NA
	}
	
	
	rep.incubationL <- c(rep.incubationL, meanduree)
  rep.incubationE <- c(rep.incubationE, y)
	rep.incubationTEMPERATURE <- c(rep.incubationTEMPERATURE, weighted.mean.temp.TSP)
  if (is.na(meanduree)) {
  		rep.incubationTSP <- c(rep.incubationTSP, NA)
  	} else {
  		rep.incubationTSP <- c(rep.incubationTSP, end.TSP.ec-begin.TSP.ec)
  	}
  rep.incubationTIME <- c(rep.incubationTIME, times[temp])

	
# fin des repetitions pour CI
}

#	incubationL1 <<- c(incubationL, mean(rep.incubationL))
#	incubationE1 <<- c(incubationE, mean(rep.incubationE))
#	incubationTSP1 <<- c(incubationTSP, mean(rep.incubationTSP, na.rm = TRUE))
#	incubationTIME1 <<- c(incubationTIME, mean(rep.incubationTIME))
#	incubationTEMPERATURE1 <<- c(incubationTEMPERATURE, mean(rep.incubationTEMPERATURE))
#	SDincubationL1 <<- c(SDincubationL, sd(rep.incubationL))
#	SDincubationE1 <<- c(SDincubationE, sd(rep.incubationE))
#	SDincubationTSP1 <<- c(SDincubationTSP, sd(rep.incubationTSP, na.rm = TRUE))
#	SDincubationTIME1 <<- c(SDincubationTIME, sd(rep.incubationTIME))
#	SDincubationTEMPERATURE1 <<- c(SDincubationTEMPERATURE, sd(rep.incubationTEMPERATURE))

incubationL <- c(incubationL, mean(rep.incubationL, na.rm = TRUE))
incubationE <- c(incubationE, mean(rep.incubationE, na.rm = TRUE))
incubationTSP <- c(incubationTSP, mean(rep.incubationTSP, na.rm = TRUE))
incubationTIME <- c(incubationTIME, mean(rep.incubationTIME, na.rm = TRUE))
incubationTEMPERATURE <- c(incubationTEMPERATURE, mean(rep.incubationTEMPERATURE, na.rm = TRUE))
SDincubationL <- c(SDincubationL, sd(rep.incubationL, na.rm = TRUE))
SDincubationE <- c(SDincubationE, sd(rep.incubationE, na.rm = TRUE))
SDincubationTSP <- c(SDincubationTSP, sd(rep.incubationTSP, na.rm = TRUE))
SDincubationTIME <- c(SDincubationTIME, sd(rep.incubationTIME, na.rm = TRUE))
SDincubationTEMPERATURE <- c(SDincubationTEMPERATURE, sd(rep.incubationTEMPERATURE, na.rm = TRUE))

}



return(data.frame(time=incubationTIME, sd.time=SDincubationTIME, 
	incubation.length=incubationL, sd.incubation.length=SDincubationL, 
	end.size=incubationE, sd.end.size=SDincubationE, 
	TSP.length=incubationTSP, sd.TSP.length=SDincubationTSP,
  weighted.incubation.Temperature=incubationTEMPERATURE, 
  sd.weighted.incubation.Temperature=SDincubationTEMPERATURE))

}
