% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Class.Constructors.R
\name{make.design}
\alias{make.design}
\title{Creates a Survey.Design object}
\usage{
make.design(
  region = make.region(),
  transect.type = "line",
  design = "systematic",
  samplers = numeric(0),
  line.length = numeric(0),
  seg.length = numeric(0),
  effort.allocation = numeric(0),
  design.angle = 0,
  spacing = numeric(0),
  edge.protocol = "minus",
  seg.threshold = numeric(0),
  bounding.shape = "rectangle",
  truncation = 1,
  coverage.grid = NULL
)
}
\arguments{
\item{region}{an object of class Region defining the survey region.}

\item{transect.type}{character variable specifying either "line" or "point"}

\item{design}{a character variable describing the type of design. Either "random",
"systematic", "eszigzag" (equal-spaced zigzag), "eszigzagcom" (equal spaced zigzag
with complementary lines) or "segmentedgrid". See details for more information.}

\item{samplers}{the number of samplers you wish the design to generate
(note that the number actually generated may differ slightly due to the
shape of the study region for some designs). This may be one value or a value
for each stratum.}

\item{line.length}{the total line length you desire or a vector of line lengths
the same length as the number of strata.}

\item{seg.length}{the length of the line transect segments for a segmented grid
design.}

\item{effort.allocation}{Used for multi-strata regions where only a total effort
value is provided. This numeric argument should have one value per stratum indicating
the proportion of the total effort to allocate to that stratum. If length is
0 (the default) and only a total line length or total number of samplers is supplied,
effort is allocated based on stratum area.}

\item{design.angle}{numeric value detailing the angle of the design. Can provide
multiple values relating to strata. The use of the angle varies with design, it
can be either the angle of the grid of points, the angle of lines or the design
axis for the zigzag design. See details. In addition, a value of -1 will cause a
random design angle to be generated.}

\item{spacing}{used by systematic designs, numeric value(s) to define spacing
between transects. Can be a vector of values with one value per stratum.}

\item{edge.protocol}{character value indicating whether a "plus" sampling or
"minus" sampling protocol is used. See details.}

\item{seg.threshold}{this is a percentage threshold value applicable to segmented
grid designs controlling which partial segments are discarded around the survey
region boundary. By default, the value of 50, means that only segments that are
more than half inside the survey region will be retained. To retain all segments,
no matter how small they are when clipped to the survey region boundary set this
value to 0.}

\item{bounding.shape}{only applicable to zigzag designs. A character value saying
whether the zigzag transects should be generated using a minimum bounding
"rectangle" or "convex.hull". The default is a minimum bounding rectangle.}

\item{truncation}{A single numeric value describing the longest distance at which
an object may be observed. Truncation distance is constant across strata.}

\item{coverage.grid}{An object of class Coverage.Grid for use when
running the coverage simulation.}
}
\value{
object of a class which inherits from class Survey.Design either
Line.Transect.Design or Point.Transect.Design
}
\description{
Creates a description of a survey design. Designs may use different
types of either point or line transect designs across strata but cannot mix
point and line transect design types within a single design object.
}
\details{
\strong{Plus versus Minus Sampling}
If you choose for your design to use a minus sampling strategy then transects will
only be generated within the survey region and will give lower coverage around the
edge of the survey region. Plus sampling generates transects within an area
greater than the study region. To do this \pkg{dssd} first puts a buffer around the
study region before generating the transects within the buffered region. The width
of the buffer is the truncation distance supplies by the user. Plus sampling
helps to ensure more even coverage around the edge of the study area. See
\emph{Buckland et. al, 2001} "Introduction to Distance Sampling" for information
on when to use plus versus minus sampling.

\strong{Point Transect Designs}
For point transect designs the user may either specify "random" or
"systematic" for the design argument. If the user specifies "random", they
should also provide a value for effort detailing the number of point transects
they wish their survey to have. For stratified designs they may specify a vector
of numbers detailing the number of transects per strata or alternatively use the
effort.allocation argument to allocate a total effort amount proportionally. If
effort.allocation is left blank then effort will be allocated according to strata
area. If the user specified"systematic" they may either provide their desired number
of samplers or a value for spacing which defines the gap between each of the
points (again a vector of spacing values can be provided for each strata).
Optionally the user may select a design.angle. For both random and systematic
point transect designs the user may select either a minus or plus sampling edge
protocol.

\strong{Line Transect Designs:}
For line transect designs the user may either specify "random" (randomly
placed full width lines), "systematic" (systematically placed full width lines),
"eszigzag" (equally spaced zigzag lines), "eszigzagcom" (two sets of complementary
equally spaced zigzag lines) or "segmentedgrid" (a grid of short line transect
segments). If the user specifies "random", they should provide the either the
number of samplers they wish the design to generate or the line length they wish
to achieve, either by strata or as a total. If the user specifies "systematic"
they should specify either the number of samplers, the desired line length or
the spacing between lines. The design angle for these parallel line designs
refers to the angle of the lines where 0 is a vertical line and moving round
in a clockwise direction. If the user specifies a zigzag design they should
specify the systematic spacing value, number of samplers or line length to be
used and should choose between generating the design in a minimum bounding
rectangle or a convex hull. The default is minimum bounding rectangle which gives
more even coverage but the convex hull is generally more efficient. A segmented
grid design may be generated using the either the number of samplers or total
line length, combined with a value for segment length. Alternatively the user
may specify a values for spacing and segment length. The segmented grid design
also uses the segment threshold argument. All the designs may be generated
using plus or minus sampling protocols. Similar to the point transect designs
different values may be specified for each strata for all of the above options.
The design angle for the zigzag designs refers to the angle of a line which
would run through the middle of each zigzag transect if the zigzags were to
be generated within a rectangle. The design angle for zigzags should usually
run along the longest dimension of the study region.

NOTE: If multiple global design effort arguments are supplied (i.e. spacing,
samplers, line.length) then only the first of spacing then line.length then
number of samplers will be used. The other values provided will be discarded.
Different design effort arguments may supplied for different strata. This is
achieved by supplying vectors of numeric values for each of the desired
effort measures, there should be 1 value for each stratum. A value indicates
the effort for that stratum and NA's should be used to ensure that only one
measure of effort is defined for each stratum.

\strong{Effort Allocation:}
For multi-strata designs users are able to define a single global effort value,
for example number of samplers or line length, and allocate proportions of it
to each stratum using the effort.allocation argument. If a global effort value
is supplied and effort.allocation is not defined then effort is assigned
based on stratum area. This should lead to a design which is at least
approximately equal effort across strata. In the case where all strata use the
same systematic design then in the absence of effort.allocation the spacing will
be calculated globally and exactly equal effort will be achieved. In the case
where different designs are chosen for different strata or a non-systematic
design is selected then effort and spacing values will be calculated at the
stratum level and this can lead to some variations in coverage between strata.

See the Getting Started Vignette and the Multiple Strata in dssd Vignette for
example designs.
}
\examples{
#Point transect example
shapefile.name <- system.file("extdata", "TrackExample.shp", package = "dssd")
region <- make.region(region.name = "study area",
                     shape = shapefile.name)

\donttest{
# Generate coverage grid (spacing quite sparse for speed)
cover <- make.coverage(region,
                       n.grid.points = 250)

# Define design
design <- make.design(region = region,
                      transect.type = "point",
                      design = "random",
                      samplers = 25,
                      design.angle = 45,
                      edge.protocol = "minus",
                      truncation = 3,
                      coverage.grid = cover)

# Generate a single survey instance
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)

# Note, the number of reps here has been set to 5 to avoid lengthy run-times,
# however, the reps should be at least 100 for an idea of design statistics
# (i.e. trackline lengths) and 500 + to give a good odea of coverage.
design <- run.coverage(design, reps = 5)
# Plot the coverage
plot(design)
# Display the design statistics
design
#Extract coverage scores
coverage.scores <- get.coverage(design)
hist(coverage.scores)

#Multi-strata line transect example
shapefile.name <- system.file("extdata", "AreaRProjStrata.shp", package = "dssd")
region <- make.region(region.name = "study area",
                     strata.name = c("North", "NW", "West Upper",
                                     "West Lower", "SW", "South"),
                     shape = shapefile.name)
plot(region)
# Make a coverage grid
cover <- make.coverage(region,
                       n.grid.points = 500)
# Define the design
design <- make.design(region = region,
                      transect.type = "line",
                      design = c("systematic", "systematic",
                                 "eszigzag", "systematic",
                                 "systematic", "eszigzagcom"),
                      line.length = 5000*1000, #5000km x 1000m (projection in m)
                      design.angle = c(160, 135, 170, 135, 50, 60),
                      edge.protocol = "minus",
                      truncation = 3000,
                      coverage.grid = cover)

# Create a single set of transects to check
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)

# Note, the number of reps here has been set to 5 to avoid lengthy run-times,
# however, the reps should be at least 100 for an idea of design statistics
# (i.e. trackline lengths) and 500 + to give a good odea of coverage.
design <- run.coverage(design, reps = 5)
# Plot the coverage
plot(design)
# Display the design statistics
design
# Extract coverage scores for the first strata
coverage.scores <- get.coverage(design, strata.id = 1)
summary(coverage.scores)
}

# Fast running example for CRAN testing purposes
# This spacing is too sparse to assess coverage in a real example and
# the number of repetitions is too low to assess design statistics
cover <- make.coverage(region,
                       n.grid.points = 50)
design <- make.design(region = region,
                      transect.type = "point",
                      design = "random",
                      samplers = 25,
                      design.angle = 45,
                      edge.protocol = "minus",
                      truncation = 3,
                      coverage.grid = cover)
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)
design <- run.coverage(design, reps = 3)
plot(design)
design

}
\author{
Laura Marshall
}
