#' Composed error multivariate distribution
#'
#' Probablitiy density function, distribution and random number generation for the composed error multivariate distribution.
#'
#' @aliases dcomperr_mv, pcomperr_mv, rcomperr_mv
#'
#' @param x1 vector of quantiles for margin 1.
#' @param mu1 vector of \eqn{\mu} for margin 1
#' @param sigma_v1 vector of \eqn{\sigma_V} for margin 1. Must be positive.
#' @param par_u1 vector of \eqn{\sigma_U} for margin 1. Must be positive.
#' @param s1 \eqn{s=-1} for production and \eqn{s=1} for cost function for margin 1.
#' @param dist1 specifying the distribution of margin 1.
#' @param x2 vector of quantiles for margin 2.
#' @param mu2 vector of \eqn{\mu} for margin 2
#' @param sigma_v2 vector of \eqn{\sigma_V} for margin 2. Must be positive.
#' @param par_u2 vector of \eqn{\sigma_U} for margin 2. Must be positive.
#' @param dist2 specifying the distribution of margin 1.
#' @param s2 \eqn{s=-1} for production and \eqn{s=1} for cost function for margin 2.
#' @param Tau matrix of Kendall's tau.
#' @param family integer, defines the bivariate copula family:\cr
#' `1` = Gaussian copula \cr
#' `3` = Clayton copula \cr
#' `4` = Gumbel copula \cr
#' @param deriv derivative of order \code{deriv} of the log density. Available are 1,2,3,4.
#' @param xg optional, index arrays for upper triangular matrices, generated by \code{trind.generator(K)} and supplied to \code{chainrule}.
#' @param log.p logical; if TRUE, probabilities p are given as log(p).
#'
#' @return \code{dcomperr_mv} gives the density and \code{rcomperr_mv} generates random numbers, with given parameters. If the derivatives are calculated these are provided as the attributes \code{gradient}, \code{hessian}, \code{l3} and \code{l4} of the output of the density.
#'
#' @details A bivariate random vector \eqn{(Y_1,Y_2)} follows a composed error multivariate distribution \eqn{f_{Y_1,Y_2}(y_1,y_2)}, which can be rewritten using Sklars' theorem via a copula
#' \deqn{f_{Y_1,Y_2}(y_1,y_2)=c(F_{Y_1}(y_1),F_{Y_2}(y_2),\tau) \cdot f_{Y_1}(y_1) f_{Y_2}(y_2) \qquad,}
#' where \eqn{c(\cdot)} is a copula function and \eqn{F_{Y_m}(y_m)},\eqn{f_{Y_m}(y_m)} are the marginal cdf and pdf respectively. \eqn{Tau} is Kendall's Tau.
#'
#' @examples
#' pdf<-dcomperr_mv(x1=5, mu1=2, sigma_v1=3, par_u1=4, s1=-1, dist1="normhnorm",
#'                  x2=-5, mu2=2, sigma_v2=3, par_u2=4, s2=-1, dist2="normhnorm",
#'             Tau=0.5, family=1, deriv = 2, xg=NULL, log.p=FALSE)
#' cdf<-pcomperr_mv(q1=0, mu1=0, sigma_v1=1, par_u1=1, s1=-1, dist1="normhnorm",
#'             q2=0, mu2=0, sigma_v2=1, par_u2=1, s2=-1, dist2="normhnorm",
#'             Tau=0.5, family=1, deriv = 0, xg=NULL, log.p=FALSE)
#' r<-rcomperr_mv(n=100, mu1=0, sigma_v1=1, par_u1=1, s1=-1, dist1="normhnorm",
#'           mu2=0, sigma_v2=1, par_u2=1, s2=-1, dist2="normhnorm",
#'           Tau=matrix(0.5,nrow=100), family=1)
#'
#' @references
#' \itemize{
#' \item \insertRef{aigner1977formulation}{dsfa}
#' }
#'
#' @export
#dcomperr_mv
dcomperr_mv<-function(x1=0, mu1=0, sigma_v1=1, par_u1=1, s1=-1, dist1="normhnorm",
                      x2=0, mu2=0, sigma_v2=1, par_u2=1, s2=-1, dist2="normhnorm",
                      Tau=0, family=1, deriv = 0, xg=NULL, log.p=FALSE){

  #Initialize values
  fy1<-NULL
  fy2<-NULL

  #Margin 1
  fy1<-dcomperr(x=x1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, dist=dist1, deriv = deriv, log.p = TRUE)
  Fy1<-pcomperr(q=x1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, dist=dist1, deriv = deriv, log.p = FALSE)

  #Margin 2
  fy2<-dcomperr(x=x2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, dist=dist2, deriv = deriv, log.p = TRUE)
  Fy2<-pcomperr(q=x2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, dist=dist2, deriv = deriv, log.p = FALSE)

  #Evaluate copula at probability integral transformed observations
  value<-dcop(U=cbind(Fy1,Fy2), Tau=Tau, family=family, deriv = deriv, disjoint = FALSE, log.p = log.p)

  #Calculate log multivariate density
  out<-sum(value, fy1, fy2)
  names(out)<-NULL

  if(deriv>0){
    #Get number of parameters
    npar_fy1<-ncol(attr(fy1,"gradient"))
    npar_fy2<-ncol(attr(fy2,"gradient"))
    npar_cop<-ncol(attr(value,"gradient"))
    npar_comperr_mv<-npar_fy1+npar_fy2+npar_cop-2

    #Check if index arrays for upper triangular is available, else create it
    if(is.null(xg)){
      xg<-mgcv::trind.generator(npar_comperr_mv)
    }

    l1<-cbind(attr(fy1,"gradient"),attr(fy2,"gradient"),attr(value,"gradient")[,-c(1:2)])
    l1[,-c((npar_fy1+npar_fy2+1):npar_comperr_mv)]<-l1[,-c((npar_fy1+npar_fy2+1):npar_comperr_mv)]+
                                                    cbind(attr(value,"gradient")[,1]*attr(Fy1,"gradient"),
                                                          attr(value,"gradient")[,2]*attr(Fy2,"gradient"))

    # xg<-mgcv::trind.generator(7)

    # new_hess<-attr(value,"hessian")[,-c(1,2,4)]
    l2_index<-unique(c(xg$i2[(1:npar_fy1),(1:npar_fy1)],
                       xg$i2[(npar_fy1+1):(npar_fy1+npar_fy2),(npar_fy1+1):(npar_fy1+npar_fy2)],
                       xg$i2[(npar_fy1+npar_fy2+1):npar_comperr_mv,(npar_fy1+npar_fy2+1):npar_comperr_mv]))
    l2<-matrix(0, nrow=length(value), ncol=max(xg$i2))

    xg_cop<-mgcv::trind.generator(npar_cop)

    l2[,l2_index]<-cbind(attr(fy1,"hessian"),attr(fy2,"hessian"),attr(value,"hessian")[,max(xg_cop$i2)])

    cop_index<-matrix(max(xg_cop$i2), nrow=npar_comperr_mv, ncol=npar_comperr_mv)
    cop_index[(1:npar_fy1),(1:npar_fy1)]<-1
    cop_index[(npar_fy1+1):(npar_fy1+npar_fy2),(1:npar_fy1)]<-2
    cop_index[(npar_fy1+npar_fy2+1):npar_comperr_mv,(1:npar_fy1)]<-3
    cop_index[(npar_fy1+1):(npar_fy1+npar_fy2),(npar_fy1+1):(npar_fy1+npar_fy2)]<-4
    cop_index[(npar_fy1+npar_fy2+1):npar_comperr_mv,(npar_fy1+1):(npar_fy1+npar_fy2)]<-5
    cop_index[upper.tri(cop_index,diag=FALSE)]<-0
    cop_index<-cop_index+t(cop_index)-diag(diag(cop_index))
    cop_index<-cop_index[lower.tri(cop_index,diag=TRUE)]

    F_index<-matrix(1:npar_comperr_mv, nrow=npar_comperr_mv, ncol=npar_comperr_mv, byrow=FALSE)
    tF_index<-t(F_index)

    F_index<-F_index[lower.tri(F_index,diag=TRUE)]
    tF_index<-tF_index[lower.tri(tF_index,diag=TRUE)]

    l2<-l2+attr(value,"hessian")[,cop_index]*
            cbind(attr(Fy1,"gradient"),attr(Fy2,"gradient"),0)[,F_index]*
            cbind(attr(Fy1,"gradient"),attr(Fy2,"gradient"),0)[,tF_index]

    cop_index_grad<-c(rep(1,ncol(attr(Fy1,"hessian"))),rep(2,ncol(attr(Fy2,"hessian"))))
    l2[,l2_index[-length(l2_index)]]<-l2[,l2_index[-length(l2_index)]]+attr(value,"gradient")[,cop_index_grad]*cbind(attr(Fy1,"hessian"),attr(Fy2,"hessian"))

    #Set gradient and hessian as attributes
    attr(out,"gradient")<-l1
    attr(out,"hessian")<-l2
  }

  if(!log.p){
    out<-exp(out)
  }

  #Return out
  return(out)
}

#' @describeIn dcomperr_mv distribution function for the composed error multivariate distribution.
#' @param q1 vector of quantiles for margin 1.
#' @param q2 vector of quantiles for margin 2.
#' @export
pcomperr_mv<-function(q1=0, mu1=0, sigma_v1=1, par_u1=1, s1=-1, dist1="normhnorm",
                 q2=0, mu2=0, sigma_v2=1, par_u2=1, s2=-1, dist2="normhnorm",
                 Tau=0, family=1, deriv = 0,  xg=NULL, log.p=FALSE){

  #Margin 1
  Fy1<-pcomperr(q=q1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, dist=dist1, deriv = deriv, xg=xg, log.p = FALSE)

  #Margin 2
  Fy2<-pcomperr(q=q2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, dist=dist2, deriv = deriv, xg=xg, log.p = FALSE)

  #Evaluate cdf of copula at probability integral transformed observations
  out<-pcop(U=cbind(Fy1,Fy2), Tau=Tau, family=family, log.p = log.p)
  names(out)<-NULL

  #Return out
  return(out)
}

#' @describeIn dcomperr_mv random number generation for the composed error multivariate distribution.
#' @param n number of observations.
#' @export
rcomperr_mv<-function(n, mu1=0, sigma_v1=1, par_u1=1, s1=-1, dist1="normhnorm",
                    mu2=0, sigma_v2=1, par_u2=1, s2=-1, dist2="normhnorm",
                    Tau=0, family=1){

  #Generate pseudo observations
  obs<-rcop(n=n, Tau=Tau, family=family)

  #Margin 1
  y1<-qcomperr(p=obs[,1], mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, dist=dist1)

  #Margin 2
  y2<-qcomperr(p=obs[,2], mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, dist=dist2)


  #Combind y1 and y2
  out<-cbind(y1,y2)

  #Return out
  return(out)
}
