#' @title Generate a flat text log file
#' to represent the state of the cache.
#' @description
#' This functionality is like
#' \code{\link{make}(..., cache_log_file = TRUE)},
#' but separated and more customizable.
#' The \code{drake_cache_log_file} function writes a flat text file
#' to represents the state of all the targets and imports in the cache.
#' If you call it after each \code{\link{make}()}
#' and put the log file under version control,
#' you can track the changes to your results over time.
#' This way, your data is versioned alongside your code
#' in a easy-to-view format. Hopefully, this functionality
#' is a step toward better data versioning tools.
#' @seealso \code{\link{drake_cache_log}},
#' \code{\link{make}},
#' \code{\link{get_cache}},
#' \code{\link{default_long_hash_algo}},
#' \code{\link{short_hash}},
#' \code{\link{long_hash}}
#' @export
#' @return There is no return value, but a log file is generated.
#' @param file character scalar, name of the flat text log file.
#' @param cache drake cache. See \code{\link{new_cache}()}.
#' If supplied, \code{path} and \code{search} are ignored.
#' @param path Root directory of the drake project,
#' or if \code{search} is \code{TRUE}, either the
#' project root or a subdirectory of the project.
#' @param search logical. If \code{TRUE}, search parent directories
#' to find the nearest drake cache. Otherwise, look in the
#' current working directory only.
#' @param verbose whether to print console messages
#' @param jobs number of jobs/workers for parallel processing
#' @param targets_only logical, whether to output information
#' only on the targets in your workflow plan data frame.
#' If \code{targets_only} is \code{FALSE}, the output will
#' include the hashes of both targets and imports. 
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' # Load drake's canonical example.
#' load_basic_example() # Get the code with drake_example()
#' # Run the project and save a flat text log file.
#' make(my_plan)
#' drake_cache_log_file() # writes drake_cache.log
#' # The above 2 lines are equivalent to make(my_plan, cache_log_file = TRUE) # nolint
#' # At this point, put drake_cache.log under version control
#' # (e.g. with 'git add drake_cache.log') alongside your code.
#' # Now, every time you run your project, your commit history
#' # of hash_lot.txt is a changelog of the project's results.
#' # It shows which targets and imports changed on every commit.
#' # It is extremely difficult to track your results this way
#' # by putting the raw '.drake/' cache itself under version control.
#' })
#' }
drake_cache_log_file <- function(
  file = "drake_cache.log",
  path = getwd(),
  search = TRUE,
  cache = drake::get_cache(path = path, search = search, verbose = verbose),
  verbose = TRUE,
  jobs = 1,
  targets_only = FALSE
){
  if (!length(file) || identical(file, FALSE)){
    return(invisible())
  } else if (identical(file, TRUE)){
    file <- formals(drake_cache_log_file)$file
  }
  drake_cache_log(
    path = path,
    search = search,
    cache = cache,
    verbose = verbose,
    jobs = jobs,
    targets_only = targets_only
  ) %>% write.table(
    file = file,
    quote = FALSE,
    row.names = FALSE
  )
}

#' @title Get a table that represents the state of the cache.
#' @description
#' This functionality is like
#' \code{\link{make}(..., cache_log_file = TRUE)},
#' but separated and more customizable. Hopefully, this functionality
#' is a step toward better data versioning tools.
#' @details A hash is a fingerprint of an object's value.
#' Together, the hash keys of all your targets and imports
#' represent the state of your project.
#' Use \code{drake_cache_log()} to generate a data frame
#' with the hash keys of all the targets and imports
#' stored in your cache.
#' This function is particularly useful if you are
#' storing your drake project in a version control repository.
#' The cache has a lot of tiny files, so you should not put it
#' under version control. Instead, save the output
#' of \code{drake_cache_log()} as a text file after each \code{\link{make}()},
#' and put the text file under version control.
#' That way, you have a changelog of your project's results.
#' See the examples below for details.
#' Depending on your project's
#' history, the targets may be different than the ones
#' in your workflow plan data frame.
#' Also, the keys depend on the short hash algorithm
#' of your cache (default: \code{\link{default_short_hash_algo}()}).
#' @seealso \code{\link{drake_cache_log_file}}
#' \code{\link{cached}},
#' \code{\link{get_cache}},
#' \code{\link{default_short_hash_algo}},
#' \code{\link{default_long_hash_algo}},
#' \code{\link{short_hash}},
#' \code{\link{long_hash}}
#' @export
#' @return Data frame of the hash keys of the targets and imports
#' in the cache
#' @param cache drake cache. See \code{\link{new_cache}()}.
#' If supplied, \code{path} and \code{search} are ignored.
#' @param path Root directory of the drake project,
#' or if \code{search} is \code{TRUE}, either the
#' project root or a subdirectory of the project.
#' @param search logical. If \code{TRUE}, search parent directories
#' to find the nearest drake cache. Otherwise, look in the
#' current working directory only.
#' @param verbose whether to print console messages
#' @param jobs number of jobs/workers for parallel processing
#' @param targets_only logical, whether to output information
#' only on the targets in your workflow plan data frame.
#' If \code{targets_only} is \code{FALSE}, the output will
#' include the hashes of both targets and imports. 
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' # Load drake's canonical example.
#' load_basic_example() # Get the code with drake_example()
#' # Run the project, build all the targets.
#' make(my_plan)
#' # Get a data frame of all the hash keys.
#' # If you want a changelog, be sure to do this after every make().
#' cache_log <- drake_cache_log()
#' head(cache_log)
#' # Save the hash log as a flat text file.
#' write.table(
#'   x = cache_log,
#'   file = "drake_cache.log",
#'   quote = FALSE,
#'   row.names = FALSE
#' )
#' # At this point, put drake_cache.log under version control
#' # (e.g. with 'git add drake_cache.log') alongside your code.
#' # Now, every time you run your project, your commit history
#' # of hash_lot.txt is a changelog of the project's results.
#' # It shows which targets and imports changed on every commit.
#' # It is extremely difficult to track your results this way
#' # by putting the raw '.drake/' cache itself under version control.
#' })
#' }
drake_cache_log <- function(
  path = getwd(),
  search = TRUE,
  cache = drake::get_cache(path = path, search = search, verbose = verbose),
  verbose = TRUE,
  jobs = 1,
  targets_only = FALSE
){
  if (is.null(cache)){
    return(
      data.frame(
        hash = character(0),
        type = character(0),
        name = character(0)
      )
    )
  }
  out <- lightly_parallelize(
    X = cache$list(),
    FUN = single_cache_log,
    jobs = jobs,
    cache = cache
  ) %>%
    do.call(what = rbind)
  if (targets_only){
    out <- out[out$type == "target", ]
  }
  out
}

single_cache_log <- function(key, cache){
  hash <- cache$get_hash(key = key)
  imported <- get_from_subspace(
    key = key,
    subspace = "imported",
    namespace = "meta",
    cache = cache
  )
  imported <- ifelse(is.na(imported), TRUE, imported)
  type <- ifelse(imported, "import", "target")
  data.frame(hash = hash, type = type, name = key, stringsAsFactors = FALSE)
}
