


#' Start a new project from oplm files
#'
#' Creates a dexter project database and fills it with response data from two oplm files
#'
#'
#' @param dbname filename/path of new dexter project database (will be overwritten if already exists)
#' @param scr_path path to the .scr file
#' @param dat_path path to the .dat file
#' @param booklet_position vector of start and end of booklet position in the dat file, e.g. c(1,4), 
#' all positions are counted from 1
#' @param responses_start start position of responses in the dat file
#' @param response_length length of individual responses, default=1
#' @param person_id optionally, a vector of start and end position of person_id in the dat file.
#' If NULL, person id's will be auto-generated.
#' @param missing_character character used to indicate missing in dat file, usually \code{" "} or \code{"9"}.
#' @param use_discrim if TRUE, the scores for the responses will be multiplied by the
#' discrimination parameters of the items
#' @param format not used, at the moment only the compressed format is supported.
#' @return a handle to the data base.
#' 
#' @details
#' start_new_project_from_oplm builds a complete dexter database from a dat and scr file in
#' the proprietary oplm format. Three custom variables are added to the database: 
#' booklet_on_off, item_local_on_off, item_global_on_off. These are taken from the scr file
#' and can be used in expressions in the various dexter functions.
#' 
#' @examples
#'\dontrun{
#' db = start_new_project_from_oplm('test.db',
#'    'path_to_scr_file', 'path_to_dat_file', 
#'    booklet_position=c(1,3), responses_start=101,
#'    person_id=c(50,62))
#'
#' par = fit_enorm(db, 
#'    (item_global_on_off==1)&(item_local_on_off==1)&(booklet_on_off==1))
#' 
#' abilities = ability(db, par, 
#'    (item_global_on_off==1)&(item_local_on_off==1)&(booklet_on_off==1))
#' 
#' head(abilities)
#'}
start_new_project_from_oplm = function(dbname, scr_path, dat_path, 
                                       booklet_position, responses_start, response_length = 1, 
                                       person_id = NULL, missing_character = '9', use_discrim=FALSE,
                                       format='compressed')
{
  if(format != 'compressed') stop(paste('Only compressed format is supported at this time', 
                                        'use the inexpand tool from oplm to compress your data'))
  if(response_length==1){ splt = NULL 
  } else {splt = paste0(rep(' ',response_length-1),collapse='')}
  
  if(response_length>1 & trimws(as.character(missing_character)) == "")
  {
    stop("a space as missing character only works with reponse length 1")
  }
  if(!is.null(person_id))
  {
    if(length(person_id)!=2) stop('person_id should be a vector of length 2')
    if(person_id[2]<person_id[1]) stop('person_id is invalid, end position is smaller than start position')
  }
  if(length(booklet_position)!=2) stop('booklet_position should be a vector of length 2')
  if(booklet_position[2]<booklet_position[1]) stop('booklet_position is invalid, end position is smaller than start position')
  
  
  scr = readSCR(scr_path)
  if(anyDuplicated(scr$itemLabels))
  {
    warning("itemlabels in screenfile have duplicates, using autogenerated item_id's")
    scr$itemLabels = paste0('item_',1:length(scr$itemLabels))
  }
  scr$itemLabels = trimws(scr$itemLabels)
  
  rules = tibble(item = rep(scr$itemLabels, as.integer(scr$maxScore)+2), 
                 response = unlist(lapply(scr$maxScore, function(n) c(0:n,missing_character))),
                 score = unlist(lapply(scr$maxScore, function(n) c(0:n,0))))
  
  if(use_discrim) rules$score = rules$score * rep(scr$discrim, as.integer(scr$maxScore)+2)
  
  db = start_new_project(rules, dbname, covariates=NULL)
  
  dbTransaction(db,{
    dbExecute(db,'ALTER TABLE dxBooklets ADD COLUMN booklet_on_off INTEGER NOT NULL DEFAULT 1;')
    dbExecute(db,'ALTER TABLE dxBooklet_design ADD COLUMN item_local_on_off INTEGER NOT NULL DEFAULT 1;')
    dbExecute(db,'ALTER TABLE dxItems ADD COLUMN item_global_on_off INTEGER NOT NULL DEFAULT 1;')
    
    dbExecute(db,'UPDATE dxItems SET item_global_on_off=:onoff WHERE item_id=:item;', 
                tibble(onoff=scr$globCal, item=scr$itemLabels))
  
    design = tibble(booklet_id = as.character(unlist(lapply(1:scr$nBook, function(x) rep(x,scr$nitBook[x])))),
                    item_nbr = unlist(scr$itemsBook),
                    item_position = unlist(lapply(scr$itemsBook, function(x) 1:length(x))),
                    onoff = unlist(scr$itemsOn)) %>%
              inner_join(tibble(item_nbr = 1:scr$nit, item_id = scr$itemLabels), by='item_nbr') %>%
              arrange(.data$booklet_id, .data$item_position) 
            
    dbExecute(db,'INSERT INTO dxBooklets(booklet_id, booklet_on_off) VALUES(:b,:onoff);',
                      tibble(b=as.character(1:scr$nBook),onoff=scr$bookOn))
    dbExecute(db,'INSERT INTO dxTestparts(booklet_id, testpart_nbr) VALUES(:b,:tp);',tibble(b=unique(design$booklet_id),tp=1))
    dbExecute(db,'INSERT INTO dxBooklet_design(booklet_id,testpart_nbr, item_id, item_position, item_local_on_off) 
                            VALUES(:booklet_id,1,:item_id,:item_position, :onoff);', design %>% select(-.data$item_nbr))                
    
    vp = 1
    
    con = file(dat_path, "r")
    while ( TRUE ) {
      lines = readLines(con, n = 2000, encoding='ascii')
      if ( length(lines) == 0 ) {
        break
      }

      bkl = as.integer(substr(lines, booklet_position[1], booklet_position[2]))
      
      rsp = strsplit(substring(lines, 
                               responses_start, 
                               responses_start + scr$nitBook[bkl] * response_length - 1),
                     split=splt)
      
      bkl=as.character(bkl)
      
      if(is.null(person_id))
      {
        prs = as.character(vp:(vp+length(lines)-1) )
        vp = vp+length(lines)
      } else 
      {
        prs=trimws(substr(lines,person_id[1],person_id[2]))
      }
      
      data = tibble(booklet_id = rep(bkl, sapply(rsp,length)), 
                    person_id = rep(prs, sapply(rsp,length)),
                    response=unlist(rsp), 
                    item_position = unlist(sapply(rsp, function(n) 1:length(n),simplify=FALSE))) %>%
        inner_join(design, by=c('booklet_id','item_position')) %>%
        select(.data$person_id, .data$booklet_id, .data$item_id, .data$response)
      
      dbExecute(db,'INSERT INTO dxPersons(person_id) VALUES(:person_id);',tibble(person_id=prs) )
      dbExecute(db,'INSERT INTO dxAdministrations(person_id,booklet_id) VALUES(:person_id,:booklet_id);', 
                      tibble(person_id=prs,booklet_id=bkl))

      dbExecute(db,'INSERT INTO dxResponses(booklet_id,person_id,item_id,response) 
                                  VALUES(:booklet_id,:person_id,:item_id,:response);', 
                      data)
    }
    close(con)
 }, on_error = function(e){dbDisconnect(db); stop(e)})
  return(db)
}

readSCR = function (file) 
{
  z = file(file, "rb")
  n = readBin(z, integer(), size = 2, 3)
  nit = n[3]
  itemLabels = sapply(1:nit, function(x) {
    sl = readBin(z, integer(), size = 1, 1)
    rawToChar(readBin(z, raw(), n = 8)[1:sl])
  })
  globCal = readBin(z, integer(), size = 1, nit)
  discrim = readBin(z, integer(), size = 1, nit)
  maxScore = readBin(z, integer(), size = 1, nit)
  parFixed = readBin(z, integer(), size = 1, nit)
  four = readBin(z, integer(), size = 1, 4)
  sl = readBin(z, integer(), size = 1, 1)
  jobname = rawToChar(readBin(z, raw(), n = 12)[1:sl])
  five = readBin(z, integer(), size = 1, 5)
  sl = readBin(z, integer(), size = 1, 1)
  title = rawToChar(readBin(z, raw(), n = 79)[1:sl])
  for (i in 1:20) {
    sl = readBin(z, integer(), size = 1, 1)
    if (sl > 0) 
      someComment = rawToChar(readBin(z, raw(), n = sl))
  }
  sl = readBin(z, integer(), size = 1, 1)
  dataDir = rawToChar(readBin(z, raw(), n = 60)[1:sl])
  sl = readBin(z, integer(), size = 1, 1)
  dataFile = rawToChar(readBin(z, raw(), n = 12)[1:sl])
  expanded = readBin(z, integer(), size = 1, 1)
  expanded = 1 - expanded
  sl = readBin(z, integer(), size = 1, 1)
  fmt = rawToChar(readBin(z, raw(), n = sl))
  nb = readBin(z, integer(), size = 2, 1)
  inUse = readBin(z, integer(), size = 1, nb)
  nMarg = readBin(z, integer(), size = 1, 1)
  nStat = readBin(z, integer(), size = 1, 1)
  margLabels = sapply(1:nMarg, function(x) {
    sl = readBin(z, integer(), size = 1, 1)
  readBin(z, raw(), n = 8)[1:sl]
  })
  statLabels = sapply(1:nStat, function(x) {
    sl = readBin(z, integer(), size = 1, 1)
    readBin(z, raw(), n = 8)[1:sl]
  })
  nitb = readBin(z, integer(), size = 2, nb)
  datum = readBin(z, integer(), size = 2, 5)
  margBook = statBook = rep(NA, nb)
  itemsBook = vector(mode = "list", length = nb)
  itemsOn = vector(mode = "list", length = nb)
  for (i in 1:nb) {
    rubbish = readBin(z, integer(), size = 2, 1)
    margBook[i] = readBin(z, integer(), size = 1, 1)
    statBook[i] = readBin(z, integer(), size = 1, 1)
    itemsBook[[i]] = readBin(z, integer(), size = 2, nitb[i])
    itemsOn[[i]] = readBin(z, integer(), size = 1, nitb[i])
  }
  close(z)
  list(nit = nit, nMarg = nMarg, nStat = nStat, itemLabels = itemLabels, 
       #margLabels = margLabels, statLabels = statLabels, 
       globCal = globCal, 
       discrim = discrim, maxScore = maxScore, parFixed = parFixed, 
       nBook = nb, bookOn = inUse, nitBook = nitb, 
       #margBook = margBook, statBook = statBook, 
       itemsBook = itemsBook, itemsOn = itemsOn, 
       expanded = expanded)
}