package ch.fgcz.proteomics.fbdm;

/**
 * @author Lucas Schmidt
 * @since 2017-11-23
 */
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import ch.fgcz.proteomics.dto.MassSpecMeasure;
import ch.fgcz.proteomics.dto.MassSpectrum;

public class DeisotoperTest {
    @Test
    public void generateIsotopicSets_two_clusters_Interleaved() {
        Configuration config = new Configuration();

        double oneSetStart = 100;
        double secondSetStart = 100.1;

        List<Double> mz = Arrays.asList(oneSetStart, secondSetStart, oneSetStart + config.getIsotopicPeakDistance(),
                secondSetStart + config.getIsotopicPeakDistance(), oneSetStart + config.getIsotopicPeakDistance() * 2,
                secondSetStart + config.getIsotopicPeakDistance() * 2);
        List<Double> intensity = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0, 7.0);
        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity);

        List<IsotopicSet> isotopicSets = Deisotoper.generateIsotopicSets(new PeakList(massSpectrum), config);
        assertEquals(2, isotopicSets.size());
        assertEquals(3, isotopicSets.get(0).getPeaksInSet().size());
        assertEquals(3, isotopicSets.get(1).getPeaksInSet().size());
    }

    @Test
    public void generateIsotopicSets_one_cluster_interrupted() {
        Configuration config = new Configuration();

        double oneSetStart = 100;
        double secondSetStart = 100.1;

        List<Double> mz2 = Arrays.asList(oneSetStart, secondSetStart, oneSetStart + config.getIsotopicPeakDistance(),
                oneSetStart + config.getIsotopicPeakDistance() * 2);
        List<Double> intensity2 = Arrays.asList(4.0, 4.0, 5.0, 6.0);
        MassSpectrum massSpectrum2 = new MassSpectrum(mz2, intensity2);

        List<IsotopicSet> isotopicSets = Deisotoper.generateIsotopicSets(new PeakList(massSpectrum2), config);
        assertEquals(1, isotopicSets.size());
        assertEquals(3, isotopicSets.get(0).getPeaksInSet().size());
    }

    @Test
    public void generateIsotopicSets_single_isotopic_set_with_two_charges_case1() {
        Configuration config = new Configuration();

        double oneSetStart = 100;

        List<Double> mz3 = Arrays.asList(oneSetStart, oneSetStart + config.getIsotopicPeakDistance() / 2.,
                oneSetStart + config.getIsotopicPeakDistance(), oneSetStart + config.getIsotopicPeakDistance() * 2.,
                oneSetStart + config.getIsotopicPeakDistance() * 3.);
        List<Double> intensity3 = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0);
        MassSpectrum massSpectrum3 = new MassSpectrum(mz3, intensity3);

        List<IsotopicSet> isotopicSets = Deisotoper.generateIsotopicSets(new PeakList(massSpectrum3), config);
        assertEquals(1, isotopicSets.size());
        assertEquals(5, isotopicSets.get(0).getPeaksInSet().size());
    }

    @Test
    public void generateIsotopicSets_single_isotopic_set_with_two_charges_case2() {
        Configuration config = new Configuration();

        double oneSetStart = 100;

        List<Double> mz4 = Arrays.asList(oneSetStart, oneSetStart + config.getIsotopicPeakDistance(),
                oneSetStart + 2. * config.getIsotopicPeakDistance(),
                oneSetStart + config.getIsotopicPeakDistance() * 2.5,
                oneSetStart + config.getIsotopicPeakDistance() * 3); // 100.0, 101.0, 102.0, 102.5, 103.0
        List<Double> intensity4 = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0);
        MassSpectrum massSpectrum4 = new MassSpectrum(mz4, intensity4);

        List<IsotopicSet> isotopicSets = Deisotoper.generateIsotopicSets(new PeakList(massSpectrum4), config);

        assertEquals(1, isotopicSets.size());
        assertEquals(5, isotopicSets.get(0).getPeaksInSet().size());
    }

    @Test
    public void generateIsotopicSets_two_non_overlappling_isotopicsets() {
        Configuration config = new Configuration();

        double oneSetStart = 100;
        double secondStart = 1000;

        List<Double> mz4 = Arrays.asList(oneSetStart, oneSetStart + config.getIsotopicPeakDistance(),
                oneSetStart + 2. * config.getIsotopicPeakDistance(), secondStart,
                secondStart + config.getIsotopicPeakDistance(), secondStart + config.getIsotopicPeakDistance() * 2); // 100.0,
                                                                                                                     // 101.0,
                                                                                                                     // 103.0
        List<Double> intensity4 = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0, 7.0);
        MassSpectrum massSpectrum4 = new MassSpectrum(mz4, intensity4);

        List<IsotopicSet> isotopicSets = Deisotoper.generateIsotopicSets(new PeakList(massSpectrum4), config);
        assertEquals(2, isotopicSets.size());
        assertEquals(3, isotopicSets.get(0).getPeaksInSet().size());
        assertEquals(3, isotopicSets.get(1).getPeaksInSet().size());
    }

    @Test
    public void testDeisotopeMSM() {
        String source = "Unit Test Case";

        double[] mz = { 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0 };
        double[] intensity = { 4.0, 4.0, 5.0, 6.0, 6.0, 7.0, 7.0, 7.0, 8.0, 8.0 };
        double peptidMass = 309.22;
        int chargeState = 2;
        int id = 0;

        double[] mz2 = { 2.0, 4.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 18.0, 20.0 };
        double[] intensity2 = { 65.0, 44.0, 23.0, 88.0, 666.0, 451.0, 44.0, 22.0, 111.0, 1000.0 };
        double peptidMass2 = 203.23;
        int chargeState2 = 2;
        int id2 = 1;

        MassSpecMeasure massSpectrometryMeasurementIn = new MassSpecMeasure(source);
        massSpectrometryMeasurementIn.addMS(mz, intensity, peptidMass, chargeState, id);
        massSpectrometryMeasurementIn.addMS(mz2, intensity2, peptidMass2, chargeState2, id2);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper());

        MassSpecMeasure massSpectrometryMeasurementOut = deisotoper.deisotopeMSM(massSpectrometryMeasurementIn, config);

        assertEquals("Source must be equal!", massSpectrometryMeasurementIn.getSource(),
                massSpectrometryMeasurementOut.getSource());

        assertEquals("Size of massSpectrumList must be equal!", massSpectrometryMeasurementIn.getMSlist().size(),
                massSpectrometryMeasurementOut.getMSlist().size());

        assertNotEquals("Size of mZ-Values must be not equal, becauase there should be isotopic sets!",
                massSpectrometryMeasurementIn.getMSlist().get(0).getMz().size(),
                massSpectrometryMeasurementOut.getMSlist().get(0).getMz().size());

        assertEquals("Size of mZ-Values must be equal, because there should not be isotopic sets!",
                massSpectrometryMeasurementIn.getMSlist().get(1).getMz().size(),
                massSpectrometryMeasurementOut.getMSlist().get(1).getMz().size());
    }

    @Test
    public void testDeisotopeMsOne() {
        List<Double> mz = Arrays.asList(1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0);
        List<Double> intensity = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0, 7.0, 7.0, 7.0, 8.0, 8.0);
        double peptidMass = 309.22;
        int chargeState = 2;
        int id = 0;

        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity, peptidMass, chargeState, id);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper(config));

        MassSpectrum massSpectrumOut = deisotoper.deisotopeMS(massSpectrum);

        assertNotEquals("Size of mZ-Values must be not equal, because there should be isotopic sets!",
                massSpectrum.getMz().size(), massSpectrumOut.getMz().size());

        assertEquals(Arrays.asList(1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0), massSpectrumOut.getMz());

        assertEquals(Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0, 7.0, 7.0, 23.0), massSpectrumOut.getIntensity());

        assertEquals(Arrays.asList(1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, -1.0), massSpectrumOut.getIsotope());

        assertEquals(Arrays.asList(1, 1, 1, 1, 1, 1, 1, -1), massSpectrumOut.getCharge());
    }

    @Test
    public void testDeisotopeMsTwo() {
        List<Double> mz = Arrays.asList(101.0, 102.0, 103.0, 104.0, 106.0, 107.0, 107.5, 108.0, 109.0, 115.0);
        List<Double> intensity = Arrays.asList(4.0, 4.0, 5.0, 6.0, 6.0, 7.0, 7.0, 7.0, 8.0, 8.0);
        double peptidMass = 309.22;
        int chargeState = 2;
        int id = 0;

        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity, peptidMass, chargeState, id);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper(config));

        MassSpectrum massSpectrumOut = deisotoper.deisotopeMS(massSpectrum);

        assertNotEquals("Size of mZ-Values must be not equal, because there should be isotopic sets!",
                massSpectrum.getMz().size(), massSpectrumOut.getMz().size());

        assertEquals(Arrays.asList(101.0, 102.0, 106.0, 107.5, 115.0), massSpectrumOut.getMz());

        assertEquals(Arrays.asList(4.0, 15.0, 20.0, 7.0, 8.0), massSpectrumOut.getIntensity());

        assertEquals(Arrays.asList(1.0, -1.0, -1.0, 1.0, -1.0), massSpectrumOut.getIsotope());

        assertEquals(Arrays.asList(1, -1, -1, 2, -1), massSpectrumOut.getCharge());
    }

    @Test
    public void testDeisotopeMsThree() {
        List<Double> mz = Arrays.asList(110.0715637, 112.0871277, 113.0712051, 115.086853, 116.0531616, 116.0708923,
                119.7680359, 120.0810165, 125.0275803, 125.2685242, 127.086731, 128.0706177, 128.0901031, 129.1023102,
                130.0863953, 130.097641, 137.0709991, 140.7889404, 141.0658722, 144.0479584, 147.1128845, 155.0814972,
                156.0848999, 157.108429, 157.1335144, 158.0923615, 161.0743408, 166.061264, 170.0599976, 173.12854,
                175.1190186, 176.1224823, 180.6107483, 183.1127319, 183.1492462, 184.1523438, 185.1649628, 187.1440735,
                189.0690918, 197.1285706, 198.5964813, 198.8642731, 199.1805878, 200.1840668, 201.1234283, 203.1211243,
                207.112854, 207.1242523, 208.5289154, 211.1439819, 212.1030121, 212.1476288, 213.1598816, 215.1389771,
                222.16008, 223.1439972, 223.1551056, 223.7337646, 226.1186676, 227.1752014, 228.1343994, 229.1187897,
                229.1292419, 231.1160889, 235.1077576, 235.1188812, 237.0979614, 240.1704407, 243.1337891, 248.1605225,
                250.1550903, 251.1025543, 251.1500549, 254.124527, 255.1446381, 260.1969299, 267.5011292, 268.1651917,
                269.1242981, 271.1510315, 277.6557922, 278.1495972, 280.6687622, 285.2397156, 286.1217041, 286.1503601,
                286.4925842, 287.1345215, 288.2026672, 294.6662292, 295.6795349, 303.2660217, 303.2738647, 304.1611633,
                305.309906, 307.1439209, 309.6772156, 312.1863098, 314.1870422, 314.5210571, 322.1860657, 324.9866943,
                326.1813354, 326.9628296, 341.0174866, 342.9970398, 344.9760132, 345.5325623, 346.9502869, 348.1657104,
                351.181427, 351.2023621, 352.2224731, 356.2401733, 359.0281067, 359.203125, 359.2399292, 359.26474,
                361.0241394, 361.2442627, 366.1765747, 366.2191772, 367.2078552, 370.1982117, 371.2112732, 375.2346497,
                382.2078857, 383.2266541, 384.2348633, 392.8924255, 395.7193298, 400.2183533, 404.2320557, 409.90271,
                415.578064, 415.9120789, 417.2459106, 418.2479553, 430.7393188, 435.7140503, 443.2564392, 443.7356873,
                444.2277527, 452.7399902, 453.2414551, 458.1020203, 461.2539368, 461.7554321, 462.2570496, 466.7385864,
                469.3226624, 477.2677612, 487.2818909, 490.2615356, 490.306366, 491.2644043, 491.2842407, 497.3199158,
                505.263031, 509.282959, 509.7840881, 517.7960815, 518.2974854, 518.7993164, 519.2999878, 522.7889404,
                523.2798462, 554.3043823, 565.8234253, 566.3268433, 566.8291016, 570.3148804, 574.3380127, 574.8396606,
                575.3407593, 575.8415527, 579.3312378, 586.3196411, 590.3520508, 603.3457642, 604.3508301, 618.3466187,
                622.8652954, 623.366333, 633.3452148, 634.3284912, 651.3571777, 652.3604126, 692.9234619, 703.4362183,
                714.4061279, 730.3641357, 730.4467773, 731.4320679, 732.4341431, 747.3897705, 790.4317627, 832.4804077,
                844.5157471, 856.3361206, 860.4744263, 861.4712524, 862.475769, 878.4868774, 904.4752808, 905.4776611,
                915.4468994, 945.5620117, 973.5585938, 974.5603027, 979.4803467, 992.5720825, 1017.561829);
        List<Double> intensity = Arrays.asList(23369.3359375, 13809.29296875, 8846.6669921875, 9419.87109375,
                8076.8959960938, 9162.0068359375, 1790.6524658203, 11649.1162109375, 1517.7386474609, 1645.7025146484,
                149996.8125, 1986.2150878906, 8371.3203125, 20422.271484375, 13433.8154296875, 2523.6166992188,
                11089.599609375, 1737.6016845703, 2229.7185058594, 3870.6616210938, 28125.26953125, 120994.3359375,
                4110.685546875, 4039.8679199219, 2107.8830566406, 18302.65234375, 15853.28125, 3275.7951660156,
                1989.6781005859, 2994.1428222656, 129489.0390625, 8772.9423828125, 9624.65234375, 2279.2514648438,
                47813.2421875, 3664.93359375, 11613.7626953125, 2321.8532714844, 12126.556640625, 8296.2685546875,
                1822.8952636719, 1722.896484375, 47673.96484375, 2785.0285644531, 10243.5849609375, 2748.9553222656,
                3608.4099121094, 3638.7277832031, 1628.3753662109, 16601.212890625, 2849.912109375, 1761.3250732422,
                2083.5766601563, 10057.7685546875, 2778.1420898438, 2034.2290039063, 2067.0754394531, 1703.9493408203,
                4084.3762207031, 9654.53125, 2552.9187011719, 3305.9052734375, 3882.103515625, 2952.1291503906,
                2129.7270507813, 7271.2568359375, 3003.4470214844, 8691.8818359375, 6821.1411132813, 3365.4367675781,
                4150.0244140625, 2061.087890625, 7729.794921875, 19128.609375, 1938.5982666016, 12525.7373046875,
                2002.1896972656, 15945.9072265625, 3485.1372070313, 8829.4130859375, 3555.3149414063, 13025.912109375,
                8675.4248046875, 1862.0024414063, 3703.4956054688, 2420.3032226563, 2036.0102539063, 3351.5231933594,
                3528.8334960938, 1840.9298095703, 2181.7253417969, 2469.3706054688, 2775.1408691406, 17814.412109375,
                2155.9870605469, 2171.2077636719, 3812.1481933594, 2256.4187011719, 7920.4360351563, 1871.3038330078,
                1997.8999023438, 2573.8469238281, 13175.9462890625, 1900.8210449219, 22400.9296875, 11549.626953125,
                38418.265625, 3260.3002929688, 2511.158203125, 2533.1789550781, 2106.8198242188, 9929.2158203125,
                2256.625, 2754.5620117188, 308489.5, 6185.2216796875, 23112.908203125, 4034.25, 2348.0844726563,
                3161.9660644531, 2893.6110839844, 7405.6201171875, 2157.6237792969, 3264.6381835938, 2554.6206054688,
                6101.7568359375, 2789.076171875, 2864.0549316406, 11902.0478515625, 1751.4621582031, 8123.2680664063,
                2398.8227539063, 14463.2314453125, 3298.9870605469, 14054.888671875, 9403.3662109375, 29118.4296875,
                4085.5808105469, 1947.6435546875, 2393.0634765625, 2090.5642089844, 2478.7414550781, 8151.3715820313,
                15998.6669921875, 3128.8696289063, 2162.8344726563, 135145.046875, 57209.77734375, 9003.2880859375,
                4027.4089355469, 2578.5427246094, 3248.123046875, 2326.5983886719, 19032.3203125, 3333.7463378906,
                10943.2353515625, 1998.2120361328, 7042.1284179688, 3352.9057617188, 21185.84375, 9314.7373046875,
                224938.703125, 120104.4609375, 32241.63671875, 8944.8037109375, 2221.5009765625, 3116.8874511719,
                8647.328125, 12209.271484375, 11649.8076171875, 2904.4047851563, 2142.3937988281, 233832.578125,
                128602.0703125, 39626.1875, 2711.2072753906, 14901.775390625, 2754.5363769531, 11371.025390625,
                15030.109375, 6700.9272460938, 7344.9223632813, 6466.0610351563, 2372.8303222656, 1939.7000732422,
                2399.7983398438, 76191.7578125, 29774.83203125, 1931.8090820313, 8459.8271484375, 2116.5856933594,
                2388.4177246094, 2518.5090332031, 9517.69140625, 3505.8095703125, 7540.4692382813, 2521.5549316406,
                3537.5859375, 2684.7912597656, 2156.8911132813, 15875.9521484375, 8220.6181640625, 2553.1765136719,
                3050.9650878906, 23942.1171875, 9961.8681640625, 1922.4764404297, 6831.3383789063, 15087.9443359375,
                8057.642578125, 1899.5068359375, 2391.9763183594, 2532.5729980469);
        double peptidMass = 358.9512;
        int chargeState = 4;
        int id = 0;

        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity, peptidMass, chargeState, id);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper(config));

        MassSpectrum massSpectrumOut = deisotoper.deisotopeMS(massSpectrum);

        assertNotEquals("Size of mZ-Values must be not equal, because there should be isotopic sets!",
                massSpectrum.getMz().size(), massSpectrumOut.getMz().size());

        assertEquals(Arrays.asList(110.0715637, 112.0871277, 113.0712051, 115.086853, 116.0531616, 116.0708923,
                119.7680359, 120.0810165, 125.0275803, 125.2685242, 127.086731, 128.0706177, 129.1023102, 130.0863953,
                130.097641, 137.0709991, 140.7889404, 141.0658722, 144.0479584, 147.1128845, 155.0814972, 157.108429,
                157.1335144, 158.0923615, 161.0743408, 166.061264, 170.0599976, 173.12854, 175.1190186, 180.6107483,
                183.1127319, 183.1492462, 185.1649628, 187.1440735, 189.0690918, 197.1285706, 198.5964813, 198.8642731,
                199.1805878, 201.1234283, 203.1211243, 207.112854, 207.1242523, 208.5289154, 211.1439819, 212.1030121,
                212.1476288, 213.1598816, 215.1389771, 222.16008, 223.1439972, 223.1551056, 223.7337646, 226.1186676,
                227.1752014, 228.1343994, 229.1187897, 229.1292419, 231.1160889, 235.1077576, 235.1188812, 237.0979614,
                240.1704407, 243.1337891, 248.1605225, 250.1550903, 251.1025543, 251.1500549, 254.124527, 255.1446381,
                260.1969299, 267.5011292, 268.1651917, 269.1242981, 271.1510315, 277.6557922, 278.1495972, 280.6687622,
                285.2397156, 286.1217041, 286.1503601, 286.4925842, 287.1345215, 288.2026672, 294.6662292, 295.6795349,
                303.2660217, 303.2738647, 304.1611633, 305.309906, 307.1439209, 309.6772156, 312.1863098, 314.1870422,
                322.1860657, 324.9866943, 326.1813354, 326.9628296, 341.0174866, 342.9970398, 344.9760132, 345.5325623,
                346.9502869, 348.1657104, 351.181427, 351.2023621, 352.2224731, 356.2401733, 359.0281067, 359.203125,
                359.2399292, 359.26474, 361.0241394, 361.2442627, 366.1765747, 366.2191772, 367.2078552, 370.1982117,
                371.2112732, 375.2346497, 382.2078857, 383.2266541, 384.2348633, 392.8924255, 395.7193298, 400.2183533,
                404.2320557, 409.90271, 415.578064, 417.2459106, 430.7393188, 435.7140503, 443.2564392, 443.7356873,
                444.2277527, 452.7399902, 458.1020203, 461.2539368, 466.7385864, 469.3226624, 477.2677612, 487.2818909,
                490.2615356, 490.306366, 491.2842407, 497.3199158, 505.263031, 509.282959, 517.7960815, 519.2999878,
                522.7889404, 523.2798462, 554.3043823, 565.8234253, 566.3268433, 570.3148804, 574.3380127, 575.8415527,
                579.3312378, 586.3196411, 590.3520508, 603.3457642, 604.3508301, 618.3466187, 622.8652954, 633.3452148,
                634.3284912, 651.3571777, 692.9234619, 703.4362183, 714.4061279, 730.3641357, 730.4467773, 731.4320679,
                747.3897705, 790.4317627, 832.4804077, 844.5157471, 856.3361206, 860.4744263, 861.4712524, 862.475769,
                878.4868774, 904.4752808, 915.4468994, 945.5620117, 973.5585938, 979.4803467, 992.5720825, 1017.561829),
                massSpectrumOut.getMz());

        assertEquals(Arrays.asList(23369.3359375, 13809.29296875, 8846.6669921875, 9419.87109375, 8076.8959960938,
                9162.0068359375, 1790.6524658203, 11649.1162109375, 1517.7386474609, 1645.7025146484, 158368.1328125,
                1986.2150878906, 20422.271484375, 13433.8154296875, 2523.6166992188, 11089.599609375, 1737.6016845703,
                2229.7185058594, 3870.6616210938, 28125.26953125, 125105.021484375, 4039.8679199219, 2107.8830566406,
                18302.65234375, 15853.28125, 3275.7951660156, 1989.6781005859, 2994.1428222656, 138261.9814453125,
                9624.65234375, 2279.2514648438, 51478.17578125, 11613.7626953125, 2321.8532714844, 12126.556640625,
                8296.2685546875, 1822.8952636719, 1722.896484375, 50458.9934082031, 10243.5849609375, 2748.9553222656,
                3608.4099121094, 3638.7277832031, 1628.3753662109, 16601.212890625, 2849.912109375, 1761.3250732422,
                2083.5766601563, 10057.7685546875, 2778.1420898438, 2034.2290039063, 2067.0754394531, 1703.9493408203,
                4084.3762207031, 9654.53125, 2552.9187011719, 3305.9052734375, 3882.103515625, 2952.1291503906,
                2129.7270507813, 7271.2568359375, 3003.4470214844, 8691.8818359375, 6821.1411132813, 3365.4367675781,
                4150.0244140625, 2061.087890625, 7729.794921875, 19128.609375, 1938.5982666016, 12525.7373046875,
                2002.1896972656, 15945.9072265625, 3485.1372070313, 8829.4130859375, 3555.3149414063, 13025.912109375,
                8675.4248046875, 1862.0024414063, 3703.4956054688, 2420.3032226563, 2036.0102539063, 3351.5231933594,
                3528.8334960938, 1840.9298095703, 2181.7253417969, 2469.3706054688, 2775.1408691406, 17814.412109375,
                2155.9870605469, 2171.2077636719, 3812.1481933594, 2256.4187011719, 9791.7398681641, 1997.8999023438,
                2573.8469238281, 13175.9462890625, 1900.8210449219, 22400.9296875, 11549.626953125, 38418.265625,
                3260.3002929688, 2511.158203125, 2533.1789550781, 2106.8198242188, 9929.2158203125, 2256.625,
                2754.5620117188, 308489.5, 6185.2216796875, 23112.908203125, 4034.25, 2348.0844726563, 3161.9660644531,
                2893.6110839844, 7405.6201171875, 2157.6237792969, 3264.6381835938, 2554.6206054688, 6101.7568359375,
                2789.076171875, 2864.0549316406, 11902.0478515625, 1751.4621582031, 8123.2680664063, 2398.8227539063,
                14463.2314453125, 3298.9870605469, 23458.2548828125, 33204.0104980469, 1947.6435546875, 2393.0634765625,
                2090.5642089844, 2478.7414550781, 8151.3715820313, 19127.5366210938, 2162.8344726563, 201358.1123046875,
                4027.4089355469, 2578.5427246094, 3248.123046875, 2326.5983886719, 29975.5556640625, 3333.7463378906,
                1998.2120361328, 7042.1284179688, 3352.9057617188, 30500.5810546875, 377284.80078125, 8944.8037109375,
                2221.5009765625, 3116.8874511719, 8647.328125, 12209.271484375, 14554.212402343801, 2142.3937988281,
                402060.8359375, 2711.2072753906, 14901.775390625, 2754.5363769531, 11371.025390625, 15030.109375,
                6700.9272460938, 7344.9223632813, 8838.8913574219, 1939.7000732422, 2399.7983398438, 105966.58984375,
                1931.8090820313, 8459.8271484375, 2116.5856933594, 2388.4177246094, 2518.5090332031, 13023.5009765625,
                7540.4692382813, 2521.5549316406, 3537.5859375, 2684.7912597656, 2156.8911132813, 15875.9521484375,
                8220.6181640625, 2553.1765136719, 3050.9650878906, 33903.9853515625, 1922.4764404297, 6831.3383789063,
                23145.5869140625, 1899.5068359375, 2391.9763183594, 2532.5729980469), massSpectrumOut.getIntensity());

        assertEquals(
                Arrays.asList(-1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                        -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0),
                massSpectrumOut.getIsotope());

        assertEquals(
                Arrays.asList(-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, -1,
                        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1),
                massSpectrumOut.getCharge());
    }

    @Test
    public void testDeisotopeMsFour() {
        List<Double> mz = Arrays.asList(110.0717316, 111.0750885, 113.0714874, 114.1435013, 114.1456909, 115.0546341,
                116.0709763, 117.0576859, 119.8187332, 120.0811386, 129.1025238, 129.8707581, 130.0653687, 130.0864563,
                131.0686188, 132.0810089, 133.0842438, 136.0759277, 141.1024475, 142.0654144, 143.1182098, 145.0609589,
                147.1130219, 148.1160126, 149.0237732, 149.087616, 156.0771942, 156.0806732, 157.0610199, 157.0974426,
                157.1336975, 159.0918732, 159.8486481, 160.0759735, 160.089035, 160.0951996, 161.0744781, 163.4476624,
                164.7080994, 168.0816803, 169.0761871, 169.0975494, 169.1337738, 170.0602875, 170.0795441, 170.1371307,
                171.1130219, 173.1287537, 175.1194458, 175.5101929, 177.3470001, 178.134201, 183.1496429, 184.072052,
                185.0710144, 185.1287384, 185.1651611, 186.1239624, 187.0867767, 187.1078339, 187.1269073, 188.0903625,
                188.4991608, 189.069458, 190.0400543, 195.0877991, 196.0715179, 197.1287231, 198.1321259, 199.1084747,
                200.1396637, 201.0697784, 201.1235657, 202.2896271, 202.9238434, 204.1346893, 204.2815552, 206.1289368,
                209.1398163, 211.0865173, 213.0662231, 213.1601715, 213.4604187, 214.1188354, 215.1390228, 224.1034546,
                225.1597748, 226.0822296, 226.1186218, 227.1220398, 228.1133728, 228.1343536, 229.0970612, 231.0803375,
                235.1192474, 237.1346741, 239.0821838, 239.1754913, 242.1865692, 244.1085968, 244.1292572, 246.0908203,
                246.1235504, 252.1343536, 254.150177, 256.1078796, 256.1299438, 256.1657715, 257.0922852, 258.1603088,
                260.1970825, 263.1506653, 266.1501465, 267.1340027, 268.1079712, 270.1815491, 274.1184692, 275.0332336,
                275.1714478, 276.1018372, 284.1969604, 285.1559143, 286.1403809, 286.1549072, 287.1716614, 288.5036011,
                294.1559143, 294.1813049, 295.1769409, 296.1029358, 296.196991, 296.8799438, 297.1558228, 299.2079773,
                303.1123352, 304.1159363, 305.1820984, 307.13974, 310.1144409, 310.1764832, 311.1712646, 311.2081604,
                313.1294861, 313.1873779, 314.1331787, 315.1662903, 322.1505737, 323.1711731, 327.2028809, 329.1090088,
                329.2189026, 331.1400146, 332.1440735, 339.2030945, 341.1819153, 342.177124, 343.1616821, 347.2290039,
                348.2324829, 349.6464844, 353.2183838, 354.2032471, 355.2129822, 355.234375, 362.2186279, 365.2189636,
                373.1173401, 374.2402344, 377.2113647, 379.2452698, 380.1928711, 380.2485046, 382.2453003, 383.20755,
                384.2113647, 390.144104, 395.1752319, 396.2247314, 397.2804871, 400.2553101, 402.1809692, 404.0759888,
                407.2248535, 407.2402039, 408.2435913, 410.2397156, 411.2001953, 423.2032776, 424.2191772, 425.2033997,
                428.1958313, 428.2505798, 429.04599, 437.2151794, 441.2462463, 442.2489929, 444.1914978, 446.2975159,
                447.9433594, 448.1584778, 452.2297668, 452.2495117, 461.2186279, 466.16922, 468.3179016, 470.2400513,
                471.2030945, 471.2417908, 472.1862183, 473.2189636, 473.3087769, 485.3086548, 487.2550354, 487.7576294,
                489.2137451, 490.1953735, 496.1797485, 501.253418, 501.7545776, 502.1252441, 503.3190613, 504.2940674,
                504.3224487, 505.3251038, 509.2524109, 511.0987244, 515.7683716, 516.269165, 518.119751, 522.77948,
                524.2856445, 525.2120361, 527.2631226, 529.7641602, 530.2431641, 530.2652588, 530.7666016, 531.2015381,
                538.2877197, 539.2434692, 542.2406616, 555.253479, 556.3461304, 559.1975098, 560.249939, 561.3143311,
                565.2365112, 567.2705078, 569.3049316, 572.2848511, 574.3565674, 575.359375, 576.223999, 576.3613281,
                581.5396118, 583.2479248, 584.286438, 584.3406372, 586.3934937, 593.9638672, 594.2989502, 595.267395,
                598.3030396, 599.2608643, 600.2443848, 601.3133545, 607.6031494, 608.8061523, 609.2641602, 609.3086548,
                612.9718018, 613.3059692, 613.6081543, 613.9425049, 614.274231, 614.7553711, 615.208374, 616.2132568,
                616.3116455, 616.8118896, 617.2729492, 617.3782959, 618.3813477, 619.9448853, 622.77771, 622.803772,
                623.3054199, 623.8068848, 624.8232422, 627.6524048, 629.3092651, 630.2705078, 631.2026978, 632.9737549,
                645.3144531, 645.6442261, 645.9807129, 647.2958374, 651.3136597, 651.6469727, 651.9811401, 652.2324829,
                652.3157959, 657.9852295, 658.2653809, 658.3421631, 658.8421631, 665.6379395, 670.2438965, 670.9919434,
                671.3244629, 672.3383179, 672.8397827, 673.3408813, 673.4249878, 673.8405151, 674.3173218, 674.4276123,
                674.6550293, 674.9882813, 675.3240356, 680.3236084, 680.6575317, 680.991394, 681.3267212, 681.6599121,
                683.4100342, 684.3305054, 684.5844727, 684.8329468, 687.3272705, 687.8265991, 688.6652222, 691.3249512,
                691.6613159, 693.6622314, 693.9984131, 694.3317871, 694.6613159, 697.3276367, 697.6617432, 697.9956055,
                698.329834, 698.6630859, 699.3313599, 699.6660767, 700.0044556, 700.3375854, 701.007019, 703.3310547,
                703.6652832, 703.9992065, 704.3331909, 704.4094238, 704.6650391, 705.0012207, 709.3349609, 709.6691284,
                710.0024414, 710.3376465, 710.6723022, 712.3526001, 715.8381348, 716.3411255, 721.3543091, 723.0233154,
                724.3676758, 727.0205078, 727.3582764, 727.6903687, 728.0245361, 728.8811646, 729.3822632, 729.8833618,
                730.3505249, 730.3828125, 730.6860962, 733.0263062, 733.3597412, 733.694397, 734.0288086, 734.3508301,
                734.4171143, 734.850647, 735.3482056, 735.3703613, 736.3536377, 736.6885376, 737.0220337, 737.3555908,
                737.6897583, 741.3156738, 742.3225708, 742.3572388, 742.6908569, 743.0253906, 746.3076172, 747.3059082,
                751.3016357, 752.3049927, 754.3757324, 757.3932495, 757.8936157, 765.3184814, 768.7052002, 769.3126221,
                769.7089844, 770.4425049, 771.4467773, 772.3446045, 773.3483887, 773.8498535, 774.3532715, 774.7095337,
                775.0443726, 776.3646851, 779.3759155, 783.3278198, 787.4681396, 788.4709473, 789.4743042, 798.4371948,
                807.7336426, 807.8608398, 808.7332764, 816.8668823, 817.3668823, 817.868042, 821.9113159, 822.413208,
                837.4520874, 838.3673706, 838.453064, 842.37146, 842.8754272, 843.3748169, 846.3772583, 846.4274292,
                851.375061, 851.8776245, 852.3814697, 852.8808594, 853.446106, 854.4024048, 854.4482422, 860.3816528,
                860.8834229, 864.3866577, 864.4413452, 865.3984985, 865.4439697, 866.4456787, 870.0910645, 871.4151611,
                876.4449463, 876.8938599, 876.9428101, 882.395874, 885.3980713, 886.479126, 886.9120483, 887.4325562,
                887.4806519, 888.486084, 890.4447632, 890.9454346, 891.4459839, 891.906189, 895.9202271, 896.4178467,
                900.9130859, 901.4122314, 901.9099731, 902.4190063, 909.9162598, 910.4176025, 910.9127197, 911.404541,
                911.7730713, 912.1104126, 912.4400635, 918.5072632, 918.9533691, 919.4550781, 919.5126343, 919.9093628,
                920.4069824, 920.9064941, 929.411438, 929.9156494, 937.4380493, 938.4431152, 946.5015259, 951.4351807,
                957.5286865, 958.4495239, 958.5076294, 960.9390869, 963.4534912, 964.4540405, 964.9503784, 965.4395752,
                969.440979, 969.9440918, 970.4411011, 975.5303345, 976.4675293, 976.5336914, 976.9666748, 977.4693604,
                977.5353394, 978.4447632, 978.9459229, 1000.522888, 1001.519958, 1002.505615, 1007.979553, 1010.979797,
                1011.478455, 1011.979065, 1019.982239, 1020.485291, 1020.985291, 1021.491028, 1027.528809, 1028.996826,
                1030.002686, 1036.484375, 1039.993774, 1040.498657, 1040.990479, 1041.497803, 1044.552734, 1045.487061,
                1045.547485, 1045.990234, 1046.487671, 1046.54126, 1046.990967, 1047.489868, 1048.489502, 1048.993042,
                1049.499634, 1050.518677, 1051.524292, 1052.551147, 1054.492432, 1054.994263, 1055.494873, 1055.997925,
                1056.494141, 1056.996094, 1058.526611, 1059.524658, 1062.562866, 1063.499878, 1063.566528, 1064.000244,
                1064.502808, 1064.568726, 1065.001099, 1065.502686, 1065.566284, 1071.55896, 1072.563721, 1078.511353,
                1079.030273, 1079.524658, 1082.025635, 1082.526001, 1087.5177, 1089.476318, 1090.027466, 1090.533691,
                1091.030884, 1091.533325, 1095.022583, 1095.523193, 1096.02417, 1096.529907, 1097.031128, 1098.027588,
                1098.53064, 1099.034912, 1099.536621, 1100.03833, 1100.538574, 1101.033813, 1104.0271, 1104.528198,
                1105.029785, 1105.529419, 1106.029785, 1107.482666, 1107.543091, 1113.032227, 1113.533325, 1114.035156,
                1114.537964, 1115.041992, 1162.06189, 1162.561035, 1184.643921, 1185.650024, 1197.094482, 1197.596924,
                1198.096802, 1198.598511, 1207.564575, 1211.092651, 1211.594482, 1212.094727, 1212.596313, 1225.564453,
                1230.63208, 1231.619629, 1236.586182, 1237.594238, 1244.60083, 1245.602661, 1248.639648, 1249.644287,
                1250.645874, 1273.647949, 1274.118042, 1274.618652, 1343.670898, 1344.67395, 1345.665161, 1373.647583,
                1456.755127, 1457.756226, 1642.819336, 1643.821167, 1644.826904, 1790.831421, 1791.825806, 1800.816895,
                1801.816162, 1818.827759, 1819.831543, 1820.828125, 2464.783936, 2509.217529, 2716.95166, 3355.099121,
                3570.154785);
        List<Double> intensity = Arrays.asList(47788.453125, 1737.6696777344, 5970.6850585938, 917.3888549805,
                939.5751953125, 822.0393676758, 5917.275390625, 749.2538452148, 688.8538818359, 5150.55859375,
                12700.4619140625, 1031.5726318359, 91633.1328125, 11911.7744140625, 6651.0571289063, 20309.021484375,
                1233.9437255859, 4209.2944335938, 1901.6849365234, 1086.9342041016, 1757.8044433594, 3372.4909667969,
                25537.744140625, 1112.3353271484, 993.8350830078, 857.3690185547, 1049.7232666016, 925.8883666992,
                944.6696166992, 1050.0875244141, 4045.3835449219, 251442.328125, 752.5776977539, 1783.7507324219,
                1205.7958984375, 25227.919921875, 7487.5688476563, 790.6477050781, 945.9669189453, 648.8961791992,
                1476.3402099609, 3617.7416992188, 46274.10546875, 3968.9240722656, 993.6923217773, 4256.0903320313,
                4586.421875, 4699.125, 1700.5161132813, 832.8430786133, 815.8041992188, 1382.4418945313,
                1709.3919677734, 3113.2119140625, 841.2858886719, 9314.4384765625, 1393.4104003906, 15200.650390625,
                15782.5048828125, 1252.0137939453, 822.8172607422, 1547.7625732422, 669.2496337891, 5049.498046875,
                794.2931518555, 1244.6156005859, 760.0761108398, 28841.677734375, 1219.2727050781, 850.3496704102,
                1204.4681396484, 1031.7841796875, 3839.8344726563, 758.8597412109, 862.5513305664, 4016.3581542969,
                708.1723632813, 780.0186767578, 1685.1071777344, 738.2857055664, 1471.0837402344, 1313.3563232422,
                791.5373535156, 8375.4609375, 1507.3167724609, 1641.0607910156, 1060.5357666016, 858.3902587891,
                6142.9721679688, 838.0795288086, 1315.7822265625, 5921.6303710938, 1482.3264160156, 1648.0867919922,
                3869.18359375, 4674.2578125, 922.9401855469, 1544.3409423828, 1026.0075683594, 980.291809082,
                5878.119140625, 9754.8544921875, 1621.3037109375, 939.0039672852, 4740.4799804688, 5829.4340820313,
                822.0532226563, 1261.6787109375, 6604.3515625, 1068.9738769531, 13190.9853515625, 3413.8022460938,
                5194.060546875, 1462.0284423828, 1501.5964355469, 1540.2293701172, 6198.84765625, 753.973449707,
                4984.0107421875, 1656.9188232422, 11110.4482421875, 4234.9633789063, 1126.1105957031, 1304.6494140625,
                799.3264770508, 746.9828491211, 991.9315185547, 1156.4425048828, 1352.9055175781, 1584.6330566406,
                3059.2697753906, 855.0618286133, 4920.0864257813, 926.9799194336, 6119.5161132813, 791.9954223633,
                844.6307983398, 836.0263671875, 1008.5850830078, 3372.005859375, 3696.4951171875, 864.3911132813,
                12844.8017578125, 908.274230957, 1834.2757568359, 3989.9555664063, 1606.6995849609, 9975.9814453125,
                1804.2072753906, 952.6251831055, 1291.9171142578, 1800.9390869141, 1090.0581054688, 1846.1746826172,
                1279.5201416016, 6770.0698242188, 1404.8444824219, 43977.2421875, 7301.9965820313, 822.2608642578,
                1296.0942382813, 987.6251220703, 1036.2705078125, 3172.0148925781, 1695.9376220703, 806.6272583008,
                1190.5167236328, 3155.5910644531, 1068.1303710938, 28719.140625, 1390.3077392578, 4200.1791992188,
                1503.8493652344, 8822.919921875, 1250.0969238281, 7431.724609375, 847.6892700195, 1560.9754638672,
                1438.1427001953, 3045.8989257813, 3680.900390625, 1240.1544189453, 1474.3619384766, 46273.578125,
                8548.70703125, 5446.5, 985.6581420898, 838.8226318359, 1018.8344726563, 1768.654296875, 1217.8601074219,
                4379.1630859375, 903.3884887695, 1103.6077880859, 1856.6112060547, 1272.8026123047, 1641.4322509766,
                6960.2138671875, 878.3200683594, 1296.8967285156, 2922.1474609375, 1119.2816162109, 1471.0753173828,
                3090.3583984375, 1306.646484375, 3230.1372070313, 1123.7708740234, 1092.0849609375, 1855.1292724609,
                1406.7728271484, 2753.2082519531, 1000.4825439453, 7944.0493164063, 4078.6096191406, 3102.9543457031,
                898.0745849609, 1282.572265625, 7385.5239257813, 3462.6994628906, 1258.6585693359, 44530.1328125,
                1704.0416259766, 10821.259765625, 1075.3754882813, 1581.2611083984, 787.3862304688, 963.9877929688,
                1621.9241943359, 3368.8366699219, 1080.9180908203, 959.7768554688, 1054.7877197266, 1427.7955322266,
                17624.2265625, 1616.53515625, 10002.4755859375, 3617.3122558594, 1195.6657714844, 1797.6691894531,
                862.2873535156, 932.1994628906, 1083.2662353516, 3190.5229492188, 1820.1529541016, 1347.7703857422,
                1844.2508544922, 1851.2824707031, 1163.8135986328, 3331.6320800781, 1020.2813110352, 90197.1484375,
                23162.349609375, 956.510559082, 3221.1303710938, 844.4879150391, 1249.3692626953, 1437.4058837891,
                1061.8962402344, 1569.0501708984, 1913.3450927734, 1157.5645751953, 1655.4517822266, 897.396484375,
                1060.1584472656, 1198.2664794922, 1432.5422363281, 1707.9252929688, 12908.39453125, 1069.7329101563,
                7162.2255859375, 1876.5844726563, 3923.8000488281, 1184.6848144531, 3432.6137695313, 1117.4447021484,
                1136.7791748047, 1573.1392822266, 961.1552124023, 5698.1157226563, 3039.9016113281, 1056.3117675781,
                3663.9233398438, 1413.5324707031, 1549.4477539063, 1459.6181640625, 16642.9296875, 10823.02734375,
                5098.0278320313, 1178.3933105469, 1375.4372558594, 3620.8732910156, 875.7815551758, 885.9979248047,
                919.8340454102, 1126.2406005859, 1096.6119384766, 1050.2271728516, 1169.9293212891, 15704.0576171875,
                17124.2109375, 12237.82421875, 4546.7646484375, 1465.4053955078, 965.6090698242, 1708.5213623047,
                3439.62109375, 1705.4487304688, 1311.0523681641, 1489.5023193359, 1736.8395996094, 4545.7817382813,
                13744.01953125, 10295.4599609375, 3270.9799804688, 24745.673828125, 924.0198974609, 5570.9072265625,
                10393.4169921875, 7389.6357421875, 7477.7163085938, 5207.5375976563, 17905.9375, 18573.828125,
                9160.0107421875, 5053.0478515625, 3213.837890625, 1573.5350341797, 1575.6889648438, 981.3447875977,
                1595.2370605469, 845.0737915039, 1371.0966796875, 1042.4802246094, 3231.9575195313, 2978.9580078125,
                1154.6602783203, 1364.9071044922, 4117.5258789063, 1526.1866455078, 43409.2890625, 52851.24609375,
                28689.986328125, 15994.0419921875, 1252.4710693359, 1559.1013183594, 3399.4890136719, 3815.3618164063,
                5219.0727539063, 1023.4760742188, 67630.171875, 85167.8125, 60315.45703125, 25921.869140625,
                3231.6879882813, 5976.658203125, 1532.7022705078, 18843.53515625, 23759.259765625, 16628.5390625,
                4171.6791992188, 1340.3514404297, 926.884765625, 1149.7276611328, 4892.671875, 3941.0012207031,
                1183.4425048828, 1559.2093505859, 7030.1254882813, 4498.8193359375, 7452.5297851563, 3116.9819335938,
                6496.1518554688, 3955.3508300781, 3760.4367675781, 3308.6469726563, 1369.5816650391, 5136.9301757813,
                11316.1884765625, 19971.083984375, 9465.33984375, 4298.310546875, 5531.0830078125, 3322.8303222656,
                1406.3735351563, 1233.7392578125, 934.8732910156, 24753.259765625, 35177.84765625, 25026.30078125,
                12226.7236328125, 3384.23046875, 1700.0692138672, 1162.2430419922, 10290.0283203125, 7447.2407226563,
                7172.306640625, 1016.0512695313, 1376.8612060547, 3149.4116210938, 1895.6629638672, 1206.7896728516,
                1386.2198486328, 1619.9240722656, 3541.0642089844, 3178.4113769531, 1382.8624267578, 1176.2454833984,
                1807.9305419922, 1717.9302978516, 1779.2250976563, 4128.1806640625, 5712.6918945313, 1786.1372070313,
                3053.48828125, 1865.8383789063, 1664.7785644531, 1201.7879638672, 1325.0793457031, 52047.2109375,
                19018.41015625, 1927.5084228516, 1429.0612792969, 3425.5654296875, 1542.8937988281, 1844.9367675781,
                1598.8416748047, 3408.3347167969, 1599.1623535156, 7023.3095703125, 5813.3061523438, 3734.0942382813,
                1226.0560302734, 1348.4309082031, 9311.2548828125, 6178.1499023438, 4597.904296875, 3469.72265625,
                1357.6405029297, 7822.185546875, 8484.1611328125, 3481.7712402344, 3242.1499023438, 1191.9638671875,
                3477.9790039063, 1092.1700439453, 1127.1335449219, 5321.73046875, 5806.08203125, 5427.0131835938,
                2302.2922363281, 3761.7521972656, 1441.3260498047, 1348.1474609375, 1499.5151367188, 3571.4887695313,
                1208.8360595703, 5409.9658203125, 3345.1018066406, 1218.5887451172, 17432.603515625, 1666.3074951172,
                1294.0152587891, 8270.4814453125, 964.1950683594, 7649.4716796875, 8050.0185546875, 3200.2397460938,
                3259.8061523438, 4073.6687011719, 2670.654296875, 5663.2109375, 7312.6772460938, 1209.1116943359,
                1299.3277587891, 4459.1015625, 3323.4653320313, 1602.1741943359, 1690.9323730469, 1405.6262207031,
                1779.6893310547, 1469.8343505859, 13962.4482421875, 3537.7595214844, 3235.1359863281, 8114.2700195313,
                2784.4916992188, 3481.5632324219, 1287.8597412109, 1085.9998779297, 938.4287109375, 3794.8276367188,
                1462.4178466797, 1041.6955566406, 1247.3677978516, 1002.8555908203, 965.198059082, 3757.123046875,
                910.4429931641, 1361.5458984375, 1331.5538330078, 1748.5361328125, 915.8793945313, 3224.9450683594,
                1431.2634277344, 1193.9223632813, 38410.78125, 8648.3349609375, 18086.669921875, 14185.359375,
                5995.3872070313, 4629.8237304688, 1783.0974121094, 1508.0793457031, 8522.0166015625, 4596.37890625,
                1015.9674682617, 997.0551147461, 3640.703125, 5179.9306640625, 3657.1584472656, 7968.984375,
                3547.0510253906, 3606.0014648438, 1639.5264892578, 1186.9573974609, 1119.6104736328, 1299.9764404297,
                1304.8156738281, 1288.7913818359, 1556.7980957031, 4113.60546875, 3889.8461914063, 4940.9985351563,
                15352.2958984375, 3231.1518554688, 20367.732421875, 11349.5322265625, 2214.8920898438, 4478.1420898438,
                1189.6170654297, 1083.1583251953, 1346.7629394531, 3179.7590332031, 1208.1240234375, 1309.6759033203,
                935.4395751953, 38785.26953125, 51337.9375, 31046.982421875, 11682.2236328125, 3838.48046875,
                1023.4222412109, 4167.138671875, 1139.1741943359, 35955.1796875, 16340.6982421875, 19491.212890625,
                17835.8203125, 11156.7216796875, 6457.1635742188, 6681.5541992188, 1085.5550537109, 1064.5690917969,
                5995.7060546875, 4848.0590820313, 1108.5704345703, 1091.1260986328, 1529.8232421875, 1008.6722412109,
                888.5922241211, 1038.0451660156, 1784.1335449219, 5423.5419921875, 8907.8046875, 7426.4814453125,
                1347.8237304688, 4102.0395507813, 7583.7836914063, 3461.7109375, 1470.7426757813, 927.6720581055,
                1235.8253173828, 1332.2719726563, 16242.1220703125, 19818.701171875, 11274.6240234375, 6657.7265625,
                1094.2983398438, 48899.4609375, 67976.3046875, 42702.62109375, 17631.08203125, 6462.6918945313,
                1251.7163085938, 1103.6932373047, 46745.48828125, 50841.171875, 35102.04296875, 14471.91796875,
                5596.046875, 4302.623046875, 1677.9708251953, 4797.9360351563, 4423.2041015625, 1689.6416015625,
                4163.3403320313, 1490.5372314453, 1037.5323486328, 1119.2351074219, 18714.068359375, 17414.689453125,
                12753.671875, 6590.3696289063, 987.9391479492, 1326.0493164063, 1874.6927490234, 1895.6868896484,
                1092.7957763672, 1381.8898925781, 1932.6861572266, 10001.8818359375, 5638.0224609375, 1056.6605224609,
                1194.8231201172, 3496.2192382813, 1795.1412353516, 4512.505859375, 1692.3529052734, 1937.5101318359,
                1100.1870117188, 5200.4814453125, 4441.986328125, 5045.6923828125, 4811.0444335938, 976.0278930664,
                1283.7696533203, 2596.3044433594, 3581.783203125, 1840.4649658203, 1829.8547363281, 1227.9721679688,
                1310.1462402344, 887.2319946289, 859.5553588867, 1039.2114257813, 1035.4152832031, 951.3851318359);
        double peptidMass = 734.569888689532;
        int chargeState = 5;
        int id = 0;

        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity, peptidMass, chargeState, id);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper(config));

        MassSpectrum massSpectrumOut = deisotoper.deisotopeMS(massSpectrum);

        assertNotEquals("Size of mZ-Values must be not equal, because there should be isotopic sets!",
                massSpectrum.getMz().size(), massSpectrumOut.getMz().size());

        assertEquals(Arrays.asList(110.0717316, 113.0714874, 114.1435013, 114.1456909, 115.0546341, 116.0709763,
                117.0576859, 119.8187332, 120.0811386, 129.1025238, 129.8707581, 130.0653687, 130.0864563, 132.0810089,
                136.0759277, 141.1024475, 142.0654144, 143.1182098, 145.0609589, 147.1130219, 149.0237732, 149.087616,
                156.0771942, 156.0806732, 157.0610199, 157.0974426, 157.1336975, 159.0918732, 159.8486481, 160.0759735,
                160.089035, 163.4476624, 164.7080994, 168.0816803, 169.0761871, 169.0975494, 169.1337738, 170.0602875,
                171.1130219, 173.1287537, 175.1194458, 175.5101929, 177.3470001, 178.134201, 183.1496429, 184.072052,
                185.1287384, 185.1651611, 186.1239624, 187.0867767, 187.1078339, 188.0903625, 188.4991608, 189.069458,
                190.0400543, 195.0877991, 196.0715179, 197.1287231, 199.1084747, 200.1396637, 201.0697784, 201.1235657,
                202.2896271, 202.9238434, 204.1346893, 204.2815552, 206.1289368, 209.1398163, 211.0865173, 213.0662231,
                213.1601715, 213.4604187, 214.1188354, 215.1390228, 224.1034546, 225.1597748, 226.0822296, 226.1186218,
                228.1133728, 228.1343536, 229.0970612, 231.0803375, 235.1192474, 237.1346741, 239.0821838, 239.1754913,
                242.1865692, 244.1085968, 244.1292572, 246.0908203, 246.1235504, 252.1343536, 254.150177, 256.1078796,
                256.1299438, 256.1657715, 257.0922852, 258.1603088, 260.1970825, 263.1506653, 266.1501465, 267.1340027,
                268.1079712, 270.1815491, 274.1184692, 275.0332336, 275.1714478, 276.1018372, 284.1969604, 285.1559143,
                286.1403809, 287.1716614, 288.5036011, 294.1559143, 294.1813049, 295.1769409, 296.1029358, 296.196991,
                296.8799438, 297.1558228, 299.2079773, 303.1123352, 304.1159363, 305.1820984, 307.13974, 310.1144409,
                310.1764832, 311.1712646, 311.2081604, 313.1294861, 313.1873779, 314.1331787, 315.1662903, 322.1505737,
                323.1711731, 327.2028809, 329.1090088, 329.2189026, 331.1400146, 332.1440735, 339.2030945, 341.1819153,
                342.177124, 343.1616821, 347.2290039, 349.6464844, 353.2183838, 354.2032471, 355.2129822, 355.234375,
                362.2186279, 365.2189636, 373.1173401, 374.2402344, 377.2113647, 379.2452698, 380.1928711, 382.2453003,
                383.20755, 384.2113647, 390.144104, 395.1752319, 396.2247314, 397.2804871, 400.2553101, 402.1809692,
                404.0759888, 407.2248535, 407.2402039, 410.2397156, 411.2001953, 423.2032776, 424.2191772, 425.2033997,
                428.1958313, 428.2505798, 429.04599, 437.2151794, 441.2462463, 444.1914978, 446.2975159, 447.9433594,
                448.1584778, 452.2297668, 452.2495117, 461.2186279, 466.16922, 468.3179016, 470.2400513, 471.2030945,
                472.1862183, 473.2189636, 473.3087769, 485.3086548, 487.2550354, 489.2137451, 490.1953735, 496.1797485,
                501.253418, 502.1252441, 503.3190613, 504.2940674, 509.2524109, 511.0987244, 515.7683716, 518.119751,
                522.77948, 524.2856445, 525.2120361, 527.2631226, 529.7641602, 530.2431641, 531.2015381, 538.2877197,
                539.2434692, 542.2406616, 555.253479, 556.3461304, 559.1975098, 560.249939, 561.3143311, 565.2365112,
                567.2705078, 569.3049316, 572.2848511, 574.3565674, 576.223999, 581.5396118, 583.2479248, 584.286438,
                584.3406372, 586.3934937, 593.9638672, 595.267395, 598.3030396, 599.2608643, 600.2443848, 601.3133545,
                607.6031494, 608.8061523, 609.2641602, 612.9718018, 613.6081543, 614.7553711, 615.208374, 616.2132568,
                616.3116455, 617.2729492, 617.3782959, 619.9448853, 622.77771, 622.803772, 624.8232422, 627.6524048,
                629.3092651, 630.2705078, 631.2026978, 632.9737549, 645.3144531, 645.6442261, 647.2958374, 651.3136597,
                652.2324829, 652.3157959, 657.9852295, 658.2653809, 658.3421631, 665.6379395, 670.2438965, 670.9919434,
                672.3383179, 673.4249878, 673.8405151, 674.3173218, 674.6550293, 680.3236084, 681.3267212, 683.4100342,
                684.3305054, 684.5844727, 687.3272705, 688.6652222, 691.3249512, 693.6622314, 693.9984131, 697.3276367,
                698.329834, 699.3313599, 700.0044556, 700.3375854, 703.3310547, 704.3331909, 704.4094238, 709.3349609,
                710.3376465, 712.3526001, 715.8381348, 721.3543091, 723.0233154, 724.3676758, 727.0205078, 727.3582764,
                728.8811646, 730.3505249, 730.3828125, 733.0263062, 734.0288086, 734.3508301, 734.4171143, 735.3703613,
                736.3536377, 737.3555908, 741.3156738, 742.3225708, 742.3572388, 746.3076172, 751.3016357, 754.3757324,
                757.3932495, 765.3184814, 768.7052002, 769.3126221, 769.7089844, 770.4425049, 771.4467773, 772.3446045,
                773.3483887, 774.3532715, 774.7095337, 776.3646851, 779.3759155, 783.3278198, 787.4681396, 798.4371948,
                807.7336426, 807.8608398, 816.8668823, 821.9113159, 837.4520874, 838.3673706, 842.37146, 842.8754272,
                846.3772583, 846.4274292, 851.375061, 851.8776245, 852.3814697, 853.446106, 854.4024048, 860.3816528,
                864.3866577, 864.4413452, 865.3984985, 870.0910645, 871.4151611, 876.4449463, 876.8938599, 882.395874,
                885.3980713, 886.479126, 886.9120483, 887.4325562, 888.486084, 890.4447632, 891.906189, 895.9202271,
                900.9130859, 902.4190063, 909.9162598, 910.9127197, 911.404541, 911.7730713, 912.1104126, 912.4400635,
                918.5072632, 918.9533691, 919.5126343, 919.9093628, 929.411438, 929.9156494, 937.4380493, 938.4431152,
                946.5015259, 951.4351807, 957.5286865, 958.4495239, 958.5076294, 960.9390869, 963.4534912, 964.9503784,
                965.4395752, 969.440979, 969.9440918, 975.5303345, 976.4675293, 978.4447632, 1000.522888, 1001.519958,
                1002.505615, 1007.979553, 1010.979797, 1019.982239, 1021.491028, 1027.528809, 1028.996826, 1030.002686,
                1036.484375, 1039.993774, 1040.498657, 1040.990479, 1044.552734, 1045.487061, 1045.547485, 1045.990234,
                1046.54126, 1048.489502, 1048.993042, 1049.499634, 1050.518677, 1051.524292, 1052.551147, 1054.492432,
                1055.997925, 1056.494141, 1058.526611, 1062.562866, 1063.499878, 1063.566528, 1065.001099, 1071.55896,
                1072.563721, 1078.511353, 1079.030273, 1079.524658, 1082.025635, 1087.5177, 1089.476318, 1090.027466,
                1090.533691, 1095.022583, 1096.529907, 1098.027588, 1099.034912, 1100.538574, 1101.033813, 1104.0271,
                1105.529419, 1107.482666, 1107.543091, 1113.032227, 1114.537964, 1115.041992, 1162.06189, 1184.643921,
                1185.650024, 1197.094482, 1198.598511, 1207.564575, 1211.092651, 1212.596313, 1225.564453, 1230.63208,
                1231.619629, 1236.586182, 1237.594238, 1244.60083, 1248.639648, 1249.644287, 1273.647949, 1274.118042,
                1343.670898, 1345.665161, 1373.647583, 1456.755127, 1642.819336, 1644.826904, 1790.831421, 1791.825806,
                1800.816895, 1818.827759, 1819.831543, 1820.828125, 2464.783936, 2509.217529, 2716.95166, 3355.099121,
                3570.154785), massSpectrumOut.getMz());

        assertEquals(Arrays.asList(49526.1228027344, 5970.6850585938, 917.3888549805, 939.5751953125, 822.0393676758,
                5917.275390625, 749.2538452148, 688.8538818359, 5150.55859375, 12700.4619140625, 1031.5726318359,
                98284.1899414063, 11911.7744140625, 21542.9652099609, 4209.2944335938, 1901.6849365234, 1086.9342041016,
                1757.8044433594, 3372.4909667969, 26650.0794677734, 993.8350830078, 857.3690185547, 1049.7232666016,
                925.8883666992, 944.6696166992, 1050.0875244141, 4045.3835449219, 276670.248046875, 752.5776977539,
                9271.3195800782, 1205.7958984375, 790.6477050781, 945.9669189453, 648.8961791992, 2470.0325317382,
                3617.7416992188, 50530.1958007813, 3968.9240722656, 4586.421875, 4699.125, 1700.5161132813,
                832.8430786133, 815.8041992188, 1382.4418945313, 1709.3919677734, 3954.4978027344, 9314.4384765625,
                1393.4104003906, 16023.4676513672, 15782.5048828125, 1252.0137939453, 1547.7625732422, 669.2496337891,
                5049.498046875, 794.2931518555, 1244.6156005859, 760.0761108398, 30060.9504394531, 850.3496704102,
                1204.4681396484, 1031.7841796875, 3839.8344726563, 758.8597412109, 862.5513305664, 4016.3581542969,
                708.1723632813, 780.0186767578, 1685.1071777344, 738.2857055664, 1471.0837402344, 1313.3563232422,
                791.5373535156, 8375.4609375, 1507.3167724609, 1641.0607910156, 1060.5357666016, 858.3902587891,
                6981.0516967774, 1315.7822265625, 5921.6303710938, 1482.3264160156, 1648.0867919922, 3869.18359375,
                4674.2578125, 922.9401855469, 1544.3409423828, 1026.0075683594, 980.291809082, 5878.119140625,
                9754.8544921875, 1621.3037109375, 939.0039672852, 4740.4799804688, 5829.4340820313, 822.0532226563,
                1261.6787109375, 6604.3515625, 1068.9738769531, 13190.9853515625, 3413.8022460938, 5194.060546875,
                1462.0284423828, 1501.5964355469, 1540.2293701172, 6198.84765625, 753.973449707, 4984.0107421875,
                1656.9188232422, 11110.4482421875, 5539.6127929688, 1126.1105957031, 799.3264770508, 746.9828491211,
                991.9315185547, 1156.4425048828, 1352.9055175781, 1584.6330566406, 3059.2697753906, 855.0618286133,
                4920.0864257813, 926.9799194336, 6119.5161132813, 791.9954223633, 844.6307983398, 836.0263671875,
                1008.5850830078, 3372.005859375, 3696.4951171875, 864.3911132813, 12844.8017578125, 908.274230957,
                1834.2757568359, 3989.9555664063, 1606.6995849609, 9975.9814453125, 1804.2072753906, 952.6251831055,
                1291.9171142578, 1800.9390869141, 1090.0581054688, 1846.1746826172, 1279.5201416016, 6770.0698242188,
                1404.8444824219, 51279.2387695313, 822.2608642578, 1296.0942382813, 987.6251220703, 1036.2705078125,
                3172.0148925781, 1695.9376220703, 806.6272583008, 1190.5167236328, 3155.5910644531, 1068.1303710938,
                32919.3198242188, 1390.3077392578, 1503.8493652344, 8822.919921875, 1250.0969238281, 7431.724609375,
                847.6892700195, 1560.9754638672, 1438.1427001953, 3045.8989257813, 3680.900390625, 1240.1544189453,
                1474.3619384766, 54822.28515625, 5446.5, 985.6581420898, 838.8226318359, 1018.8344726563,
                1768.654296875, 1217.8601074219, 4379.1630859375, 903.3884887695, 1103.6077880859, 3129.4138183594,
                1641.4322509766, 6960.2138671875, 878.3200683594, 1296.8967285156, 2922.1474609375, 1119.2816162109,
                1471.0753173828, 3090.3583984375, 1306.646484375, 4322.2221679688, 1123.7708740234, 1855.1292724609,
                1406.7728271484, 2753.2082519531, 1000.4825439453, 12022.6589355469, 3102.9543457031, 898.0745849609,
                1282.572265625, 10848.2233886719, 1258.6585693359, 56426.7680664063, 1704.0416259766, 1581.2611083984,
                787.3862304688, 2585.9119873047002, 3368.8366699219, 1080.9180908203, 959.7768554688, 1054.7877197266,
                1427.7955322266, 31244.0144042969, 1616.53515625, 1195.6657714844, 1797.6691894531, 862.2873535156,
                932.1994628906, 1083.2662353516, 3190.5229492188, 1820.1529541016, 1347.7703857422, 1844.2508544922,
                1851.2824707031, 1163.8135986328, 3331.6320800781, 1020.2813110352, 116580.6284179688, 956.510559082,
                844.4879150391, 1249.3692626953, 1437.4058837891, 1061.8962402344, 1569.0501708984, 3070.9096679687,
                1655.4517822266, 897.396484375, 1060.1584472656, 1198.2664794922, 1432.5422363281, 1707.9252929688,
                20070.6201171875, 1069.7329101563, 5800.3845214844005, 5734.743286132801, 1136.7791748047,
                1573.1392822266, 961.1552124023, 8738.0173339844, 1056.3117675781, 5077.4558105469005, 1549.4477539063,
                1459.6181640625, 32563.9848632813, 1178.3933105469, 1375.4372558594, 3620.8732910156, 875.7815551758,
                885.9979248047, 919.8340454102, 1126.2406005859, 2146.8391113282, 1169.9293212891, 45066.0927734375,
                4546.7646484375, 1465.4053955078, 965.6090698242, 1708.5213623047, 5145.0698242188, 1311.0523681641,
                1489.5023193359, 6282.6213378907, 27310.4594726563, 35139.0908203125, 924.0198974609, 5570.9072265625,
                20074.889648437602, 45639.7763671875, 8266.8857421875, 1573.5350341797, 3170.9260253907, 981.3447875977,
                2216.1704711914, 1042.4802246094, 6210.9155273438, 2680.8469238281, 5482.4329833985, 124950.521484375,
                17246.5130615234, 4958.5903320313, 4838.8378906251, 5219.0727539063, 213113.44140625, 33431.2296142578,
                3231.6879882813, 59231.333984375, 5512.0306396485, 926.884765625, 6042.3995361328, 3941.0012207031,
                1183.4425048828, 1559.2093505859, 7030.1254882813, 15068.331054687602, 14211.939453125, 8445.5771484376,
                1369.5816650391, 40752.6123046875, 4298.310546875, 8171.1958007813, 3322.8303222656, 934.8732910156,
                84957.408203125, 15610.9541015625, 1700.0692138672, 1162.2430419922, 24909.5756835938,
                2392.9124755860003, 5045.074584961, 1206.7896728516, 3006.1439208984, 3541.0642089844, 3178.4113769531,
                1382.8624267578, 1176.2454833984, 1807.9305419922, 1717.9302978516, 1779.2250976563, 9840.872558593801,
                1786.1372070313, 4919.3266601563, 1664.7785644531, 1201.7879638672, 1325.0793457031, 72993.1295166016,
                1429.0612792969, 5270.5021972655995, 1542.8937988281, 6606.3387451172, 12836.615722656301,
                5082.5251464844005, 1226.0560302734, 13909.1591796875, 6178.1499023438, 3469.72265625, 1357.6405029297,
                7822.185546875, 11726.311035156301, 3481.7712402344, 2284.1339111327998, 3477.9790039063,
                6448.8640136719005, 5806.08203125, 10630.0914306641, 2302.2922363281, 1348.1474609375, 1499.5151367188,
                8981.454589843801, 1208.8360595703, 3345.1018066406, 1218.5887451172, 25703.0849609375, 1666.3074951172,
                1294.0152587891, 964.1950683594, 18899.7299804688, 3259.8061523438, 6744.3229980469005,
                14184.9998779297, 1299.3277587891, 7782.5668945313, 1602.1741943359, 1690.9323730469, 1405.6262207031,
                1779.6893310547, 1469.8343505859, 13962.4482421875, 6772.8955078125, 8114.2700195313, 7553.9146728516,
                1085.9998779297, 938.4287109375, 3794.8276367188, 1462.4178466797, 1041.6955566406, 1247.3677978516,
                1002.8555908203, 965.198059082, 3757.123046875, 910.4429931641, 2693.0997314452998, 1748.5361328125,
                915.8793945313, 4418.8674316407, 1431.2634277344, 61127.2749023438, 28829.0815429688, 3291.1767578125,
                8522.0166015625, 4596.37890625, 1015.9674682617, 997.0551147461, 12477.7922363281, 15122.0368652344,
                1639.5264892578, 1186.9573974609, 1119.6104736328, 1299.9764404297, 1304.8156738281, 1288.7913818359,
                5446.6442871094005, 4113.60546875, 4940.9985351563, 27891.4451904297, 3231.1518554688, 24845.8745117188,
                2214.8920898438, 1083.1583251953, 1346.7629394531, 3179.7590332031, 1208.1240234375, 1309.6759033203,
                935.4395751953, 121170.189453125, 12705.6458740234, 3838.48046875, 5306.3128662109, 35955.1796875,
                45333.240234375, 27012.9455566407, 7767.1092529297, 5995.7060546875, 4848.0590820313, 1108.5704345703,
                1091.1260986328, 1529.8232421875, 1897.2644653319999, 1038.0451660156, 1784.1335449219, 12850.0234375,
                10255.628417968801, 15147.5341796876, 2398.4147338868, 2568.0972900391, 47335.447265625, 6657.7265625,
                1094.2983398438, 159578.38671875, 24093.7739257813, 1251.7163085938, 1103.6932373047, 132688.703125,
                14471.91796875, 5596.046875, 5980.5938720703, 4797.9360351563, 4423.2041015625, 7343.5191650391,
                1037.5323486328, 1119.2351074219, 48882.4296875, 6590.3696289063, 987.9391479492, 1326.0493164063,
                1874.6927490234, 1895.6868896484, 1092.7957763672, 3314.5760498047002, 10001.8818359375,
                6694.6829833984, 1194.8231201172, 5291.3604736329, 6204.8587646484, 1937.5101318359, 1100.1870117188,
                9642.4677734375, 9856.736816406301, 976.0278930664, 1283.7696533203, 2596.3044433594, 5422.2481689453,
                1829.8547363281, 1227.9721679688, 1310.1462402344, 887.2319946289, 859.5553588867, 1039.2114257813,
                1035.4152832031, 951.3851318359), massSpectrumOut.getIntensity());

        assertEquals(Arrays.asList(-1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, 2.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, 1.0, -1.0, -1.0, 1.0, -1.0, 1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, 2.0, -1.0,
                -1.0, 2.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
                -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0), massSpectrumOut.getIsotope());

        assertEquals(Arrays.asList(-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, 1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, 2, -1, -1, 2, -1, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, 2, -1, -1, -1, -1, -1, 1, -1, -1, 1, -1, -1, -1, -1, -1, -1,
                -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1),
                massSpectrumOut.getCharge());
    }

    @Test
    public void testDeisotopeMsFiveNegative() {
        List<Double> mz = Arrays.asList(1.0, 3.0, 5.0, 8.0, 12.0, 16.0, 22.0, 32.0);
        List<Double> intensity = Arrays.asList(1.0, 3.0, 5.0, 8.0, 12.0, 16.0, 22.0, 32.0);
        double peptidMass = 500.0;
        int chargeState = 2;
        int id = 0;

        MassSpectrum massSpectrum = new MassSpectrum(mz, intensity, peptidMass, chargeState, id);

        Configuration config = new Configuration();

        DeisotoperMassSpectrumAdapter deisotoper = new DeisotoperMassSpectrumAdapter(new Deisotoper(config));

        MassSpectrum massSpectrumOut = deisotoper.deisotopeMS(massSpectrum);

        assertEquals("Size of mZ-Values must be equal, because there should be no isotopic sets!",
                massSpectrum.getMz().size(), massSpectrumOut.getMz().size());
    }
}
