% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format.r
\name{format}
\alias{format}
\alias{format.datetimeoffset}
\alias{format_iso8601}
\alias{format_pdfmark}
\alias{format_edtf}
\alias{format_exiftool}
\alias{format_strftime}
\alias{format_nanotime}
\title{Convert datetime objects to character}
\usage{
\method{format}{datetimeoffset}(x, ...)

format_iso8601(
  x,
  offsets = TRUE,
  precision = NULL,
  sep = ":",
  mode = c("normal", "xmp"),
  ...
)

format_pdfmark(x)

format_edtf(x, offsets = TRUE, precision = NULL, usetz = FALSE, ...)

format_exiftool(x, mode = c("normal", "xmp", "pdf"), ...)

format_strftime(
  x,
  format = "\%Y-\%m-\%d \%H:\%M:\%S",
  tz = get_tz(x),
  usetz = FALSE,
  fill = mode_tz(x)
)

format_nanotime(
  x,
  format = "\%Y-\%m-\%dT\%H:\%M:\%E9S\%Ez",
  tz = get_tz(x),
  fill = ""
)
}
\arguments{
\item{x}{A \code{\link[=datetimeoffset]{datetimeoffset()}} object.}

\item{...}{Ignored}

\item{offsets}{Include the UTC offsets in the formatting}

\item{precision}{The amount of precision: either "year", "month", "day", "hour", "minute", "second",
"decisecond", "centisecond", "millisecond",
"hundred microseconds", "ten microseconds", "microsecond",
"hundred nanoseconds", "ten nanoseconds", or "nanosecond".
If \code{NULL} then full precision for the object is shown.}

\item{sep}{UTC offset separator.  Either ":" or "".}

\item{mode}{If \code{mode = "xmp"} only output valid XMP metadata datetime values.
If \code{mode = "pdf"} only output supported PDF docinfo datetime values.}

\item{usetz}{Include the time zone in the formatting}

\item{format}{For \code{format_strftime()} see \code{\link[base:strptime]{base::strftime()}}.
For \code{format_nanotime()} see \url{https://github.com/google/cctz/blob/6e09ceb/include/time_zone.h#L197}.}

\item{tz}{A character string specifying the time zone to be used for the conversion.
Can be a length greater than one.}

\item{fill}{If timezone and UTC offset info is missing what
timezone to assume.  See \code{\link[=fill_tz]{fill_tz()}}.}
}
\value{
A character vector
}
\description{
\code{format()} returns a datetime string
with as much \strong{known} information possible (RFC 3339 with de facto standard time zone extension).
\code{format_iso8601()} returns an ISO 8601 datetime string.
\code{format_pdfmark()} returns a pdfmark datetime string with as much \strong{known} information possible.
\code{format_strftime()} allows \code{\link[base:strptime]{base::strftime()}} style formatting.
\code{format_nanotime()} allows CCTZ style formatting.
\code{format_edtf()} returns an Extended Date Time Format (EDTF) string.
\code{format_exiftool()} returns the date/time string expected by \code{exiftool}.
}
\examples{
  # ISO 8601 datetimes
  format_iso8601(as_datetimeoffset("2020-05"))
  format_iso8601(as_datetimeoffset("2020-05-10 20:15"))
  format_iso8601(as_datetimeoffset("2020-05-10 20:15:05-07"))
  if (requireNamespace("lubridate"))
    lubridate::format_ISO8601(as_datetimeoffset("2020-05-10 20:15:05-07"))

  # pdfmark datetimes
  format_pdfmark(as_datetimeoffset("2020-05"))
  format_pdfmark(as_datetimeoffset("2020-05-10 20:15"))
  format_pdfmark(as_datetimeoffset("2020-05-10 20:15:05-07"))

  # strftime style formatting
  dt <- as_datetimeoffset("2020-05-10 20:15")
  format_strftime(dt)
  format_strftime(dt, format = "\%c")

  # CCTZ style formatting
  if (requireNamespace("nanotime")) {
    dt <- as_datetimeoffset(Sys.time())
    format_nanotime(dt, format = "\%F \%H:\%M:\%E7S \%Ez") # SQL Server datetimeoffset
  }

  # EDTF style formatting
  format_edtf(as_datetimeoffset("2020-05"))
  format_edtf(as_datetimeoffset("2020-05-10T20:15:05-07"))
  dt <- datetimeoffset(2020, NA_integer_, 10)
  format_edtf(dt)

  # `exiftool` formatting
  format_exiftool(as_datetimeoffset("2020:05:10"))
  format_exiftool(as_datetimeoffset("2020:05:10 20:15"))
  format_exiftool(as_datetimeoffset("2020:05:10 20:15:05-07:00"))
}
