---
title: "Advanced Tuning for Models"
author: "Peter Hurford"
date: "2019-05-23"
output:
  rmarkdown::html_vignette:
    fig_caption: yes  
vignette: >
  %\VignetteIndexEntry{Advanced Tuning for Models}
  %\VignetteEngine{knitr::rmarkdown}
  %\VignetteEncoding{UTF-8}
---

DataRobot is very good at choosing optimal hyperparameters for models to maximize speed and accuracy. However, sometimes we wish to change those hyperparameters ourselves either because we know something DataRobot does not about how to maximize accuracy, we want to experiment with different approaches, or we have some other reason to use a particular parameter. To do this, we use the advanced tuning interface.


## Interactive Advanced Tuning Interface

The easiest way to do advanced tuning is to set up a model and use `RunInteractiveTuning` to start an interactive tuning job:


```r
tuningJobId <- RunInteractiveTuning(myModel)
```

This function will guide you through an interactive step-by-step process to tune each hyperparameter of your model. You will get to see the default, current value, and possible values for each choice and then iterate through them. You can simply press ENTER to skip any hyperparameter you wish to leave at its current value. Afterwards, you will receive a `jobId` value that you can use to get your model:


```r
tunedModel <- GetModelFromJobId(myModel$projectId, tuningJobId)
```

Note that occasionally the interactive tuning interface may duplicate parameters -- this is because the model itself has multiple parameters with the same name (e.g., parameters for one hot encoding text values followed by parameters for one hot encoding numeric variables will use the same names). They are listed in the order they are found in the blueprint but unfortunately more user-facing information cannot be provided.


## Get Data on Parameters Available for Tuning

If you wish to see the underlying data of which parameters are available for tuning for a model and what their default, current, and possible values are, you can turn to `GetTuningParameters`:


```r
parameters <- GetTuningParameters(myModel)
```

You can get more concisely printed tuning parameters by using `summary`:


```r
summary(GetTuningParameters(myModel))
```


## Programatic Tuning Interface

Lastly, while interactive tuning is useful, it may be needlessly slow to iterate through every parameter or you may wish to do tuning outside of an interactive mode. To do this, we also offer a programatic tuning interface using `StartTuningSession`:


```r
myXGBModel <- GetModel(projectId, modelId)
RunTune <- StartTuningSession(myXGBModel)
tuningJob <- RunTune(myXGBModel, colsample_bytree = 0.4, colsample_bylevel = 0.8)
tunedModel <- GetModelFromJobId(projectId, tuningJob)
```

`RunTune` is a function that is dynamically generated by `StartTuningSession` that then lets you fill in all the hyperparameters for your particular model. If you pass an XGB model, `RunTune` will have the hyperparameters to tune XGB (e.g., `colsample_bytree`), whereas if you pass an Elastic Net model, the function will instead have hyperparameters for Elastic Nets (e.g., `lambda`) instead.
