% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statistics.R
\name{profile}
\alias{profile}
\title{Profile-likelihood (PL) computation}
\usage{
profile(obj, pars, whichPar, alpha = 0.05, limits = c(lower = -Inf, upper =
  Inf), method = c("integrate", "optimize"), stepControl = NULL,
  algoControl = NULL, optControl = NULL, verbose = FALSE, ...)
}
\arguments{
\item{obj}{Objective function \code{obj(pars, fixed, ...)} returning a list with "value",
"gradient" and "hessian". If attribute "valueData" and/or "valuePrior are returned they are attached to the return value.}

\item{pars}{Parameter vector corresponding to the log-liklihood optimum.}

\item{whichPar}{Numeric or character. The parameter for which the profile is computed.}

\item{alpha}{Numeric, the significance level based on the chisquare distribution with df=1}

\item{limits}{Numeric vector of length 2, the lower and upper deviance from the original 
value of \code{pars[whichPar]}}

\item{method}{Character, either \code{"integrate"} or \code{"optimize"}. This is a short-cut for
setting stepControl, algoControl and optControl by hand.}

\item{stepControl}{List of arguments controlling the step adaption. Defaults to integration set-up, i.e.
\code{list(stepsize = 1e-4, min = 1e-4, max = Inf, atol = 1e-2, rtol = 1e-2, limit = 100)}}

\item{algoControl}{List of arguments controlling the fast PL algorithm. defaults to
\code{list(gamma = 1, W = "hessian", reoptimize = FALSE, correction = 1, reg = .Machine$double.eps)}}

\item{optControl}{List of arguments controlling the \code{trust()} optimizer. Defaults to
\code{list(rinit = .1, rmax = 10, iterlim = 10, fterm = sqrt(.Machine$double.eps), mterm = sqrt(.Machine$double.eps))}.
See \link{trust} for more details.}

\item{verbose}{Logical, print verbose messages.}

\item{...}{Arguments going to obj()}
}
\value{
Named list of length one. The name is the parameter name. The list enty is a
matrix with columns "value" (the objective value), "constraint" (deviation of the profiled paramter from
the original value), "stepsize" (the stepsize take for the iteration), "gamma" (the gamma value employed for the
iteration), "valueData" and "valuePrior" (if specified in obj), one column per parameter (the profile paths).
}
\description{
Profile-likelihood (PL) computation
}
\details{
Computation of the profile likelihood is based on the method of Lagrangian multipliers
and Euler integration of the corresponding differential equation of the profile likelihood paths.

\code{algoControl}: Since the Hessian which is needed for the differential equation is frequently misspecified, 
the error in integration needs to be compensated by a correction factor \code{gamma}. Instead of the
Hessian, an identity matrix can be used. To guarantee that the profile likelihood path stays on
the true path, each point proposed by the differential equation can be used as starting point for
an optimization run when \code{reoptimize = TRUE}. The correction factor \code{gamma} is adapted
based on the amount of actual correction. If this exceeds the value \code{correction}, \code{gamma} is
reduced. In some cases, the Hessian becomes singular. This leads to problems when inverting the
Hessian. To avoid this problem, the pseudoinverse is computed by removing all singular values lower
than \code{reg}.

\code{stepControl}: The Euler integration starts with \code{stepsize}. In each step the predicted change
of the objective function is compared with the actual change. If this is larger than \code{atol}, the
stepsize is reduced. For small deviations, either compared the abolute tolerance \code{atol} or the
relative tolerance \code{rtol}, the stepsize may be increased. \code{max} and \code{min} are upper and lower
bounds for \code{stepsize}. \code{limit} is the maximum number of steps that are take for the profile computation.
}
\examples{
## Parameter transformation
trafo <- eqnvec(a = "exp(loga)", 
                b = "exp(logb)", 
                c = "exp(loga)*exp(logb)*exp(logc)")
p <- P(trafo)

## Objective function
obj1 <- constraintL2(mu = c(a = .1, b = 1, c = 10), sigma = .6)
obj2 <- constraintL2(mu = c(loga = 0, logb = 0), sigma = 10)
obj <- obj1*p + obj2

## Initialize parameters and obtain fit
pars <- c(loga = 1, logb = 1, logc = 1)
myfit <- trust(obj, pars, rinit = 1, rmax = 10)
myfit.fixed <- trust(obj, pars[-1], rinit = 1, rmax = 10, fixed = pars[1])

## Compute profiles by integration method
profiles.approx <- do.call(
  rbind, 
  lapply(1:3, function(i) {
    profile(obj, myfit$argument, whichPar = i, limits = c(-10, 10),
            method = "integrate")
  })
)

## Compute profiles by repeated optimization 
profiles.exact <- do.call(
  rbind, 
  lapply(1:3, function(i) {
    profile(obj, myfit$argument, whichPar = i, limits = c(-10, 10),
            method = "optimize")
  })
)

## Compute profiles for fit with fixed element by integration method
profiles.approx.fixed <- do.call(
  rbind, 
  lapply(1:2, function(i) {
    profile(obj, myfit.fixed$argument, whichPar = i, limits = c(-10, 10),
            method = "integrate",
            fixed = pars[1])
  })
)

## Plotting
plotProfile(profiles.approx)
plotProfile(list(profiles.approx, profiles.exact))
plotProfile(list(profiles.approx, profiles.approx.fixed))

plotPaths(profiles.approx, sort = TRUE)
plotPaths(profiles.approx, whichPar = "logc")
plotPaths(list(profiles.approx, profiles.approx.fixed), whichPar = "logc")
}

