% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/preference_order.R
\name{preference_order}
\alias{preference_order}
\title{Compute the preference order for predictors based on a user-defined function.}
\usage{
preference_order(
  df = NULL,
  response = NULL,
  predictors = NULL,
  f = f_rsquared,
  encoding_method = "mean",
  workers = 1
)
}
\arguments{
\item{df}{(required; data frame) A data frame with numeric and/or character predictors predictors, and optionally, a response variable. Default: NULL.}

\item{response}{(required, character string) Name of a numeric response variable. Character response variables are ignored. Please, see 'Details' to better understand how providing this argument or not leads to different results when there are character variables in 'predictors'. Default: NULL.}

\item{predictors}{(optional; character vector) character vector with predictor names in 'df'. If omitted, all columns of 'df' are used as predictors. Default:'NULL'}

\item{f}{(optional: function) A function that returns a value representing the relationship between a given predictor and the response. Higher values are ranked higher. The available options are:
\itemize{
\item \code{\link[=f_rsquared]{f_rsquared()}} (default option): returns the R-squared of the correlation between a numeric response and a numeric predictor.
\item f_gam_deviance: fits a univariate GAM model between a numeric response and a numeric predictor to return the explained deviance. Requires the package \code{mgcv}.
\item \code{\link[=f_rf_rsquared]{f_rf_rsquared()}} also named \code{\link[=f_rf_deviance]{f_rf_deviance()}}: fits a univariate random forest model with \code{ranger::ranger()} between a numeric response and a numeric predictor to return the R-squared of the observations against the out-of-bag predictions. Requires the package \code{ranger}.
\item \code{\link[=f_logistic_auc_balanced]{f_logistic_auc_balanced()}}: fits a logistic univariate GLM of a balanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
\item \code{\link[=f_logistic_auc_unbalanced]{f_logistic_auc_unbalanced()}}: fits a quasibinomial univariate GLM with weighted cases of an unbalanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
\item \code{\link[=f_gam_auc_balanced]{f_gam_auc_balanced()}}: fits a logistic univariate GAM of a balanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
\item \code{\link[=f_gam_auc_unbalanced]{f_gam_auc_unbalanced()}}: fits a quasibinomial univariate GAM with weighted cases of an unbalanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
\item \code{\link[=f_rf_auc_balanced]{f_rf_auc_balanced()}}: fits a random forest model of a balanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
\item \code{\link[=f_rf_auc_unbalanced]{f_rf_auc_unbalanced()}}: fits a random forest model with weighted cases of an unbalanced binary response (0s and 1s) against a numeric predictor to return the Area Under the Curve of the observations against the predictors.
}}

\item{encoding_method}{(optional; character string). Name of the target encoding method to convert character and factor predictors to numeric. One of "mean", "rank", "loo", "rnorm" (see \code{\link[=target_encoding_lab]{target_encoding_lab()}} for further details). Default: "mean"}

\item{workers}{(integer) number of workers for parallel execution. Default: 1}
}
\value{
A data frame with the columns "predictor" and "value". The former contains the predictors names in order, ready for the argument \code{preference_order} in \code{\link[=cor_select]{cor_select()}}, \code{\link[=vif_select]{vif_select()}} and \code{\link[=collinear]{collinear()}}. The latter contains the result of the function \code{f} for each combination of predictor and response.
}
\description{
This function calculates the preference order of predictors based on a user-provided function that takes a predictor, a response, and a data frame as arguments.
}
\examples{

data(
  vi,
  vi_predictors
)

#subset to limit example run time
vi <- vi[1:1000, ]

#computing preference order
#with response
#numeric and categorical predictors in the output
#as the R-squared between each predictor and the response
preference.order <- preference_order(
  df = vi,
  response = "vi_mean",
  predictors = vi_predictors,
  f = f_rsquared,
  workers = 1
  )

preference.order

#using it in variable selection with collinear()
selected.predictors <- cor_select(
  df = vi,
  response = "vi_mean", #don't forget the response!
  predictors = vi_predictors,
  preference_order = preference.order,
  max_cor = 0.75
  )

selected.predictors

#check their correlations
selected.predictors.cor <- cor_df(
  df = vi,
  response = "vi_mean",
  predictors = selected.predictors
)

#all correlations below max_cor
selected.predictors.cor

#USING A CUSTOM FUNCTION
#custom function to compute RMSE between a predictor and a response
#x is a predictor name
#y is a response name
#df is a data frame with multiple predictors and one response
#must return a single number, with higher number indicating higher preference
#notice we use "one minus RMSE" to give higher rank to variables with lower RMSE
f_rmse <- function(x, y, df){

  xy <- df[, c(x, y)] |>
    na.omit() |>
    scale()

  1 - sqrt(mean((xy[, 1] - xy[, 2])^2))

}

preference.order <- preference_order(
  df = vi,
  response = "vi_mean",
  predictors = vi_predictors,
  f = f_rmse,
  workers = 1
)

preference.order

}
\author{
Blas M. Benito
}
