% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ORII.R
\name{ineffactorOR2}
\alias{ineffactorOR2}
\title{Inefficiency factor in the OR2 model}
\usage{
ineffactorOR2(x, betadraws, sigmadraws, accutoff, maxlags, verbose)
}
\arguments{
\item{x}{covariate matrix of size \eqn{(n x k)} including a column of ones with or without column names.
This input is used to extract column names, if available, but not used in calculation.}

\item{betadraws}{dataframe of the Gibbs draws of \eqn{\beta}, size \eqn{(k x nsim)}.}

\item{sigmadraws}{dataframe of the Gibbs draws of \eqn{\sigma}, size \eqn{(1 x nsim)}.}

\item{accutoff}{cut-off to identify the number of lags and form batches, default is 0.05.}

\item{maxlags}{maximum lag at which to calculate the acf, default is 400.}

\item{verbose}{whether to print the final output and provide additional information or not, default is TRUE.}
}
\value{
Returns a column vector of inefficiency factors for each component of \eqn{\beta} and \eqn{\sigma}.
}
\description{
This function calculates the inefficiency factor from the MCMC draws
of \eqn{(\beta, \sigma)} in the OR2 model (ordinal quantile model with exactly 3 outcomes). The
inefficiency factor is calculated using the batch-means method.
}
\details{
Calculates the inefficiency factor of \eqn{(\beta, \sigma)} using the batch-means
method based on the Gibbs draws. Inefficiency factor can be interpreted as the cost of
working with correlated draws. A low inefficiency factor indicates better mixing
and an efficient algorithm.
}
\examples{
set.seed(101)
data("data25j3")
y <- data25j3$y
xMat <- data25j3$x
k <- dim(xMat)[2]
b0 <- array(rep(0, k), dim = c(k, 1))
B0 <- 10*diag(k)
n0 <- 5
d0 <- 8
output <- quantregOR2(y = y, x = xMat, b0, B0, n0, d0, gammacp2 = 3,
burn = 10, mcmc = 40, p = 0.25, accutoff = 0.5, maxlags = 400, verbose = FALSE)
betadraws <- output$betadraws
sigmadraws <- output$sigmadraws

inefficiency <- ineffactorOR2(xMat, betadraws, sigmadraws, 0.5, 400, TRUE)

# Summary of Inefficiency Factor:
#            Inef Factor
# beta_1       1.5686
# beta_2       1.5240
# beta_3       1.4807
# sigma        2.4228

}
\references{
Greenberg, E. (2012). `"Introduction to Bayesian Econometrics."` Cambridge University
Press, Cambridge. DOI: 10.1017/CBO9780511808920

Chib, S. (2012), `"Introduction to simulation and MCMC methods."` In Geweke J., Koop G., and Dijk, H.V.,
editors, `"The Oxford Handbook of Bayesian Econometrics"`, pages 183--218. Oxford University Press,
Oxford. DOI: 10.1093/oxfordhb/9780199559084.013.0006
}
\seealso{
pracma, \link[stats]{acf}
}
