% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/draw_all_admix.R
\name{draw_all_admix}
\alias{draw_all_admix}
\title{Simulate random allele frequencies and genotypes from the BN-PSD admixture model}
\usage{
draw_all_admix(admix_proportions, inbr_subpops, m_loci,
  want_genotypes = TRUE, want_p_ind = FALSE, want_p_subpops = FALSE,
  want_p_anc = TRUE, low_mem = FALSE, verbose = FALSE,
  require_polymorphic_loci = TRUE)
}
\arguments{
\item{admix_proportions}{The \eqn{n \times k}{n-by-k} matrix of admixture proportions.}

\item{inbr_subpops}{The length-\eqn{k} vector (or scalar) of intermediate subpopulation \eqn{F_{ST}}{FST} values.}

\item{m_loci}{The number of loci to draw.}

\item{want_genotypes}{If TRUE (default), includes the matrix of random genotypes in the return list.}

\item{want_p_ind}{If TRUE (NOT default), includes the matrix of individual-specific allele frequencies in the return list.}

\item{want_p_subpops}{If TRUE (NOT default), includes the matrix of random intermediate subpopulation allele frequencies in the return list.}

\item{want_p_anc}{If TRUE (default), includes the matrix of random ancestral allele frequencies in the return list.}

\item{low_mem}{If TRUE, uses a low-memory algorithm to raw genotypes without storing or returning the corresponding `p_ind` matrix.}

\item{verbose}{If TRUE, prints messages for every stage in the algorithm.}

\item{require_polymorphic_loci}{If TRUE (default), returned genotype matrix will not include any fixed loci (loci that happened to be fixed are drawn again, starting from their ancestral allele frequencies, and checked iteratively until no fixed loci remain, so that the final number of polymorphic loci is exactly \eqn{m_loci}).}
}
\value{
A named list that includes the following randomly-generated data in this order:
\describe{
  \item{X:}{
    An \eqn{m \times n}{m-by-n} matrix of genotypes.
    Included if \code{want_genotypes = TRUE}.
  }
  \item{p_anc:}{
    A length-\eqn{m} vector of ancestral allele frequencies.
    Included if \code{want_p_anc = TRUE}.
  }
  \item{p_subpops:}{
    An \eqn{m \times k}{m-by-k} matrix of intermediate subpopulation allele frequencies
    Included if \code{want_p_subpops = TRUE}.
  }
  \item{p_ind:}{
    An \eqn{m \times n}{m-by-n} matrix of individual-specific allele frequencies.
    Included only if both \code{want_p_ind = TRUE} and \code{low_mem = FALSE}.
  }
}
}
\description{
This function returns simulated ancestral, intermediate, and individual-specific allele frequencies and genotypes given the admixture structure, as determined by the admixture proportions and the vector of intermediate subpopulation \eqn{F_{ST}}{FST} values.
The function is a wrapper around \code{\link{draw_p_anc}}, \code{\link{draw_p_subpops}}, \code{\link{make_p_ind_admix}}, and \code{\link{draw_genotypes_admix}} with additional features such as requiring polymorphic loci.
Importantly, by default fixed loci are re-drawn from the start (starting from the ancestral allele frequencies) so no fixed loci are in the output and no biases are introduced by re-drawing genotypes conditional on any of the previous allele frequencies (ancestral, intermediate, or individual-specific).
Below \eqn{m} is the number of loci, \eqn{n} is the number of individuals, and \eqn{k} is the number of intermediate subpopulations.
}
\examples{
# dimensions
# number of loci
m_loci <- 10
# number of individuals
n_ind <- 5
# number of intermediate subpops
k_subpops <- 2

# define population structure
# FST values for k = 2 subpopulations
inbr_subpops <- c(0.1, 0.3)
# admixture proportions from 1D geography
admix_proportions <- admix_prop_1d_linear(n_ind, k_subpops, sigma = 1)

# draw all random allele freqs and genotypes
out <- draw_all_admix(admix_proportions, inbr_subpops, m_loci)

# return value is a list with these items:

# genotypes
X <- out$X

# ancestral AFs
p_anc <- out$p_anc

# # these are excluded by default, but would be included if ...
# # ... `want_p_subpops == TRUE`
# # intermediate subpopulation AFs
# p_subpops <- out$p_subpops
# 
# # ... `want_p_ind == TRUE` and `low_mem = FALSE`
# # individual-specific AFs
# p_ind <- out$p_ind

}
