\name{pointsOnBezier}
\alias{pointsOnBezier}
\title{Generates points along a Bezier curve or spline}
\description{
This function provides two different functionalities for generating points along a Bezier curve or spline. The first generates approximately evenly spaced points along a Bezier, optimizing point position according to specified convergence criteria. A second functionality places points along a Bezier such that the distance between consecutive points does not exceed a specified Euclidean distance. This second functionality does not generate evenly spaced points along the curve, instead providing a more rapid routine for generating a large number of points on a Bezier more evenly spaced than with parametric point generation.
}
\usage{
pointsOnBezier(p, n = NULL, t1 = 0, t2 = NULL, deg = NULL, 
               max.dist = NULL, max.dist.factor = 0.1, 
               relative.min.slope = 1e-7, absolute.min.slope = 0, 
               sub.relative.min.slope = 1e-4, sub.absolute.min.slope = 0, 
               print.progress = FALSE)
}
\arguments{
  \item{p}{control points, input either as vector, matrix or list (see \code{\link{bezier}}).}
  \item{n}{the number of points to generate along the Bezier. Ignored if \code{max.dist} is non-\code{NULL} (see Details).}
  \item{t1}{a parametric value for a Bezier curve or spline at which the points will start.}
  \item{t2}{a parametric value for a Bezier curve or spline at which the points will end. Default is the end of the Bezier curve or spline.}
  \item{deg}{a numeric indicating the degree (or order) of a Bezier spline. For Bezier curves, the degree is computed based on the number of control points.}
  \item{max.dist}{the maximum Euclidean distance (not distance along the curve) between consecutive points for the more rapid routine.}
  \item{max.dist.factor}{a factor used to approximate point position based on maximum distance criteria (see Details). Ignored if \code{max.dist} is \code{NULL}.}
  \item{relative.min.slope}{parameter passed to \code{\link{bezierArcLength}} for estimating total arc length. Ignored if \code{max.dist} is non-\code{NULL}.}
  \item{absolute.min.slope}{parameter passed to \code{\link{bezierArcLength}} for estimating total arc length. Ignored if \code{max.dist} is non-\code{NULL}.}
  \item{sub.relative.min.slope}{parameter passed to \code{\link{compareBezierArcLength}} for estimating total arc length (see \code{\link{compareBezierArcLength}}). Ignored if \code{max.dist} is non-\code{NULL}.}
  \item{sub.absolute.min.slope}{parameter passed to \code{\link{compareBezierArcLength}} for estimating total arc length (see \code{\link{compareBezierArcLength}}). Ignored if \code{max.dist} is non-\code{NULL}.}
  \item{print.progress}{logical indicating whether iterations should be printed for tracking function progress.}
}
\details{
Points can easily be generated along a Bezier curve or spline using parametric values (provided by the function \code{\link{bezier}}), however these points are not evenly spaced along the curve. Points generated by parametric values will be closer together in regions with the highest curvature and furthest apart in regions that approach a straight line. This function provides two different functionalities for generating points along a Bezier curve or spline that are more evenly spaced than those generated using parametric values. The first generates \code{n} approximately evenly spaced points along a Bezier, optimizing point position according to specified convergence criteria. A second functionality places points along a Bezier such that the distance between consecutive points does not exceed a specified Euclidean distance (\code{max.dist}).

The input of the control points \code{p} is identical to \code{\link{bezier}} and can be a vector, matrix or list (see Details in \code{\link{bezier}}). As with \code{\link{bezier}}, when control points are input as a list and the number of control points differs for different dimensions, the degree will be elevated so that all dimensions are of uniform degree (see \code{\link{elevateBezierDegree}}). \code{t1} and \code{t2} are parametric values along the Bezier curve or spline between which points will be generated. The default values for \code{t1} and \code{t2} are the start and end points of the Bezier curve or spline, respectively. For a Bezier spline, if \code{t2} is not specified, it is calculated based on the number of control points and the degree (\code{deg}). When using \code{pointsOnBezier} for Bezier splines, \code{deg} must be specified or else the points will be treated as a single Bezier curve.

If \code{n} is non-\code{NULL}, \code{pointsOnBezier} generates \code{n} evenly spaced points along a Bezier curve or spline. This requires accurate approximation of Bezier arc length. An initial estimation of the total arc length between \code{t1} and \code{t2} is made (using \code{\link{bezierArcLength}}) to determine the interval at which points should be placed to equally subdivide the curve. \code{optim} is used to find the optimal position of each point, calling \code{\link{bezierArcLength}} via \code{\link{compareBezierArcLength}}, such that the arc length between points is nearly equal to this interval. When positioning each point, the arc length is estimated from \code{t1} (rather than from the previous point) so that errors are not compounded. As a consequence of repeated calls to \code{optim} and \code{\link{bezierArcLength}}, this functionality can be rather slow.

The parameters ending in \code{min.slope} are convergence criteria passed to \code{\link{bezierArcLength}}. The parameters \code{relative.min.slope} and \code{absolute.min.slope} are the criteria used in the initial arc length estimation, while \code{sub.relative.min.slope} and \code{sub.absolute.min.slope} are the criteria used to estimate arc length in placing each point along the curve. Larger convergence criteria values will cause \code{pointsOnBezier} to run faster but at lower accuracy. For a complete description of the convergence criteria, see Details in \code{\link{bezierArcLength}}.

\code{pointsOnBezier} runs an alternative routine when \code{max.dist} is non-\code{NULL}. In this case, \code{n} and the convergence criteria are ignored, \code{\link{bezierArcLength}} is not called and \code{pointsOnBezier} generates points along a Bezier such that the distance between consecutive points does not exceed the specified Euclidean distance \code{max.dist}. The parameter \code{max.dist.factor} is a factor that is used to iteratively increase the parametric value to reach the next point without exceeding \code{max.dist}. The lower \code{max.dist.factor} is, the closer the interpoint Euclidean distance will be to \code{max.dist} but the longer \code{pointsOnBezier} will take to run (see Examples). If \code{max.dist} does not evenly divide the total arc length between \code{t1} and \code{t2}, the interval between the second-to-last point and the end point may not be close to \code{max.dist}. If \code{max.dist} evenly divides the arc length, if \code{max.dist.factor} is low and if \code{max.dist} is small, the points will be more evenly spaced than with parametric point generation.

In the case of Bezier splines, note that borders between spline segments are not respected and arc lengths are calculated across spline segments. In order to generate points within spline segments, \code{pointsOnBezier} should be called separately for each segment (see Examples).
}
\value{
a list with the following elements:
  \item{points}{evenly spaced points along a Bezier curve or spline.}
  \item{error}{an vector of the error for each point along the curve or spline. If \code{max.dist} is \code{NULL}, this is the \code{value} output from \code{optim} for each point estimation. If \code{max.dist} is non-\code{NULL}, this is \code{max.dist} minus the actual Euclidean distance between consecutive points.}
  \item{t}{the parametric values corresponding to each point in \code{points}.}
}
\author{Aaron Olsen}
\seealso{
\code{\link{bezier}}, \code{\link{bezierArcLength}}, \code{\link{compareBezierArcLength}}, \code{\link{elevateBezierDegree}}
}
\examples{

## EVENLY SPACED POINTS ALONG A BEZIER CURVE ##
## BEZIER CURVE CONTROL POINTS
p <- matrix(c(3,2, 3,0, 5,5), nrow=3, ncol=2, byrow=TRUE)

## GET PARAMETRIC BEZIER POINTS
bp <- bezier(t=seq(0, 1, length=100), p=p)

## PLOT PARAMETRIC BEZIER POINTS
## NOTE THAT THEY ARE NOT EVENLY SPACED ALONG THE CURVE
plot(bp, cex=0.5)

## GET EVENLY SPACED POINTS ALONG CURVE
pob <- pointsOnBezier(p=p, n=10, print.progress=TRUE)

## ADD POINTS TO PLOT
## NOTE THAT THESE POINTS ARE EVENLY SPACED ALONG CURVE
points(pob$points, col='red')

## FUNCTION WILL RUN FASTER BY INCREASING CONVERGENCE CRITERIA
pob_faster <- pointsOnBezier(p=p, n=10, sub.relative.min.slope=1e-2, print.progress=TRUE)

## SOME DEVIATION BUT POINTS ARE NEARLY IDENTICAL
points(pob_faster$points, col='blue', cex=1.5)
}

\keyword{ bezier }
\keyword{ arc length }