# ----- File serving -----------------------------------------------------------

#' @export
#' @title File-serving middleware
#'
#' @description Binds to GET requests that aren't handled by specified paths.
#' The result is to return files that are found on the host machine at the
#' requested path. Binary file types like \code{.png}, \code{.gif} or
#' \code{.pdf} are returned as raw bytes. All others are returned as characters.
#'
#' Mime types are guessed using the \pkg{mime} package. The \code{rawTypesPattern}
#' parameter is used to match mime types that should be returned as raw bytes.
#'
#' @details All files to be served in this manner must exist underneath the
#' host machine directory specified with \code{rootPath}. The directory
#' structure underneath \code{rootPath} will be mapped onto URLs underneath
#' \code{urlPath}. This helps when deploying web services at preordained URLs.
#'
#' For example, specifying:
#'
#' \preformatted{
#' beakr %>%
#'   ...
#'   serverStaticFiles("/data-service/A", "/data/project_A/public")
#'   ...
#'   listen()
#' }
#'
#' will allow a file found on the host machine at:
#' \preformatted{
#'   /data/project_A/public/2019/averages.csv"
#' }
#' to be downloaded from:
#' \preformatted{
#'   http://<ip-address>:<port>/data-service/A/2019/averages.csv"
#' }
#'
#' @param beakr \code{Beakr} instance or \code{NULL}.
#' @param urlPath String representing the URL directory undnerneath which static
#' file paths will appear.
#' @param rootPath String representing the absolute path used as the root
#' directory when searching for files on host machine. Defaults to the directory
#' in which the script is running.
#' @param rawTypesPattern String pattern identifying mime types to be returned
#' as raw bytes.
#'
#' @return A \code{Beakr} instance with added middleware.

serveStaticFiles <- function(
  beakr = NULL,
  urlPath = NULL,
  rootPath = getwd(),
  rawTypesPattern = "image|json|octet|pdf|video"
) {

  # ----- Validate parameters --------------------------------------------------

  if ( is.null(beakr) )
    stop("'beakr' is not defined")

  if ( is.null(rootPath) )
    stop("'rootPath' is not defined")

  # Guarantee that urlPath exists and begins with "/"
  if ( is.null(urlPath) ) {
    urlPath <- "/"
  } else if ( substr(urlPath, 1, 1) != "/" ) {
    urlPath <- paste0("/", urlPath)
  }

  if ( is.null(rawTypesPattern) )
    rawTypesPattern <- "image|json|octet|pdf|video"

  # ----- Create middleware function -------------------------------------------

  # # == BEGIN debugging, requires MazamaCoreUtils logging to be set up ==
  #
  # if ( !is.null(urlPath) )
  #   logger.trace("urlPath = %s", urlPath)
  #
  # if ( !is.null(rootPath) )
  #   logger.trace("rootPath = %s", rootPath)
  #
  # # == END debugging ==

  beakr <-
    httpGET(
      beakr = beakr,
      path = NULL,
      FUN = function(req, res, err) {

        # Return Not Found if req$path doesn't inlcude urlPath
        if ( !stringr::str_detect(req$path, urlPath) ) {
          res$setStatus(404L)
          return(NULL)
        }

        # Get the URL-relative path
        relativePath <- stringr::str_replace(req$path, urlPath, "")
        relativePath <- stringr::str_replace(relativePath, "^/", "")
        filePath <- file.path(rootPath, relativePath)

        if ( !file.exists(filePath) ) {

          res$setStatus(404L)
          return(NULL)

        } else {

          # Set the response content type
          mimeType <- mime::guess_type(filePath)
          res$setContentType(mimeType)

          # Read raw bytes no matter what the mimeType
          data <- readBin( con  = filePath,
                           what = "raw",
                           n    = file.info(filePath)$size )

          # Return either raw bytes or characters depending on the mimeType
          if ( grepl(rawTypesPattern, mimeType) ) {
            return(data)
          } else {
            return(rawToChar(data))
          }

        }

      } # END FUN
    ) # END httpGET()

  return(beakr)

}


# ----- Error handling ---------------------------------------------------------

#' @export
#' @title Error-handling middleware
#'
#' @description This default error handler should be added
#' at the end of the beakr pipeline, right before \code{listen()}. Errors
#' generated by any previous step will be returned within a JSON wrapper.
#'
#' The general structure for a stand-alone executable script with a
#' \pkg{beakr} webservice typically looks like this:
#'
#' \preformatted{
#' newBeakr() %>%
#'
#'   httpGET(<route_A>, function(req, res, err) {
#'     ...
#'   }) %>%
#'
#'   httpGET(<route_B>, function(req, res, err) {
#'     ...
#'   }) %>%
#'
#'   serveStaticFiles(...) %>%
#'
#'   handleErrors() %>%
#'
#'   listen()
#' }
#'
#' @param beakr Beakr instance
#'
#' @return A \code{Beakr} instance with added middleware.
#'

handleErrors <- function(
  beakr = NULL
) {

  # TODO:  Could support FUN = NULL in function signature so other error
  # TODO:  functions could be supplied.

  if ( is.null(beakr) )
    stop("'beakr' is not defined")

  beakr <-
    .routeMiddleware(
      beakr = beakr,
      FUN = jsonError,
      path = NULL,
      method = NULL,
      websocket = FALSE
    )

  return(beakr)

}


#' @export
#' @title JSON error function
#'
#' @description This function is used to add a JSON error response to the
#' \code{res} object. It is called by the \code{handleErrors()} utility
#' function.
#'
#' @param req \code{Request} object.
#' @param res \code{Response} object.
#' @param err \code{Error} Error object.
#'
#' @return The incoming \code{res} object is modified.
#'
#' @seealso \link{Request}, \link{Response}, \link{Error}

jsonError <- function(req, res, err) {

  res$setContentType("application/json")

  if ( err$occurred ) {
    res$status <- 500L
    error_str <- paste(err$errors, collapse = "\n")

    cat("ERROR:\n", error_str, "\n")

    res$json(list( status = "error",
                   status_code = 500L,
                   errors = error_str ))

  } else {
    res$status = 404L
    res$json(list( status = "Page not found.",
                   status_code = 404L ))
  }

}
