% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bamlss.R
\name{BAMLSS}
\alias{BAMLSS}
\alias{family.BAMLSS}
\alias{mean.BAMLSS}
\alias{variance.BAMLSS}
\alias{skewness.BAMLSS}
\alias{kurtosis.BAMLSS}
\alias{random.BAMLSS}
\alias{pdf.BAMLSS}
\alias{log_pdf.BAMLSS}
\alias{cdf.BAMLSS}
\alias{quantile.BAMLSS}
\alias{support.BAMLSS}
\alias{is_discrete.BAMLSS}
\alias{is_continuous.BAMLSS}
\alias{format.BAMLSS}
\alias{print.BAMLSS}
\title{Create \pkg{distributions3} Object}
\usage{
BAMLSS(family, ...)
}
\arguments{
\item{family}{object. BAMLSS family specifications recognized by
\code{\link[bamlss]{bamlss.family}}, including \code{family.bamlss} objects,
family-generating functions (e.g., \code{\link[bamlss]{gaussian_bamlss}}),
or characters with family names (e.g., \code{"gaussian"} or \code{"binomial"}).}

\item{\dots}{further arguments passed as parameters to the BAMLSS family.
Can be scalars or vectors.}
}
\value{
A \code{BAMLSS} distribution object.
}
\description{
A single class and corresponding methods encompassing all \code{bamlss.family}
distributions (from the \pkg{bamlss} package) using the workflow from the
\pkg{distributions3} package.
}
\details{
The constructor function \code{BAMLSS} sets up a distribution
object, representing a distribution from the BAMLSS (Bayesian additive
model of location, scale, and shape) framework by the corresponding parameters
plus a \code{family} attribute, e.g., \code{\link[bamlss]{gaussian_bamlss}} for the
normal distribution or \code{\link[bamlss]{binomial_bamlss}} for the binomial
distribution. The parameters employed by the family vary across the families
but typically capture different distributional properties (like location, scale,
shape, etc.).

All parameters can also be vectors, so that it is possible to define a vector
of BAMLSS distributions from the same family with potentially different parameters.
All parameters need to have the same length or must be scalars (i.e.,
of length 1) which are then recycled to the length of the other parameters.

For the \code{BAMLSS} distribution objects there is a wide range
of standard methods available to the generics provided in the \pkg{distributions3}
package: \code{\link[distributions3]{pdf}} and \code{\link[distributions3]{log_pdf}}
for the (log-)density (PDF), \code{\link[distributions3]{cdf}} for the probability
from the cumulative distribution function (CDF), \code{quantile} for quantiles,
\code{\link[distributions3]{random}} for simulating random variables,
and \code{\link[distributions3]{support}} for the support interval
(minimum and maximum). Internally, these methods rely on the usual d/p/q/r
functions provided in \pkg{bamlss}, see the manual pages of the individual
families. The methods \code{\link[distributions3]{is_discrete}} and
\code{\link[distributions3]{is_continuous}} can be used to query whether the
distributions are discrete on the entire support or continuous on the entire
support, respectively.

See the examples below for an illustration of the workflow for the class and methods.
}
\examples{
\dontshow{ if(!requireNamespace("bamlss")) {
  if(interactive() || is.na(Sys.getenv("_R_CHECK_PACKAGE_NAME_", NA))) {
    stop("not all packages required for the example are installed")
  } else q() }
}
## package and random seed
library("distributions3")
set.seed(6020)

## three Weibull distributions
X <- BAMLSS("weibull", lambda = c(1, 1, 2), alpha = c(1, 2, 2))
X

## moments (FIXME: mean and variance not provided by weibull_bamlss)
## mean(X)
## variance(X)

## support interval (minimum and maximum)
support(X)
is_discrete(X)
is_continuous(X)

## simulate random variables
random(X, 5)

## histograms of 1,000 simulated observations
x <- random(X, 1000)
hist(x[1, ], main = "Weibull(1,1)")
hist(x[2, ], main = "Weibull(1,2)")
hist(x[3, ], main = "Weibull(2,2)")

## probability density function (PDF) and log-density (or log-likelihood)
x <- c(2, 2, 1)
pdf(X, x)
pdf(X, x, log = TRUE)
log_pdf(X, x)

## cumulative distribution function (CDF)
cdf(X, x)

## quantiles
quantile(X, 0.5)

## cdf() and quantile() are inverses
cdf(X, quantile(X, 0.5))
quantile(X, cdf(X, 1))

## all methods above can either be applied elementwise or for
## all combinations of X and x, if length(X) = length(x),
## also the result can be assured to be a matrix via drop = FALSE
p <- c(0.05, 0.5, 0.95)
quantile(X, p, elementwise = FALSE)
quantile(X, p, elementwise = TRUE)
quantile(X, p, elementwise = TRUE, drop = FALSE)

## compare theoretical and empirical mean from 1,000 simulated observations
## (FIXME: mean not provided by weibull_bamlss)
## cbind(
##   "theoretical" = mean(X),
##   "empirical" = rowMeans(random(X, 1000))
## )
}
\seealso{
\code{\link[bamlss]{bamlss.family}}
}
