% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bage_mod-methods.R
\name{forecast.bage_mod}
\alias{forecast.bage_mod}
\title{Use a Model to Make a Forecast}
\usage{
\method{forecast}{bage_mod}(
  object,
  newdata = NULL,
  labels = NULL,
  output = c("augment", "components"),
  include_estimates = FALSE,
  ...
)
}
\arguments{
\item{object}{A \code{bage_mod} object,
typically created with \code{\link[=mod_pois]{mod_pois()}},
\code{\link[=mod_binom]{mod_binom()}}, or \code{\link[=mod_norm]{mod_norm()}}.}

\item{newdata}{Data frame with data for
future periods.}

\item{labels}{Labels for future values.}

\item{output}{Type of output returned}

\item{include_estimates}{Whether to
include historical estimates along
with the forecasts. Default is \code{FALSE}.}

\item{...}{Not currently used.}
}
\value{
A \link[tibble:tibble-package]{tibble}.
}
\description{
Forecast rates, probabilities, means, and
other model parameters.
}
\section{How the forecasts are constructed}{


Internally, the steps involved in a forecast are:
\enumerate{
\item Forecast time-varying main effects and interactions,
e.g. a time main effect, or an age-time interaction.
\item Combine forecasts for the time-varying main effects and
interactions with non-time-varying parameters, e.g.
age effects or dispersion.
\item Use the combined parameters to generate values for
rates, probabilities or means.
\item Optionally, generate values for the outcome variable.
}

\code{forecast()} generates values for the outcome variable when,
\itemize{
\item \code{output} is \code{"augment"},
\item a value has been supplied for \code{newdata},
\item \code{newdata} included a value for the exposure,
size, or weights variable (except if \code{exposure = 1}
or \code{weights = 1} in the original call to
\code{\link[=mod_pois]{mod_pois()}} or \code{\link[=mod_norm]{mod_norm()}}).
}

\href{https://bayesiandemography.github.io/bage/articles/vig2_math.html}{Mathematical Details}
gives more details on the internal calculations in forecasting.
}

\section{Output format}{


When \code{output} is \code{"augment"} (the default),
the return value from \code{forecast()}
looks like output from function \code{\link[=augment]{augment()}}. When \code{output} is
\code{"components"}, the return value looks like output
from \code{\link[=components]{components()}}.

When \code{include_estimates} is \code{FALSE} (the default),
the output of \code{forecast()} excludes values for
time-varying parameters for the period covered by the data.
When \code{include_estimates} is \code{TRUE}, the output
includes these values.
Setting \code{include_estimates} to \code{TRUE} can be helpful
when creating graphs that combine estimates and forecasts.
}

\section{Forecasting with covariates}{


Models that contain \link[=set_covariates]{covariates} can be used
in forecasts, provided that
\itemize{
\item all coefficients (the \eqn{\zeta_p}) are estimated
from historical data via \code{\link[=fit]{fit()}}, and
\item if any covariates (the columns of \eqn{\pmb{Z}})
are time-varying, then future values for these
covariates are supplied via the \code{newdata} argument.
}
}

\section{Fitted and unfitted models}{


\code{forecast()} is typically used with a
\link[=fit]{fitted} model, i.e. a model in which parameter
values have been estimated from the data.
The resulting forecasts reflect data and priors.

\code{forecast()} can, however, be used with an
unfitted model. In this case, the forecasts
are based entirely on the priors. See below for
an example. Experimenting with forecasts
based entirely on the priors can be helpful for
choosing an appropriate model.
}

\section{Warning}{


The interface for \code{forecast()} has not been finalised.
}

\examples{
## specify and fit model
mod <- mod_pois(injuries ~ age * sex + ethnicity + year,
                data = nzl_injuries,
                exposure = popn) |>
  fit()
mod

## forecasts
mod |>
  forecast(labels = 2019:2024)

## combined estimates and forecasts
mod |>
  forecast(labels = 2019:2024,
           include_estimates = TRUE)

## hyper-parameters
mod |>
  forecast(labels = 2019:2024,
           output = "components")

## hold back some data and forecast
library(dplyr, warn.conflicts = FALSE)
data_historical <- nzl_injuries |>
  filter(year <= 2015)
data_forecast <- nzl_injuries |>
  filter(year > 2015) |>
  mutate(injuries = NA)
mod_pois(injuries ~ age * sex + ethnicity + year,
         data = data_historical,
         exposure = popn) |>
  fit() |>
  forecast(newdata = data_forecast)

## forecast using GDP per capita in 2023 as a covariate
mod_births <- mod_pois(births ~ age * region + time,
                       data = kor_births,
                       exposure = popn) |>
  set_covariates(~ gdp_pc_2023) |>
  fit()
mod_births |>
  forecast(labels = 2024:2025)
}
\seealso{
\itemize{
\item \code{\link[=mod_pois]{mod_pois()}} Specify a Poisson model
\item \code{\link[=mod_binom]{mod_binom()}} Specify a binomial model
\item \code{\link[=mod_norm]{mod_norm()}} Specify a normal model
\item \code{\link[=fit]{fit()}} Fit a model
\item \code{\link[=augment]{augment()}} Extract values for rates,
probabilities, or means, together
with original data
\item \code{\link[=components]{components()}} Extract values for hyper-parameters
\item \href{https://bayesiandemography.github.io/bage/articles/vig2_math.html}{Mathematical Details}
vignette
}
}
