% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/date2week.R, R/week2date.R
\name{date2week}
\alias{date2week}
\alias{week2date}
\title{Convert date to a an arbitrary week definition}
\usage{
date2week(x, week_start = 1, floor_day = FALSE, numeric = FALSE,
  factor = FALSE, ...)

week2date(x, week_start = 1, floor_day = FALSE)
}
\arguments{
\item{x}{a \link{Date}, \link{POSIXt}, \link{character}, or any data that can be easily
converted to a date with \code{\link[=as.POSIXlt]{as.POSIXlt()}}.}

\item{week_start}{a number indicating the start of the week based on the ISO
8601 standard from 1 to 7 where 1 = Monday OR an abbreviation of the
weekdate in an English or current locale. \emph{Note: using a non-English locale
may render your code non-portable.}}

\item{floor_day}{when \code{TRUE}, the days will be set to the start of the week.}

\item{numeric}{if \code{TRUE}, only the numeric week be returned. If \code{FALSE}
(default), the date in the format "YYYY-Www-d" will be returned.}

\item{factor}{if \code{TRUE}, a factor will be returned with levels spanning the
range of dates. If \code{floor_date = FALSE}, then this will use the sequence
of days between the first and last date, but if \code{floor_date = TRUE}, then
the sequence of weeks between the first and last date will be used. \emph{Take
caution when using this with a large date range as the resulting factor can
contain all days between dates}.}

\item{...}{arguments passed to \code{\link[=as.POSIXlt]{as.POSIXlt()}}, unused in all other cases.}
}
\description{
Convert date to a an arbitrary week definition
}
\details{
Weeks differ in their start dates depending on context. The ISO
8601 standard specifies that Monday starts the week
(\url{https://en.wikipedia.org/wiki/ISO_week_date}) while the US CDC uses
Sunday as the start of the week
(\url{https://wwwn.cdc.gov/nndss/document/MMWR_Week_overview.pdf}). For
example, MSF has varying start dates depending on country in order to
better coordinate response.

While there are packages that provide conversion for ISOweeks and epiweeks,
these do not provide seamless conversion from dates to epiweeks with
non-standard start dates. This package provides a lightweight utility to
be able to convert each day.
}
\note{
\code{date2week()} will initially convert the input with \code{\link[=as.POSIXlt]{as.POSIXlt()}} and
use that to calculate the week. If the user supplies character input, it
is expected that the input will be of the format yyyy-mm-dd \emph{unless} the
user explicitly passes the "format" parameter to \code{\link[=as.POSIXlt]{as.POSIXlt()}}. If the
input is not in yyyy-mm-dd and the format parameter is not passed,
\code{date2week()} will result in an error.
}
\examples{

# The same set of days will occur in different weeks depending on the start
# date. Here we can define a week before and after today

print(dat <- Sys.Date() + -6:7)

# By default, the weeks are defined as ISO weeks, which start on Monday
print(iso_dat <- date2week(dat))

# If you want lubridate-style numeric-only weeks, you need look no further
# than the "numeric" argument
date2week(dat, 1, numeric = TRUE)

# You can also convert to factor and include all of the missing dates, but
# beware that this may result in a very large factor due to the number of
# levels present
date2week(Sys.Date() + c(0, 10), factor = TRUE)


# The aweek class can be converted back to a date with `as.Date()`
as.Date(iso_dat)

# If you want to show only the first day of the week, you can use the 
# `floor_day` argument
as.Date(iso_dat, floor_day = TRUE)

# This also works with `factor`:
as.Date(iso_dat, floor_day = TRUE, factor = TRUE)

# ISO week definition: Monday -- 1
date2week(dat, 1)
date2week(dat, "Monday")

# Tuesday -- 2
date2week(dat, 2)
date2week(dat, "Tuesday")

# Wednesday -- 3
date2week(dat, 3)
date2week(dat, "W") # you can use valid abbreviations

# Thursday -- 4
date2week(dat, 4)
date2week(dat, "Thursday")

# Friday -- 5
date2week(dat, 5)
date2week(dat, "Friday")

# Saturday -- 6
date2week(dat, 6)
date2week(dat, "Saturday")

# Epiweek definition: Sunday -- 7 
date2week(dat, 7)
date2week(dat, "Sunday")
}
\seealso{
\code{\link[=as.Date.aweek]{as.Date.aweek()}}, \code{\link[=print.aweek]{print.aweek()}}
}
\author{
Zhian N. Kamvar
}
