% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/random.transactions.R
\name{random.transactions}
\alias{random.transactions}
\alias{random.patterns}
\title{Simulate a Random Transactions}
\usage{
random.transactions(
  nItems,
  nTrans,
  method = "independent",
  ...,
  verbose = FALSE
)

random.patterns(
  nItems,
  nPats = 2000,
  method = NULL,
  lPats = 4,
  corr = 0.5,
  cmean = 0.5,
  cvar = 0.1,
  iWeight = NULL,
  verbose = FALSE
)
}
\arguments{
\item{nItems}{an integer. Number of items to simulate}

\item{nTrans}{an integer. Number of transactions to simulate}

\item{method}{name of the simulation method used (see Details Section).}

\item{...}{further arguments used for the specific simulation method (see
details).}

\item{verbose}{report progress?}

\item{nPats}{number of patterns (potential maximal frequent itemsets) used.}

\item{lPats}{average length of patterns.}

\item{corr}{correlation between consecutive patterns.}

\item{cmean}{mean of  the corruption level (normal distribution).}

\item{cvar}{variance of the corruption level.}

\item{iWeight}{item selection weights to build patterns.}
}
\value{
Returns a \verb{ntrans x nitems} \link{transactions} object.
}
\description{
Simulate random \link{transactions} using different methods.
}
\details{
Currently two simulation methods are implemented:
\itemize{
\item \code{"independent"} (Hahsler et al, 2006): All items
are treated as independent. The transaction size is determined by
\code{rpois(lambda - 1) + 1}, where \code{lambda} can be specified (defaults to 3).
Note that one subtracted from lambda and added to the size to avoid
empty transactions. The items in the transactions are randomly chosen using
the numeric probability vector \code{iProb} of length \code{nItems}
(default: 0.01 for each item).
\item \code{"agrawal"} (see Agrawal and Srikant, 1994): This
method creates transactions with correlated items using \code{random.patters()}.
The simulation is a two-stage process. First, a set of \code{nPats} patterns
(potential maximal frequent itemsets) is generated.  The length of the
patterns is Poisson distributed with mean \code{lPats} and consecutive
patterns share some items controlled by the correlation parameter
\code{corr}.  For later use, for each pattern a pattern weight is generated
by drawing from an exponential distribution with a mean of 1 and a
corruption level is chosen from a normal distribution with mean \code{cmean}
and variance \code{cvar}.
The function returns the patterns as an \code{itemsets} objects which can be
supplied to \code{random.transactions()} as the argument \code{patterns}.  If
no argument \code{patterns} is supplied, the default values given above are
used.

In the second step, the transactions are generated using the patterns.  The
length the transactions follows a Poisson distribution with mean
\code{lPats}. For each transaction, patterns are randomly chosen using the
pattern weights till the transaction length is reached. For each chosen
pattern, the associated corruption level is used to drop some items before
adding the pattern to the transaction.
}
}
\examples{
## generate random 1000 transactions for 200 items with
## a success probability decreasing from 0.2 to 0.0001
## using the method described in Hahsler et al. (2006).
trans <- random.transactions(
  nItems = 200, nTrans = 1000,
  lambda = 5, iProb = seq(0.2, 0.0001, length = 200)
)

## size distribution
summary(size(trans))

## display random data set
image(trans)

## use the method by Agrawal and Srikant (1994) to simulate transactions
## which contains correlated items. This should create data similar to
## T10I4D100K (we just create 100 transactions here to speed things up).
patterns <- random.patterns(nItems = 1000)
summary(patterns)

trans2 <- random.transactions(
  nItems = 1000, nTrans = 100,
  method = "agrawal", patterns = patterns
)
image(trans2)

## plot data with items ordered by item frequency
image(trans2[, order(itemFrequency(trans2), decreasing = TRUE)])
}
\references{
Michael Hahsler, Kurt Hornik, and Thomas Reutterer (2006).
Implications of probabilistic data modeling for mining association rules. In
M. Spiliopoulou, R. Kruse, C. Borgelt, A. Nuernberger, and W. Gaul, editors,
\emph{From Data and Information Analysis to Knowledge Engineering, Studies
in Classification, Data Analysis, and Knowledge Organization}, pages
598--605. Springer-Verlag.

Rakesh Agrawal and Ramakrishnan Srikant (1994). Fast algorithms for mining
association rules in large databases. In Jorge B. Bocca, Matthias Jarke, and
Carlo Zaniolo, editors, \emph{Proceedings of the 20th International
Conference on Very Large Data Bases, VLDB}, pages 487--499, Santiago, Chile.
}
\seealso{
Other itemMatrix and transactions functions: 
\code{\link{abbreviate}()},
\code{\link{c}()},
\code{\link{crossTable}()},
\code{\link{duplicated}()},
\code{\link{extract}},
\code{\link{hierarchy}},
\code{\link{image}()},
\code{\link{inspect}()},
\code{\link{is.superset}()},
\code{\link{itemFrequency}()},
\code{\link{itemFrequencyPlot}()},
\code{\link{itemMatrix-class}},
\code{\link{match}()},
\code{\link{merge}()},
\code{\link{sample}()},
\code{\link{sets}},
\code{\link{size}()},
\code{\link{supportingTransactions}()},
\code{\link{tidLists-class}},
\code{\link{transactions-class}},
\code{\link{unique}()}
}
\author{
Michael Hahsler
}
\concept{itemMatrix and transactions functions}
\keyword{datagen}
