% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Diversity.R
\name{rarefyDiversity}
\alias{rarefyDiversity}
\title{Generate a clonal diversity index curve}
\usage{
rarefyDiversity(data, group, clone = "CLONE", copy = NULL, min_q = 0,
  max_q = 4, step_q = 0.05, min_n = 30, max_n = NULL, ci = 0.95,
  nboot = 2000)
}
\arguments{
\item{data}{data.frame with Change-O style columns containing clonal assignments.}

\item{group}{name of the \code{data} column containing group identifiers.}

\item{clone}{name of the \code{data} column containing clone identifiers.}

\item{copy}{name of the \code{data} column containing copy numbers for each 
sequence. If \code{copy=NULL} (the default), then clone abundance
is determined by the number of sequences. If a \code{copy} column
is specified, then clone abundances is determined by the sum of 
copy numbers within each clonal group.}

\item{min_q}{minimum value of \eqn{q}.}

\item{max_q}{maximum value of \eqn{q}.}

\item{step_q}{value by which to increment \eqn{q}.}

\item{min_n}{minimum number of observations to sample.
A group with less observations than the minimum is excluded.}

\item{max_n}{maximum number of observations to sample. If \code{NULL} the maximum
if automatically determined from the size of the largest group.}

\item{ci}{confidence interval to calculate; the value must be between 0 and 1.}

\item{nboot}{number of bootstrap realizations to generate.}
}
\value{
A \code{\link{DiversityCurve}} object summarizing the diversity scores.
}
\description{
\code{rarefyDiversity} divides a set of clones by a group annotation,
uniformly resamples the sequences from each group, and calculates diversity
scores (\eqn{D}) over an interval of diversity orders (\eqn{q}).
}
\details{
Clonal diversity is calculated using the generalized diversity index (Hill numbers) 
proposed by Hill (Hill, 1973). See \code{\link{calcDiversity}} for further details.

Diversity is calculated on the estimated complete clonal abundance distribution.
This distribution is inferred by using the Chao1 estimator to estimate the number
of seen clones, and applying the relative abundance correction and unseen clone
frequency described in Chao et al, 2015.

To generate a smooth curve, \eqn{D} is calculated for each value of \eqn{q} from
\code{min_q} to \code{max_q} incremented by \code{step_q}.  Variability in total 
sequence counts across unique values in the \code{group} column is corrected by
repeated resampling from the estimated complete clonal distribution to a 
common number of sequences.

The diversity index (\eqn{D}) for each group is the mean value of over all resampling 
realizations. Confidence intervals are derived using the standard deviation of the 
resampling realizations, as described in Chao et al, 2015.
}
\examples{
# Load example data
file <- system.file("extdata", "ExampleDb.gz", package="alakazam")
df <- readChangeoDb(file)

# Group by sample identifier
div <- rarefyDiversity(df, "SAMPLE", step_q=1, max_q=10, nboot=100)
plotDiversityCurve(div, legend_title="Sample")
                   
# Grouping by isotype rather than sample identifier
div <- rarefyDiversity(df, "ISOTYPE", min_n=40, step_q=1, max_q=10, nboot=100)
plotDiversityCurve(div, legend_title="Isotype")

}
\references{
\enumerate{
  \item  Hill M. Diversity and evenness: a unifying notation and its consequences. 
           Ecology. 1973 54(2):427-32.
  \item  Chao A. Nonparametric Estimation of the Number of Classes in a Population. 
           Scand J Stat. 1984 11, 265270.
  \item  Chao A, et al. Rarefaction and extrapolation with Hill numbers: 
           A framework for sampling and estimation in species diversity studies. 
           Ecol Monogr. 2014 84:45-67.
  \item  Chao A, et al. Unveiling the species-rank abundance distribution by 
           generalizing the Good-Turing sample coverage theory. 
           Ecology. 2015 96, 11891201.
}
}
\seealso{
See \code{\link{calcDiversity}} for the basic calculation and 
          \code{\link{DiversityCurve}} for the return object. 
          See \code{\link{testDiversity}} for significance testing.
          See \code{\link{plotDiversityCurve}} for plotting the return object.
}

