% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trtsel.R
\name{trtsel}
\alias{trtsel}
\title{create a trtsel object}
\usage{
trtsel(formula, treatment.name, data, fittedrisk.t0 = NULL,
  fittedrisk.t1 = NULL, thresh = 0, prediction.time = NULL,
  study.design = c("RCT", "nested case-control",
  "stratified nested case-control"), cohort.attributes = NULL,
  link = c("logit", "probit", "cauchit", "log", "cloglog"),
  default.trt = c("trt all", "trt none"))
}
\arguments{
\item{formula}{a 'formula' object consisting of outcome ~ markers and marker by treatment interactions for
the treatment selection model to be evaluated. The outcome can be either binary or a
 'Surv' object for time-to-event outcomes. Binary variable should equal 1 for cases and 0 for controls.}

\item{treatment.name}{Name of the treatment variable in data.frame "data". The treatment variable must be coded 1 for
"treated" and 0 for "un-treated."}

\item{data}{data.frame object used to fit and evaluate the model.}

\item{fittedrisk.t0}{Instead of providing a marker, fitted risks for T=0 and
T=1 may be provided. This should be the column name of the fitted risk for
T=0 that can be found in 'data'. If fitted risks are provided, a model will
not be fit, and all bootstrap confidence intervals will be conditional on
the model fit provided.}

\item{fittedrisk.t1}{Name of for the fitted risks given treatment = 1.}

\item{thresh}{The treatment effect threshold used to define the treatment
rule: Do not treat if the marker-specific treatment effect is less than
"thresh". This is a numeric constant with a default value of 0.}

\item{prediction.time}{a landmark prediction time used only when the outcome is a time-to-event Surv object}

\item{study.design}{Character string indicating the study design used to
collect the data. The three options are "RCT" (default),
"nested case-control", or "stratified nested case-control".  A "RCT" design is simply a randomized trial comparing T = 0 to T = 1 with
the marker measured at baseline.  A nested case-control or stratified nested
case-control study samples cases and controls for marker measurement,
perhaps stratified on treatment assignment, from a randomized trial
comparing T = 0 to T = 1.  See Janes et al. (2013) for a full description of
these designs.

If a "nested case-control" or "stratified nested case-control" design is
specified, "cohort.attributes"" must be provided (see below).}

\item{cohort.attributes}{If a "nested case-control" or "stratified nested
case-control" design is specified, "cohort.attributes" must be provided.
Order does matter.

For the "nested case-control" design, specify the following attributes of
the randomized trial "cohort" from which the case-control sample was
selected: \cr \cr cohort.attributes = c(cohort sample size, \cr proportion
treated in cohort (Pr(trt==1)), \cr adverse event prevalance in cohort
(Pr(event==1)), \cr fraction of cases sampled from cohort) \cr \cr

For the "stratitified nested case-control" design, specify the following
attributes of the randomized trial "cohort" from which the stratified
case-control sample was selected: \cr \cr cohort.attributes = c(cohort
sample size, \cr Pr(trt==0 & event==0) in cohort, \cr Pr(trt==0 & event==1)
in cohort, \cr Pr(trt==1 & event==0) in cohort, \cr fraction of cases with
trt == 0 sampled from cohort, \cr fraction of cases with trt == 1 sampled
from cohort )\cr \cr}

\item{link}{Link function used to fit the risk model for binary outcomes. Options are
"logit"(default), "probit", "cauchit", "log" and "cloglog." Link functions
other than "logit" are available only when the outcome is binary and study.design = "randomized
cohort".}

\item{default.trt}{The default treatment assignment to compare with
marker-based treatment. Can either be set at "trt all" (default) or "trt
none". Use "trt all" if everyone is treated and the aim is to discover those
who would benefit from no treatment, but use "trt none" if the common
practice is to treat no-one and the goal is to discover those who would
benefit from treatment.}
}
\value{
An object of class "trtsel," which is a list containing:

\item{model.fit }{ A list containing "coefficients" -- a 4 x 4 matrix with
columns for coefficient estimates, standard errors, t-statistics, and
two-sided p-values.  "cohort.attributes" -- the vector of cohort.attributes
provided "study.design" -- character string of study.design provided }
\item{derived.data }{ A data.frame with event, trt, and marker information along with
"fittedrisk.t0" (risk estimates given no treatment), "fittedrisk.t1" (risk
estimates given treatment), "trt.effect" (treatment effect estimates), and
"trt.rec" (indicator of trt recommendation) columns.  }
\item{functions}{ For internal package use only }
}
\description{
Creates an object of class "trtsel" given a data.frame containing marker,
treatment, and adverse event status information.  The functions "plot.trtsel",
"evaluate.trtsel", and "calibrate.trtsel" can then be used to plot risk and treatment
effect curves, estimate summary measures, and check model calibration. The
function "compare.trtsel" can also be used to compare two treatment selection
markers.
}
\examples{


data(tsdata)

###########################
## Create trtsel objects 
###########################

trtsel.Y1 <- trtsel(event ~ Y1*trt, 
                   treatment.name = "trt", 
                   data = tsdata, 
                   study.design = "RCT",
                   link = "logit", 
                   default.trt = "trt all")

trtsel.Y1

trtsel.Y2 <- trtsel(event ~ Y2*trt, 
                   treatment.name = "trt", 
                   data = tsdata, 
                   default.trt = "trt all")
trtsel.Y2


# calculate fitted risks using a logistic model 
#(one can use any model here, the point is that the fitted risks are provided )
mymod <- glm(event~trt*Y2, data= tsdata, family = binomial("logit"))

tsdata$fitted.t0 <- predict(mymod, newdata=data.frame(trt = 0, Y2 = tsdata$Y2), type = "response")
tsdata$fitted.t1 <- predict(mymod, newdata=data.frame(trt = 1, Y2 = tsdata$Y2), type = "response")

#all bootstrapping done using this object will be conditional on the model fit. 

myfitted.trtsel <- trtsel( event~trt, treatment.name = "trt", 
                           data = tsdata,
                           fittedrisk.t0 = "fitted.t0",
                           fittedrisk.t1 = "fitted.t1",
                           default.trt = "trt all")



}
\references{
Janes, Holly; Brown, Marshall D; Pepe, Margaret; Huang, Ying;
"An Approach to Evaluating and Comparing Biomarkers for Patient Treatment
Selection" The International Journal of Biostatistics. Volume 0, Issue 0,
ISSN (Online) 1557-4679, ISSN (Print) 2194-573X, DOI: 10.1515/ijb-2012-0052,
April 2014
}
\seealso{
\code{\link{plot.trtsel}} for plotting risk curves and more,
\code{\link{evaluate.trtsel}} for evaluating marker performance,
\code{\link{calibrate.trtsel}} for assessing model calibration, and
\code{\link{compare.trtsel}} to compare two trtsel object.
}
