#' Tidy eta table for further process
#'
#' @description Tidy the table of power dissipation efficiency factor eta, with different logarithm strain
#' rates and temperature for further analysis, based on the given dynamic material model.
#' @param x A calculation result returned from the function \code{\link[TPMplt:DMMprocess]{DMMprocess}}.
#' @return A matrix with melted temperatures, logarithm strain rates and values of power dissipation factor
#' in respective columns.
#' @export etatidy
#' @seealso \code{\link[TPMplt:xitidy]{xitidy}}, \code{\link[TPMplt:DMMprocess]{DMMprocess}}
#'
#' @examples
#' epstable <- epsExtract(TPMdata, 0.7, 2, 3)
#' DMM <- DMMprocess(epstable)
#' etaM <- etatidy(DMM)
#' etaM
#' @keywords internal
etatidy <- function(x){

  # input data check
  if(!inherits(x, "DMMresult")){
    stop("the input data should be DMMresult generated by SRTprocess() function.", call. = FALSE)
  }

  lgbase <- x$MaterialCoefficients$base
  x <- x[2]$tablelist$etatable

  T <- as.numeric(colnames(x))
  SR <- as.numeric(rownames(x))
  lgSR <- log(SR, base = lgbase)
  rpt <- length(lgSR)*length(T)

  i <- 1
  j <- 1
  temp_vec1 <- c()
  temp_vec2 <- c()
  temp_vec3 <- c()
  for(i in 1:length(lgSR)){
    for (j in 1:length(T)) {
      temp_vec1 <- append(temp_vec1, T[j])
      temp_vec2 <- append(temp_vec2, lgSR[i])
      temp_vec3 <- append(temp_vec3, x[i,j])
    }
  }
  M <- cbind(temp_vec1, temp_vec2, temp_vec3)
  colnames(M) <- c("Temperature", "lgStrainRate", "Value")
  return(M)
}

#' Tidy xi table for further process
#'
#' @description Tidy the table of xi values, with different logarithm strain rates and temperatures for further
#' analysis, based on the given dynamic material model.
#' @param x A calculation result returned from the function \code{\link[TPMplt:DMMprocess]{DMMprocess}}.
#' @return A matrix with melted temperatures, logarithm strain rates and values of xi values in respective columns.
#' @export xitidy
#' @seealso \code{\link[TPMplt:etatidy]{etatidy}}, \code{\link[TPMplt:DMMprocess]{DMMprocess}}
#'
#' @examples
#' epstable <- epsExtract(TPMdata, 0.7, 2, 3)
#' DMM <- DMMprocess(epstable)
#' xiM <- xitidy(DMM)
#' xiM
#' @keywords internal
xitidy <- function(x){

  # input data check
  if(!inherits(x, "DMMresult")){
    stop("the input data should be DMMresult generated by SRTprocess() function.", call. = FALSE)
  }

  lgbase <- x$MaterialCoefficients$base
  x <- x[2]$tablelist$xitable

  T <- as.numeric(colnames(x))
  SR <- as.numeric(rownames(x))
  lgSR <- log(SR, base = lgbase)
  rpt <- length(lgSR)*length(T)

  i <- 1
  j <- 1
  temp_vec1 <- c()
  temp_vec2 <- c()
  temp_vec3 <- c()
  for(i in 1:length(lgSR)){
    for (j in 1:length(T)) {
      temp_vec1 <- append(temp_vec1, T[j])
      temp_vec2 <- append(temp_vec2, lgSR[i])
      temp_vec3 <- append(temp_vec3, x[i,j])
    }
  }
  M <- cbind(temp_vec1, temp_vec2, temp_vec3)
  colnames(M) <- c("Temperature", "lgStrainRate", "Value")
  return(M)
}


#' Make grid mesh for plots
#'
#' @description Make a grid for ploting thermal processing map.
#' @param x A calculation result returned from the function \code{\link[TPMplt:DMMprocess]{DMMprocess}}.
#' @param seqby  A numeric value to specify the grid density. The default value used is 80, namely the
#' default mesh used 80*80 for original plot.
#'
#' @return A grid table with the ranges for logarithm strain rate and temperature.
#' @export MakeGrid
#'
#' @examples
#' epstable <- epsExtract(TPMdata, 0.7, 2, 3)
#' DMM <- DMMprocess(epstable)
#' prdptr <- MakeGrid(DMM)
#' prdptr
#' @keywords internal
MakeGrid <- function(x, seqby=80){

  # input data check
  if(!inherits(x, "DMMresult")){
    stop("the input data should be TPMdata generated by SRTprocess() function.", call. = FALSE)
  }

  M <- etatidy(x)

  GridT <- seq(min(M[,1]), max(M[,1]), by=(max(M[,1]-min(M[,1])))/seqby)
  GridlgSR <- seq(min(M[,2]), max(M[,2]), by=(max(M[,2]-min(M[,2])))/seqby)
  predictor <- expand.grid(T=GridT, lgSR=GridlgSR)

  len <- length(predictor[,1])
  value <- rep(1, len)
  result <- cbind(predictor, value)

  return(result)
}
