% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Coefficient_Plot_Functions.R
\name{plot.cv.SplitGLM}
\alias{plot.cv.SplitGLM}
\title{Plot of coefficients paths for cv.SplitGLM Object}
\usage{
\method{plot}{cv.SplitGLM}(
  x,
  group_index = NULL,
  plot_type = c("Coef", "CV-Error")[1],
  active_only = TRUE,
  path_type = c("Log-Lambda", "L1-Norm")[1],
  labels = TRUE,
  ...
)
}
\arguments{
\item{x}{An object of class cv.SplitGLM.}

\item{group_index}{The group for which to return the coefficients. Default is the ensemble coefficients.}

\item{plot_type}{Plot of coefficients, "Coef" (default), or cross-validated error or deviance, "CV-Error".}

\item{active_only}{Only include the variables selected in final model (default is TRUE).}

\item{path_type}{Plot of coefficients paths as a function of either "Log-Lambda" (default) or "L1-Norm".}

\item{labels}{Include the labels of the variables (default is FALSE).}

\item{...}{Additional arguments for compatibility.}
}
\value{
The coefficients for the cv.SplitGLM object.
}
\description{
\code{plot.cv.SplitGLM} returns the coefficients for a cv.SplitGLM object.
}
\examples{
\donttest{
# Data simulation
set.seed(1)
n <- 50
N <- 2000
p <- 1000
beta.active <- c(abs(runif(p, 0, 1/2))*(-1)^rbinom(p, 1, 0.3))
# Parameters
p.active <- 100
beta <- c(beta.active[1:p.active], rep(0, p-p.active))
Sigma <- matrix(0, p, p)
Sigma[1:p.active, 1:p.active] <- 0.5
diag(Sigma) <- 1

# Train data
x.train <- mvnfast::rmvn(n, mu = rep(0, p), sigma = Sigma) 
prob.train <- exp(x.train \%*\% beta)/
              (1+exp(x.train \%*\% beta))
y.train <- rbinom(n, 1, prob.train)
mean(y.train)
# Test data
x.test <- mvnfast::rmvn(N, mu = rep(0, p), sigma = Sigma)
prob.test <- exp(x.test \%*\% beta)/
             (1+exp(x.test \%*\% beta))
y.test <- rbinom(N, 1, prob.test)
mean(y.test)

# SplitGLM - CV (Multiple Groups)
split.out <- cv.SplitGLM(x.train, y.train,
                         glm_type="Logistic",
                         G=10, include_intercept=TRUE,
                         alpha_s=3/4, alpha_d=1,
                         n_lambda_sparsity=50, n_lambda_diversity=50,
                         tolerance=1e-3, max_iter=1e3,
                         n_folds=5,
                         active_set=FALSE,
                         n_threads=1)
                         
# Plot of coefficients paths (function of Log-Lambda)
plot(split.out, plot_type="Coef", path_type="Log-Lambda", group_index=1, labels=FALSE)

# Plot of coefficients paths (function of L1-Norm)
plot(split.out, plot_type="Coef", path_type="L1-Norm", group_index=1, labels=FALSE)

# Plot of CV error
plot(split.out, plot_type="CV-Error")

}

}
\seealso{
\code{\link{cv.SplitGLM}}
}
\author{
Anthony-Alexander Christidis, \email{anthony.christidis@stat.ubc.ca}
}
