% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neighbourhood.R
\name{adj}
\alias{adj}
\title{Fast \code{adjacent} function, and Just In Time compiled version}
\usage{
adj(
  x = NULL,
  cells,
  directions = 8,
  sort = FALSE,
  pairs = TRUE,
  include = FALSE,
  target = NULL,
  numCol = NULL,
  numCell = NULL,
  match.adjacent = FALSE,
  cutoff.for.data.table = 2000,
  torus = FALSE,
  id = NULL,
  numNeighs = NULL,
  returnDT = FALSE
)
}
\arguments{
\item{x}{\verb{Raster*} object for which adjacency will be calculated.}

\item{cells}{vector of cell numbers for which adjacent cells should be found.
Cell numbers start with 1 in the upper-left corner and increase
from left to right and from top to bottom.}

\item{directions}{the number of directions in which cells should be connected:
4 (rook's case), 8 (queen's case), or \code{"bishop"} to connect
cells with one-cell diagonal moves.
Or a neighbourhood matrix (see Details).}

\item{sort}{logical. Whether the outputs should be sorted or not, using cell ids
of the \code{from} cells (and \code{to} cells, if \code{match.adjacent}
is \code{TRUE}).}

\item{pairs}{logical. If \code{TRUE}, a matrix of pairs of adjacent cells is returned.
If \code{FALSE}, a vector of cells adjacent to cells is returned}

\item{include}{logical. Should the focal cells be included in the result?}

\item{target}{a vector of cells that can be spread to. This is the inverse of a mask.}

\item{numCol}{numeric indicating number of columns in the raster.
Using this with \code{numCell} is a bit faster execution time.}

\item{numCell}{numeric indicating number of cells in the raster.
Using this with \code{numCol} is a bit faster execution time.}

\item{match.adjacent}{logical. Should the returned object be the same as
\code{raster::adjacent}.
Default \code{FALSE}, which is faster.}

\item{cutoff.for.data.table}{numeric. If the number of cells is above this value,
the function uses data.table which is faster with
large numbers of cells. Default is 5000, which appears
to be the turning point where data.table becomes faster.}

\item{torus}{Logical. Should the spread event wrap around to the other side of the raster?
Default is \code{FALSE}.}

\item{id}{numeric If not \code{NULL} (default), then function will return \code{"id"} column.}

\item{numNeighs}{A numeric scalar, indicating how many neighbours to return. Must be
less than or equal to \code{directions}; which neighbours are random
with equal probabilities.}

\item{returnDT}{A logical. If TRUE, then the function will return the result
as a \code{data.table}, if the internals used \code{data.table},
i.e., if number of cells is greater than \code{cutoff.for.data.table}.
User should be warned that this will therefore cause the output
format to change depending \code{cutoff.for.data.table}.
This will be faster for situations where \code{cutoff.for.data.table = TRUE}.}
}
\value{
Either a matrix (if more than 1 column, i.e., \code{pairs = TRUE},
and/or \code{id} is provided), a vector (if only one column), or a \code{data.table}
(if \code{cutoff.for.data.table} is less than \code{length(cells)} \emph{and}
\code{returnDT} is \code{TRUE}.
To get a consistent output, say a matrix, it would be wise to test the output
for its class.
The variable output is done to minimize coercion to maintain speed.
The columns will be one or more of \code{id}, \code{from}, \code{to}.
}
\description{
Faster function for determining the cells of the 4, 8 or bishop
neighbours of the \code{cells}. This is a hybrid function that uses
matrix for small numbers of loci (<1e4) and data.table for larger numbers of loci
}
\details{
Between 4x (large number loci) to 200x (small number loci) speed gains over
\code{adjacent} in raster package. There is some extra speed gain if
\code{NumCol} and \code{NumCells} are passed rather than a raster.
Efficiency gains come from:
\enumerate{
\item use \code{data.table} internally
\itemize{
\item no need to remove NAs because wrapped or outside points are
just removed directly with data.table
\item use data.table to sort and fast select (though not fastest possible)
}
\item don't make intermediate objects; just put calculation into return statement
}

The steps used in the algorithm are:
\enumerate{
\item Calculate indices of neighbouring cells
\item Remove "to" cells that are
\itemize{
\item <1 or >numCells (i.e., they are above or below raster), using a single modulo
calculation
\item where the modulo of "to" cells is equal to 1 if "from" cells are 0 (wrapped right
to left)
\item or where the modulo of the "to" cells is equal to 0 if "from" cells are 1 (wrapped
left to right)
}
}
}
\examples{
library(terra)
a <- rast(ext(0, 1000, 0, 1000), res = 1)
sam <- sample(1:ncell(a), 1e4)
numCol <- ncol(a)
numCell <- ncell(a)
adj.new <- adj(numCol = numCol, numCell = numCell, cells = sam, directions = 8)
adj.new <- adj(numCol = numCol, numCell = numCell, cells = sam, directions = 8,
               include = TRUE)

}
\seealso{
\code{\link[terra:adjacent]{terra::adjacent()}}
}
\author{
Eliot McIntire
}
