#%%%%%%%%%%%%%%%%%%%%%
# GLOBAL LIBRARY #####
#%%%%%%%%%%%%%%%%%%%%%

require(DT)
require(plotly)
require(shinyjs)

#%%%%%%%%%%%%%%%%%%%%%
# UI #################
#%%%%%%%%%%%%%%%%%%%%%

ui = tagList(
  tags$head(tags$link(rel = "stylesheet",
                      type = "text/css",
                      href = "style.css"),
            tags$script(type = "text/javascript",
                        src = "busy.js"),
            tags$script(type = "text/javascript",
                        src = "report_generating_message.js"),
            tags$script(type = "text/javascript",
                        src = "report_downloading_message.js"),
            tags$link(rel = "stylesheet",
                      type = "text/css",
                      href = "flag-icon.css"),
            tags$link(rel = "stylesheet",
                      type = "text/css",
                      href = "margins_and_paddings.css"),
            tags$link(rel = "shortcut icon", href = "Rlogo.png"),
            tags$style(type = "text/css",
                       ".panel-footer {position: fixed; right: 0; bottom: 0; left: 0;}")),

  div(class = "busy",
      p("Loading"),
      img(src = "busy_indicator.gif", height = 100, width = 100)
  ),

  shinyjs::useShinyjs(),

  tags$head(includeScript("google-analytics.js")),

  navbarPage(title = div(HTML('<font size = "5"> ShinyItemAnalysis </font>
                               <font size = "2"> Test and item analysis </font> </p>')),
             windowTitle = 'ShinyItemAnalysis',
             position = 'fixed-top',
             selected = 'About',
             collapsible = TRUE,
             footer = list(
               div(class = "clear"),
               div(class = "panel-footer",
                   HTML('<p> <font size = "4"> ShinyItemAnalysis </font>
                             <font size = "2"> Test and item analysis | Version 1.2.6 </font>
                             <span style = "float:right">
                                <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysis/" id = "tooltipweb" target="_blank"> <img src = "web_icon.png", style = "width: 25px;"> </a>
                                <a href = "https://github.com/patriciamar/ShinyItemAnalysis/" id = "tooltipgithub" target="_blank"> <img src = "github_icon.png", style = "width: 25px;"> </a>
                                <a href = "https://CRAN.R-project.org/package=ShinyItemAnalysis/" id = "tooltipcran" target="_blank"> <img src = "cran_icon.png", style = "width: 25px;"> </a>
                             </span> </p>
                         <script>
                           $("#tooltipweb").attr("title", "Web");
                           $("#tooltipgithub").attr("title", "GitHub");
                           $("#tooltipcran").attr("title", "CRAN");
                         </script>'),
                   p(HTML('&copy; 2018  Patricia Martinkova, Adela Drabinova, Ondrej Leder and Jakub Houdek'
                   ),
                    HTML('<span style = "float:right">'),
                        textOutput('counter', inline = T),
                    HTML('</span>')))),
             theme = "bootstrap.css",
             #%%%%%%%%%%%%%%%%%%%%%
             # MAIN PANEL #########
             #%%%%%%%%%%%%%%%%%%%%%

             #%%%%%%%%%%%%%%%%%%%%%
             # ABOUT ##############
             #%%%%%%%%%%%%%%%%%%%%%
             tabPanel("About",
                      h3('Description'),
                      p(code('ShinyItemAnalysis'), ' provides analysis of educational tests (such as admission tests)
                        and their items including:' ),
                      tags$ul(
                        tags$li("Exploration of total and standard scores on ", strong('Summary'), "page. "),
                        tags$li("Correlation structure and predictive validity analysis on ", strong('Validity'), "page. "),
                        tags$li("Item and distractor analysis on ", strong('Item analysis'), "page. "),
                        tags$li('Item analysis by logistic models on ', strong('Regression'), "page. "),
                        tags$li('Item analysis by item response theory models on ', strong('IRT models'), "page. "),
                        tags$li('Differential item functioning (DIF) and differential distractor functioning (DDF)
                                methods on ', strong('DIF/Fairness'), "page. ")
                        ),
                      p('This application is based on the free statistical software',
                        a('R', href = 'https://cran.r-project.org/', target = "_blank"),
                        ' and its ',
                        a('shiny', href = 'http://www.rstudio.com/shiny/', target = "_blank"),
                        'package. '),
                      p('For all graphical outputs a download button is provided. Moreover, on ', strong('Reports'),
                        'page HTML or PDF report can be created. Additionaly, all application outputs are
                        complemented by selected R code hence the similar analysis can be run and modified in R.'),

                      h4('Data'),
                      p('For demonstration purposes, by default, 20-item dataset', code('GMAT'),'
                        from R ', code('difNLR'),' package is used. Other three datasets are available: ',
                        code('GMAT2'), 'and', code('MSAT-B'), 'from', code('difNLR'), 'package and ',
                        code('Medical 100'), 'from', code('ShinyItemAnalysis'), 'package.
                        You can change the dataset (and try your own one) on page', strong('Data.')),

                      h4('Availability'),
                      p('Application can be downloaded as R package from ',
                        a('CRAN. ',
                          href = 'https://CRAN.R-project.org/package=ShinyItemAnalysis',
                          target = '_blank'),
                        'It is also available online at ',
                        a('Czech Academy of Sciences ',
                          href = 'https://shiny.cs.cas.cz/ShinyItemAnalysis/',
                          target = "_blank"),
                        HTML(
                          paste(
                            "<a href = 'https://shiny.cs.cas.cz/ShinyItemAnalysis/' target = '_blank'>",
                            "<img src = 'Flag_of_the_Czech_Republic.png' height = '16' border = '0' align = 'middle'>",
                            "</a>.",
                            sep = ""
                          )
                        ),
                        a('Charles University',
                          href = 'http://shiny.statest.cz:3838/ShinyItemAnalysis/',
                          target = "_blank"),
                        HTML(
                          paste(
                            "<a href = 'http://shiny.statest.cz:3838/ShinyItemAnalysis/' target = '_blank'>",
                            "<img src = 'Flag_of_the_Czech_Republic.png' height = '16' border = '0' align = 'middle'>",
                            "</a>",
                            sep = ""
                          )
                        ),
                        'or',
                        a('shinyapps.io',
                          href = 'https://patriciamar.shinyapps.io/ShinyItemAnalysis/',
                          target = "_blank"),
                        HTML(
                          paste(
                            "<a href = 'https://patriciamar.shinyapps.io/ShinyItemAnalysis/' target = '_blank'>",
                            "<img src = 'Flag_of_the_United_States.png' height = '16' border = '0' align = 'middle'>",
                            "</a>.",
                            sep = ""
                          )
                        )
                      ),

                      h4('Version'),
                      p('Current version of ', code('ShinyItemAnalysis'), ' available on ',
                        a('CRAN', href = 'https://CRAN.R-project.org/package=ShinyItemAnalysis', target = "_blank"), 'is 1.2.6.
                        Version available',
                        a('online', href = 'https://shiny.cs.cas.cz/ShinyItemAnalysis/', target = "_blank"), 'is 1.2.6.
                        The newest development version available on ',
                        a('GitHub', href = 'https://github.com/patriciamar/ShinyItemAnalysis', target = "_blank"), 'is 1.2.6.',
                        br(),
                        'See also older versions: ',
                        a('0.1.0, ', href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV01/", target = "_blank"),
                        a('0.2.0, ', href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV02/", target = "_blank"),
                        a('1.0.0, ', href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV100/", target = "_blank"),
                        a('1.1.0. ', href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV110/", target = "_blank")),

                      h4('Authors and contributors'),

                      fluidRow(
                        column(1, align = "center",
                               img(src = 'patricia_new.png', width = 75),
                               HTML("<figcaption><a href='http://www.cs.cas.cz/martinkova/' target='_blank'>Patricia<br>Martinkova</a></figcaption>")),
                        column(1, align = "center",
                               img(src = 'adela.jpg', width = 75),
                               HTML("<figcaption><a href='http://www.cs.cas.cz/drabinova/' target='_blank'>Adela<br>Drabinova</a></figcaption>")),
                        column(1, align = "center",
                               img(src = 'leder.png', width = 75),
                               HTML("<figcaption><a href='https://www.linkedin.com/in/ond%C5%99ej-leder-3864b1119' target='_blank'>Ondrej<br>Leder</a></figcaption>")),
                        column(1, align = "center",
                               img(src = 'jakub_new.png', width = 75),
                               HTML("<figcaption>Jakub<br>Houdek</figcaption>")),
                        column(1, align = "center",
                               img(src = 'Lubos.jpg', width = 75),
                               HTML("<figcaption>Lubomir<br>Stepanek</figcaption>"))
                      ),


                      h4('List of packages used'),
                      fluidRow(
                        column(2,
                               code('library(corrplot)'), br(),
                               code('library(CTT)'), br(),
                               code('library(data.table)'), br(),
                               code('library(deltaPlotR)'), br(),
                               code('library(DT)'), br(),
                               code('library(difNLR)'), br(),
                               code('library(difR)'), br(),
                               code('library(ggplot2)'), br(),
                               code('library(grid)'), br()),
                        column(2,
                               code('library(gridExtra)'), br(),
                               code('library(knitr)'), br(),
                               code('library(latticeExtra)'), br(),
                               code('library(ltm)'), br(),
                               code('library(mirt)'), br(),
                               code('library(moments)'), br(),
                               code('library(msm)'), br(),
                               code('library(nnet)'), br(),
                               code('library(plotly)'), br()),
                        column(2,
                               code('library(psych)'), br(),
                               code('library(psychometric)'), br(),
                               code('library(reshape2)'), br(),
                               code('library(rmarkdown)'), br(),
                               code('library(shiny)'), br(),
                               code('library(shinyjs)'), br(),
                               code('library(stringr)'), br(),
                               code('library(WrightMap)'), br(),
                               code('library(xtable)'), br())
                      ),
                      h4('References'),
                      p('To cite package', code('ShinyItemAnalysis'), 'in publications please use:'),
                      p('Martinkova P., Drabinova A., Leder O., & Houdek J. (2018). ShinyItemAnalysis: Test and item analysis
  via shiny. R package version 1.2.6. https://CRAN.R-project.org/package=ShinyItemAnalysis'),
                      p('Martinkova, P., Drabinova, A., & Houdek, J. (2017). ShinyItemAnalysis: Analyza prijimacich a jinych
  znalostnich ci psychologickych testu [ShinyItemAnalysis: Analyzing admission and other educational and
  psychological tests]. TESTFORUM, 6(9), 16-35. doi:10.5817/TF2017-9-129'),

                      h4('Bug reports'),
                      p("If you discover a problem with this application please contact the project maintainer
                        at martinkova(at)cs.cas.cz or use ", a("GitHub.",
                                                               href = "https://github.com/patriciamar/ShinyItemAnalysis/issues",
                                                               target = "_blank")),

                      h4('Acknowledgments'),
                      p(" Project was supported by grant funded by Czech Science foundation under number ",
                        a("GJ15-15856Y.",
                          href = "http://www.cs.cas.cz/martinkova/psychometrics.html",
                          target = "_blank")),

                      h4('License'),
                      p("This program is free software and you can redistribute it and or modify it under the terms
                         of the",
                        a("GNU GPL 3",
                          href = "https://www.gnu.org/licenses/gpl-3.0.en.html",
                          target = "_blank"),
                        "as published by the Free Software Foundation. This program is distributed in the hope that
                         it will be useful, but without any warranty; without even the implied warranty of
                         merchantability of fitness for a particular purpose." ),
                      br(),
                      br()
                      ),
             #%%%%%%%%%%%%%%%%%%%%%
             # DATA ###############
             #%%%%%%%%%%%%%%%%%%%%%


             tabPanel("Data",
               tabsetPanel(
               # ** Summary ####
               tabPanel("Data",
                      h3("Data"),
                      h4("Training datasets"),
                      p('For demonstration purposes, 20-item dataset ' , code("GMAT"),'
                         and dataset', code("GMATkey"),' from R ', code('difNLR'),' package are used.
                         On this page, you may select one of four datasets offered from ', code('difNLR'),
                        ' and ', code('ShinyItemAnalysis'), 'packages or you may upload your own dataset
                         (see below). To return to demonstration dataset,
                         refresh this page in your browser' , strong("(F5)"), '.'),
                      p('Used dataset ', code("GMAT"), a(' (Martinkova, et al., 2017) ',
                                 href = "http://www.lifescied.org/content/16/2/rm2.full.pdf+html?with-ds=yes",
                        target = "_blank"),
                        ' is generated based on parameters of real Graduate Management
                        Admission Test (GMAT) data set (Kingston et al., 1985). However, first two items were
                        generated to function differently in uniform and non-uniform way respectively.
                        The data set represents responses of 2,000 subjects (1,000 males, 1,000 females) to
                        multiple-choice test of 20 items. The distribution of total scores is the same for both groups.
                        See ', a('Martinkova, et al. (2017) ',
                                 href = "http://www.lifescied.org/content/16/2/rm2.full.pdf+html?with-ds=yes",
                                 target = "_blank"),
                        'for further discussion. '),
                      p('Dataset ', code("GMAT2"), ' (Drabinova & Martinkova, 2016) is also generated based on parameters of GMAT (Kingston et
                         al., 1985) from R ', code('difNLR'),' package . Again, first two items were generated
                         to function differently in uniform and non-uniform way respectively. The data set
                         represents responses of 1,000 subjects (500 males, 500 females) to multiple-choice test
                         of 20 items. '),
                      p('Dataset ', code("MSAT-B"), ' (Drabinova & Martinkova, 2017) is a subset of real Medical School Admission
                         Test in Biology in Czech Republic. The data set represents responses of 1,407 subjects (484 males,
                         923 females) to multiple-choice test of 20 items. First item was previously detected as
                         functioning differently. For more details of item selection see Drabinova and Martinkova (2017).
                         Dataset can be found in R ', code('difNLR'),' package.'),
                      p('Dataset ', code("Medical 100"), ' is a real data set of admission test to medical school
                         from R ', code('ShinyItemAnalysis'),' package. The data set represents responses of
                         2,392 subjects (750 males, 1,633 females and 9 subjects without gender specification)
                         to multiple-choice test of 100 items. '),
                      br(),
                      selectInput("dataSelect", "Select dataset",
                                  c("GMAT" = "GMAT_difNLR",
                                    "GMAT2" = "GMAT2_difNLR",
                                    "MSAT-B" = "MSATB_difNLR",
                                    "Medical 100" = "dataMedical_ShinyItemAnalysis"
                                  ),
                                  selected="GMAT_difNLR"),
                      tags$hr(),
                      h4("Upload your own datasets"),
                      p('Main ', strong('data'), ' file should contain responses of individual students (rows) to given items
                        (columns). Header may contain item names, no row names should be included. If responses
                        are in unscored ABCD format, the ', strong('key'), ' provides correct response for each item. If responses are
                        scored 0-1, key is vector of 1s.'),
                      p(strong('Group'), ' is 0-1 vector, where 0 represents reference group
                        and 1 represents focal group. Its length need to be the same as number of individual
                        students in main dataset. If the group is not provided then it wont be possible to run DIF and DDF
                        detection procedures on ', strong('DIF/Fairness'), ' page. '),
                      p(strong('Criterion variable'), ' is either
                        discrete or continuous vector (e.g. future study success or future GPA in case of admission tests) which
                        should be predicted by the measurement. Again, its length needs to be the same as number of individual
                        students in the main dataset. If the criterion variable is not provided then it wont be possible to run
                        validity analysis in ', strong('Predictive validity'), ' section on ', strong('Validity'), ' page.'),
                      p('In all data sets', strong('header'), 'should be either included or excluded. Columns of dataset are by default renamed to Item and number of particular column. If you
                        want to keep your own names, check box ', strong('Keep items names'), 'below. Missing values in scored dataset are by default evaluated as 0. If you want to keep them as missing,
                        check box' , strong('Keep missing values'), 'below.'),
                      fluidRow(
                        column(3, offset = 0, fileInput(
                          'data', 'Choose data (csv file)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, fileInput(
                          'key', 'Choose key (csv file)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, fileInput(
                          'groups', 'Choose groups for DIF (optional)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, fileInput(
                          'criterion_variable', 'Choose criterion variable (optional)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        )),
                      div(style = "display: inline-block; vertical-align: top; horizontal-align: center; width: 150px;",
                          actionButton(inputId = "submitButton", label = "Submit Data")),
                      div(style = "display: inline-block; vertical-align: top; horizontal-align: center; width: 80%;",
                          htmlOutput("checkDataText"),
                          htmlOutput("checkDataColumns01Text")),
                      tags$hr(),
                      h4("Data specification"),
                      fluidRow(
                        column(3, offset = 0,
                               checkboxInput('header', 'Header', TRUE),
                               checkboxInput('itemnam', 'Keep items names', FALSE),
                               checkboxInput('missval', 'Keep missing values', FALSE)),
                        column(3, offset = 1, radioButtons('sep', 'Separator',
                                                           c(Comma = ',',
                                                             Semicolon = ';',
                                                             Tab = '\t'
                                                           ),
                                                           ','
                        )
                        ),
                        column (3, offset = 0, radioButtons('quote', 'Quote',
                                                            c(None = '',
                                                              'Double Quote' = '"',
                                                              'Single Quote' = "'"
                                                            ),
                                                            '"'
                        )
                        )
                      ),
                      br(),
                      br()),
               tabPanel("Data exploration",
                      h3("Data exploration"),
                      p("Here you can explore uploaded dataset. Rendering of tables can take some time."),
                      h4("Main dataset"),
                      # tableOutput("headdata_print"),
                      # verbatimTextOutput("headdata_print"),
                      DT::dataTableOutput('headdata'),
                      #shiny::dataTableOutput('headdata'),
                      h4("Key (correct answers)"),
                      # tableOutput("key_print"),
                      # verbatimTextOutput("key_print"),
                      DT::dataTableOutput('key'),
                      #shiny::dataTableOutput('key'),
                      h4("Scored test"),
                      # tableOutput("sc01_print"),
                      # verbatimTextOutput("sc01_print"),
                      DT::dataTableOutput('sc01'),
                      #shiny::dataTableOutput('sc01'),
                      h4("Group vector"),
                      # tableOutput("group_print"),
                      # verbatimTextOutput("group_print"),
                      DT::dataTableOutput('group'),
                      #shiny::dataTableOutput('group'),
                      h4("Criterion variable vector"),
                      # tableOutput("critvar_print"),
                      # verbatimTextOutput("critvar_print"),
                      DT::dataTableOutput('critvar'),
                      #shiny::dataTableOutput('critvar'),
                      br(),
                      br(),
                      br()
                      ))),
             #%%%%%%%%%%%%%%%%%%%%%
             # SUMMARY ############
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Summary",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Analysis of total scores"),
                                 h4("Summary table"),
                                 p("Table below summarizes basic characteristics of total scores including
                                   minimum and maximum, mean, median, standard deviation, skewness and kurtosis.
                                   The kurtosis here is estimated by sample kurtosis \\(\\frac{m_4}{m_2^2}\\),
                                   where \\(m_4\\) is the fourth central moment and \\(m_2\\) is the second
                                   central moment. The skewness is estimated by sample skewness
                                   \\(\\frac{m_3}{m_2^{3/2}}\\), where \\(m_3\\) is the third central moment.
                                   The kurtosis for normally distributed scores is near the value of 3 and the skewness
                                   is near the value of 0. "),
                                 tableOutput('totalscores_table'),
                                 h4("Histogram of total score"),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "slider_totalscores_histogram",
                                                           label = "Cut-score",
                                                           min = 0,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1))),
                                 p('For selected cut-score, blue part of histogram shows students with total score
                                   above the cut-score, grey column shows students with total score equal
                                   to the cut-score and red part of histogram shows students below the cut-score.'),
                                 plotOutput('totalscores_histogram'),
                                 downloadButton(outputId = "DB_totalscores_histogram", label = "Download figure"),
                                 br(),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'), br(),
                                     code('library(moments)'), br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('score <- apply(data, 1, sum) # Total score'),
                                     br(),
                                     br(),
                                     code('# Summary of total score'),
                                     br(),
                                     code('c(min(score), max(score), mean(score), median(score), sd(score), skewness(score), kurtosis(score))'),
                                     br(),
                                     code('# Histogram'),
                                     br(),
                                     code('cut <- median(score) # cut-score'), br(),
                                     code('col <- c(rep("red", cut - min(score) + 2), "gray",
                                          rep("blue", max(score) - cut + 1)) # colors for histogram'), br(),
                                     code('hist(score, breaks = 0:ncol(data), col = col) ')),
                                 br()
                                 ),
                        # * STANDARD SCORES ####
                        tabPanel("Standard scores",
                                 h3('Standard scores'),
                                 p(strong('Total score'), 'also known as raw score is a total number of correct
                                   answers. It can be used to compare individual score to a norm group, e.g. if the mean
                                   is 12, then individual score can be compared to see if it is below or above this average. ', br(),
                                   strong('Percentile'), 'indicates the value below which a percentage of observations
                                   falls, e.g. a individual score at the 80th percentile means that the individual score
                                   is the same or higher than the scores of 80% of all respondents. ', br(),
                                   strong('Success rate'), 'is the percentage of success, e.g. if the maximum points of test
                                   is equal to 20 and individual score is 12 then success rate is 12/20 = 0.6, i.e. 60%.', br(),
                                   strong('Z-score'), 'or also standardized score is a linear transformation of total
                                   score with a mean of 0 and with variance of 1. If X is total score, M its mean and SD its
                                   standard deviation then Z-score = (X - M) / SD. ', br(),
                                   strong('T-score'), 'is transformed Z-score with a mean of 50 and standard deviation
                                   of 10. If Z is Z-score then T-score = (Z * 10) + 50. '),
                                 h4("Table by score"),
                                 tableOutput('scores_tables'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('score <- apply(data, 1, sum) # Total score'),
                                     br(),
                                     code('tosc <- sort(unique(score)) # Levels of total score'),
                                     br(),
                                     code('perc <- cumsum(prop.table(table(score))) # Percentiles'),
                                     br(),
                                     code('sura <- 100 * (tosc / max(score)) # Success rate'),
                                     br(),
                                     code('zsco <- sort(unique(scale(score))) # Z-score'),
                                     br(),
                                     code('tsco <- 50 + 10 * zsco # T-score')),
                                 br()
                        )
                        ),
             #%%%%%%%%%%%%%%%%%%%%%
             # VALIDITY ###########
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Validity",
                        # * CORRELATION STRUCTURE ####
                         tabPanel("Correlation structure",
                                  h3("Correlation structure"),
                                  h4("Polychoric correlation heat map"),
                                  p('Polychoric correlation heat map is a correlation plot which displays a polychoric
                                     correlations of items. The size and shade of circles indicate how much the
                                     items are correlated (larger and darker circle means larger correlation).
                                     The color of circles indicates in which way the items are correlated - blue
                                     color shows possitive correlation and red color shows negative correlation.'),
                                  p("Polychoric correlation heat map can be reorder using hierarchical",
                                    HTML("<b>clustering method</b>"), "below.
                                    Ward's method aims at finding compact clusters based on minimizing the within-cluster
                                    sum of squares.
                                    Ward's n. 2 method used squared disimilarities.
                                    Single method connects clusters with the nearest neighbours, i.e. the distance between
                                    two clusters is calculated as the minimum of distances of observations in one cluster and
                                    observations in the other clusters.
                                    Complete linkage with farthest neighbours, i.e. maximum of distances.
                                    Average linkage method used the distance based on weighted average of the individual distances.
                                    With McQuitty method used unweighted average.
                                    Median linkage calculates the distance as the median of distances between an observation
                                    in one cluster and observation in the other cluster.
                                    Centroid method used distance between centroids of clusters. "),
                                  p("With", HTML("<b>number  of clusters</b>"), "larger than 1, the rectangles representing
                                    clusters are drawn. "),
                                  fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                numericInput(inputId = 'corr_plot_clust',
                                                             label = 'Number of clusters',
                                                             value = 1,
                                                             min = 1,
                                                             max = 1)),
                                            div(style = "display: inline-block; vertical-align: top; width: 5%;"),
                                            div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                selectInput(inputId = 'corr_plot_clustmethod',
                                                            label = 'Clustering method',
                                                            choices = list("None" = "none",
                                                                           "Ward's"  = "ward.D",
                                                                           "Ward's n. 2" = "ward.D2",
                                                                           "Single" = "single",
                                                                           "Complete" = "complete",
                                                                           "Average" = "average",
                                                                           "McQuitty" = "mcquitty",
                                                                           "Median" = "median",
                                                                           "Centroid" = "centroid"),
                                                            selected = "none"))),
                                  plotOutput('corr_plot'),
                                  downloadButton(outputId = "DB_corr_plot", label = "Download figure"),
                                  br(),
                                  h4("Scree plot"),
                                  p('A scree plot displays the eigenvalues associated with an component or a factor in descending order
                                     versus the number of the component or factor. '),
                                  plotOutput('scree_plot'),
                                  downloadButton(outputId = "DB_scree_plot", label = "Download figure"),
                                  h4("Selected R code"),
                                  div(code('library(corrplot)'),
                                      br(),
                                      code('library(difNLR)'),
                                      br(),
                                      code('library(psych)'),
                                      br(),
                                      code('data(GMAT)'),
                                      br(),
                                      code('data  <- GMAT[, 1:20]'),
                                      br(),
                                      br(),
                                      code('# Correlation plot'),
                                      br(),
                                      code('corP <- polychoric(data)'),
                                      br(),
                                      code('corrplot(corP$rho)'),
                                      br(),
                                      code('corP$rho # Correlation matrix'),
                                      br(),
                                      br(),
                                      code('# Scree plot'),
                                      br(),
                                      code('plot(1:length(eigen(corP$rho)$values), eigen(corP$rho)$values,
                                            ylab = "Eigen value", xlab = "Component Number")'),
                                      br(),
                                      code('lines(1:length(eigen(corP$rho)$values), eigen(corP$rho)$values)'),
                                      br(),
                                      code('eigen(corP$rho) # Eigen values and vectors')),
                                  br()
                                    ),
                        # * PREDICTIVE VALIDITY ####
                        tabPanel('Predictive validity',
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Predictive validity'),
                                            p('This section requires criterion variable (e.g. future study success or future GPA in case
                                              of admission tests) which should be predicted by the measurement. This outcome variable
                                              can be uploaded in ', strong('Data'), 'section. Then you can explore how data predict this
                                              variable. '),
                                            h4('Descriptive plots of criterion variable on total score'),
                                            p('Total scores are plotted according to criterion variable. Boxplot or scatterplot is displayed
                                              depending on outcome variable - whether it is discrete or continuous. Scatterplot is
                                              provided with red linear regression line. '),
                                            plotOutput('validity_plot'),
                                            downloadButton(outputId = "DB_validity_plot", label = "Download figure"),
                                            h4('Correlation of criterion variable and total score'),
                                            p('Test for association between total score and criterion variable is based on Spearman`s \\(\\rho\\).
                                              This rank-based measure has been recommended if bivariate normal distribution is not guaranteed.
                                              The null hypothesis is that correlation is 0. '),
                                            tableOutput('validity_table'),
                                            htmlOutput('validity_table_interpretation'),
                                            h4("Selected R code"),
                                            div(code('library(ShinyItemAnalysis)'),
                                                br(),
                                                code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data01  <- GMAT[, 1:20]'),
                                                br(),
                                                code('score  <- apply(data01, 1, sum)'),
                                                br(),
                                                code('criterion  <- GMAT[, "criterion"]'),
                                                br(),
                                                br(),
                                                code('# Descriptive plots'),
                                                br(),
                                                code('### boxplot'),
                                                br(),
                                                code('boxplot(criterion ~ score)'),
                                                br(),
                                                code('### scatterplot'),
                                                br(),
                                                code('plot(criterion ~ score)'),
                                                br(),
                                                code('abline(lm(criterion ~ score), col = "red") '),
                                                br(),
                                                br(),
                                                code('# Correlation'),
                                                br(),
                                                code('cor.test(criterion, score, method = "spearman", exact = F)')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Predictive validity'),
                                            p('This section requires criterion variable (e.g. future study success or future GPA in case
                                              of admission tests) which should be predicted by the measurement. This outcome variable
                                              can be uploaded in ', strong('Data'), 'section. Here you can explore how data predict this
                                              variable item by item. '),
                                            p('In distractor analysis based on criterion variable, we are interested in how test takers
                                              select the correct answer and how the distractors (wrong answers) with respect to group based
                                              on criterion variable.'),
                                            h4('Distractor plot'),
                                            htmlOutput("validity_distractor_text"),
                                            p('With option ', strong('Combinations'), 'all item selection patterns are plotted (e.g. AB, ACD, BC). With
                                              option', strong('Distractors'), 'answers are splitted into distractors (e.g. A, B, C, D).'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = 'validity_group',
                                                                      label = 'Number of groups:',
                                                                      min   = 1,
                                                                      max   = 5,
                                                                      value = 3)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 15%; ",
                                                          radioButtons(inputId = 'type_validity_combinations_distractor',
                                                                       label = 'Type',
                                                                       choices = list("Combinations", "Distractors"))),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "validitydistractorSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            plotOutput('validity_distractor_plot'),
                                            downloadButton(outputId = "DB_validity_distractor_plot", label = "Download figure"),
                                            h4('Correlation of criterion variable and scored item'),
                                            p('Test for association between total score and criterion variable is based on Spearman`s \\(\\rho\\).
                                              This rank-based measure has been recommended if bivariate normal distribution is not guaranteed.
                                              The null hypothesis is that correlation is 0. '),
                                            tableOutput('validity_table_item'),
                                            htmlOutput('validity_table_item_interpretation'),
                                            h4("Selected R code"),
                                            div(code('library(ShinyItemAnalysis)'),
                                                br(),
                                                code('library(difNLR)'),
                                                br(),
                                                code('data("GMAT", "GMATtest", "GMATkey")'),
                                                br(),
                                                code('data  <- GMATtest[, 1:20]'),
                                                br(),
                                                code('data01  <- GMAT[, 1:20]'),
                                                br(),
                                                code('key  <- GMATkey'),
                                                br(),
                                                code('criterion  <- GMAT[, "criterion"]'),
                                                br(),
                                                br(),
                                                code('# Distractor plot for item 1 and 3 groups'),
                                                br(),
                                                code('plotDistractorAnalysis(data, key, num.groups = 3, item = 1, matching = criterion)'),
                                                br(),
                                                br(),
                                                code('# Correlation for item 1'),
                                                br(),
                                                code('cor.test(criterion, data01[, 1], method = "spearman", exact = F)')),
                                            br()
                                   )
                                            ))),

             #%%%%%%%%%%%%%%%%%%%%%
             # ITEM ANALYSIS ######
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu('Item analysis',
                        # * TRADITIONAL ITEM ANALYSIS ####
                        tabPanel("Traditional item analysis",
                                 h3("Traditional item analysis"),
                                 p('Traditional item analysis uses proportions of correct answers or correlations to estimate item properties.'),
                                 h4("Item difficulty/discrimination plot"),
                                 p("Displayed is difficulty (red) and discrimination (blue)
                                   for all items. Items are ordered by difficulty. ", br(),
                                   strong("Difficulty"),' of items is estimated as percent of students who
                                   answered correctly to that item.', br(),
                                   strong("Discrimination"),' is by default described by difference of percent correct
                                   in upper and lower third of students (Upper-Lower Index, ULI). By rule of
                                   thumb it should not be lower than 0.2 (borderline in the plot), except for
                                   very easy or very difficult items. Discrimination can be customized (see also Martinkova, Stepanek, et al.
                                   (2017)) by changing number of groups and by changing which groups should be compared: '),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = 'DDplotNumGroupsSlider',
                                                           label = 'Number of groups:',
                                                           min   = 1,
                                                           max   = 5,
                                                           value = 3)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "DDplotRangeSlider",
                                                           label = "Which two groups to compare:",
                                                           min = 1,
                                                           max = 3,
                                                           step = 1,
                                                           value = c(1, 3)))),
                                 htmlOutput("DDplot_text"),
                                 br(),
                                 plotOutput('DDplot'),
                                 downloadButton("DB_DDplot", label = "Download figure"),
                                 h4("Cronbach's alpha"),
                                 p("Chronbach's alpha is an estimate of the reliability of a psychometric test. It is a function
                                   of the number of items in a test, the average covariance between item-pairs, and the variance
                                   of the total score (Cronbach, 1951)."),
                                 tableOutput('cronbachalpha_table'),
                                 h4("Traditional item analysis table"),
                                 htmlOutput("itemanalysis_table_text"),
                                 tableOutput('itemanalysis_table'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(psych)'),
                                     br(),
                                     code('library(psychometric)'),
                                     br(),
                                     code('library(ShinyItemAnalysis)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Difficulty and discrimination plot'),
                                     br(),
                                     code('DDplot(data, k = 3, l = 1, u = 3)'),
                                     br(),
                                     br(),
                                     code('# Cronbach alpha'),
                                     br(),
                                     code('psych::alpha(data)'),
                                     br(),
                                     br(),
                                     code('# Table'),
                                     br(),
                                     code('tab <- round(data.frame(item.exam(data, discr = TRUE)[, c(4, 1, 5, 2, 3)],
                                          psych::alpha(data)$alpha.drop[, 1], gDiscrim(data, k = 3, l = 1, u = 3)), 2)'),
                                     br(),
                                     code('colnames(tab) <- c("Difficulty", "SD", "Dsicrimination ULI", "Discrimination RIT", "Discrimination RIR", "Alpha Drop", "Customized Discrimination")'),
                                     br(),
                                     code('tab')),
                                 br()
                                 ),
                        # * DISTRACTORS ####
                        tabPanel("Distractors",
                                 h3("Distractor analysis"),
                                 p('In distractor analysis, we are interested in how test takers select
                                   the correct answer and how the distractors (wrong answers) were able
                                   to function effectively by drawing the test takers away from the correct answer.'),
                                 h4("Distractors plot"),
                                 htmlOutput("distractor_text"),
                                 p('With option ', strong('Combinations'), 'all item selection patterns are plotted (e.g. AB, ACD, BC). With
                                   option', strong('Distractors'), 'answers are splitted into distractors (e.g. A, B, C, D).'),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = 'gr',
                                                           label = 'Number of groups:',
                                                           min   = 1,
                                                           max   = 5,
                                                           value = 3)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 15%; ",
                                               radioButtons(inputId = 'type_combinations_distractor',
                                                            label = 'Type',
                                                            choices = list("Combinations", "Distractors"))),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "distractorSlider",
                                                           label = "Item",
                                                           min = 1,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1,
                                                           animate = TRUE))),
                                 plotOutput('distractor_plot'),
                                 downloadButton("DB_distractor_plot", label = "Download figure"),
                                 br(),
                                 h4("Table with counts"),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_counts'))),
                                 h4("Table with proportions"),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_proportions'))),
                                 br(),
                                 h4('Barplot of item response patterns'),
                                 plotOutput("distractor_barplot_item_response_patterns"),
                                 downloadButton( "DB_distractor_barplot_item_response_patterns", label = "Download figure"),
                                 h4('Histogram of total scores'),
                                 plotOutput('distractor_histogram'),
                                 downloadButton("DB_distractor_histogram", label = "Download figure"),
                                 br(),
                                 h4('Table of total scores by groups'),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_total_score_by_group'))),
                                 br(),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(ShinyItemAnalysis)'),
                                     br(),
                                     code('data(GMATtest)'),
                                     br(),
                                     code('data  <- GMATtest[, colnames(GMATtest) != "group"]'),
                                     br(),
                                     code('data(GMATkey)'),
                                     br(),
                                     code('key  <- GMATkey'),
                                     br(),
                                     br(),
                                     code('# Combinations - plot for item 1 and 3 groups'),
                                     br(),
                                     code('plotDistractorAnalysis(data, key, num.group = 3, item = 1,
                                          multiple.answers = T)'),
                                     br(),
                                     code('# Distractors - plot for item 1 and 3 groups'),
                                     br(),
                                     code('plotDistractorAnalysis(data, key, num.group = 3, item = 1,
                                          multiple.answers = F)'),
                                     br(),
                                     code('# Table with counts and margins - item 1 and 3 groups'),
                                     br(),
                                     code('DA <- DistractorAnalysis(data, key, num.groups = 3)[[1]]'),
                                     br(),
                                     code('dcast(as.data.frame(DA), response ~ score.level, sum, margins = T, value.var = "Freq")'),
                                     br(),
                                     code('# Table with proportions - item 1 and 3 groups'),
                                     br(),
                                     code('DistractorAnalysis(data, key, num.groups = 3, p.table = T)[[1]]'),
                                     br(),
                                     code('tab')),
                                 br()
                                 )
                        ),

             #%%%%%%%%%%%%%%%%%%%%%
             # REGRESSION #########
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Regression",
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 h3("Logistic regression on total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'),'can model dependency of probability of correct answer on total score by
                                   S-shaped logistic curve. Parameter', strong( "b0"),' describes horizontal position of the fitted curve,
                                   parameter ', strong( 'b1'),' describes its slope.'),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to total score.
                                   Their size is determined by count of respondents who achieved given level of
                                   total score.'),
                                 sliderInput("logregSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('logreg_plot'),
                                 downloadButton("DB_logreg_plot", label = "Download figure"),
                                 h4("Equation"),
                                 withMathJax(),
                                 ('$$\\mathrm{P}(Y = 1|X, b_0, b_1) = \\mathrm{E}(Y|X, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 X\\right)}}{1+e^{\\left( b_{0} + b_1 X\\right) }} $$'),

                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('logreg_table'))),
                                 htmlOutput("logreg_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     code('score  <- apply(data, 1, sum)'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ score, family = binomial)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, b0, b1){exp(b0 + b1 * x) / (1 + exp(b0 + b1 * x))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, b0 = coef(fit)[1], b1 = coef(fit)[2]), 0, 20,
                                          xlab = "Total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        # * LOGISTIC Z ####
                        tabPanel("Logistic Z",
                                 h3("Logistic regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by S-shaped logistic curve. Parameter ', strong( 'b0'), ' describes
                                   horizontal position of the fitted curve (difficulty), parameter ', strong('b1'),' describes its slope at
                                   inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput("zlogregSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('z_logreg_plot'),
                                 downloadButton("DB_z_logreg_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1) = \\mathrm{E}(Y|Z, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 Z\\right) }}{1+e^{\\left( b_{0} + b_1 Z\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('z_logreg_table'))),
                                 htmlOutput("z_logreg_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ stand.score, family = binomial)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, b0, b1){exp(b0 + b1 * x) / (1 + exp(b0 + b1 * x))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, b0 = coef(fit)[1], b1 = coef(fit)[2]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        # * LOGISTIC IRT Z ####
                        tabPanel("Logistic IRT Z",
                                 h3("Logistic regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. Note change in parametrization - the IRT parametrization
                                   used here corresponds to the parametrization used in IRT models.
                                   Parameter', strong('b') , 'describes horizontal position of the fitted curve (difficulty),
                                   parameter' , strong('a') , ' describes its slope at inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput("zlogreg_irtSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('z_logreg_irt_plot'),
                                 downloadButton("DB_z_logreg_irt_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, a, b) = \\mathrm{E}(Y|Z, a, b) = \\frac{e^{ a\\left(Z - b\\right) }}{1+e^{a\\left(Z - b\\right)}} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('z_logreg_irt_table'))),
                                 htmlOutput("z_logreg_irt_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, 1:20]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ stand.score, family = binomial)'),
                                     br(),
                                     code('# Coefficients - tranformation'),
                                     br(),
                                     code('coef <- c(a = coef(fit)[2], b = - coef(fit)[1] / coef(fit)[2])'),
                                     br(),
                                     code('coef'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, a, b){exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, a = coef[1], b = coef[2]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),

                        # * NONLINEAR Z ####
                        tabPanel("Nonlinear IRT Z",
                                 h3("Nonlinear regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Nonlinear regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. The IRT parametrization used here corresponds
                                   to the parametrization used in IRT models. Parameter ', strong( 'b'),' describes horizontal position of the fitted curve (difficulty),
                                   parameter ',strong( 'a'), ' describes its slope at inflection point (discrimination). This model allows for nonzero lower left
                                   asymptote ', strong( 'c'), ' (pseudo-guessing). '),
                                 br(),
                                 h4("Plot with estimated nonlinear curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput("nlsSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('nlr_plot'),
                                 downloadButton("DB_nlr_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1, c) = \\mathrm{E}(Y|Z, b_0, b_1, c) = c + \\left( 1-c \\right) \\cdot \\frac{e^{a\\left(Z-b\\right) }}{1+e^{a\\left(Z-b\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('nlr_table'))),
                                 htmlOutput("nlr_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('Data  <- GMAT[, 1:20]'),
                                     br(),
                                     code('stand.score  <- scale(apply(Data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# NLR model for item 1'),
                                     br(),
                                     code('fun <- function(x, a, b, c){c + (1 - c) * exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('fit <- nls(data[, 1] ~ fun(stand.score, a, b, c), algorithm = "port",
                                          start = startNLR(data, GMAT[, "group"], model = "3PLcg")[1, 1:3])'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, a = coef(fit)[1], b = coef(fit)[2], c = coef(fit)[3]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        "----",
                        # * MODELS COMPARISON ####
                        tabPanel("Model comparison",
                                 h3("Logistic regression model selection"),
                                 p('Here you can compare classic 2PL logistic regression model to non-linear model
                                   item by item using some information criterions: '),
                                 tags$ul(
                                   tags$li(strong('AIC'), 'is the Akaike information criterion (Akaike, 1974), '),
                                   tags$li(strong('BIC'), 'is the Bayesian information criterion (Schwarz, 1978)')
                                 ),
                                 p('Another approach to nested models can be likelihood ratio chi-squared test.
                                   Significance level is set to 0.05. As tests are performed item by item, it is
                                   possible to use multiple comparison correction method. '),
                                 selectInput("correction_method_regrmodels", "Correction method",
                                             c("BH" = "BH",
                                               "Holm" = "holm",
                                               "Hochberg" = "hochberg",
                                               "Hommel" = "hommel",
                                               "BY" = "BY",
                                               "FDR" = "fdr",
                                               "none" = "none"),
                                             selected="none"),
                                 h4("Table of comparison statistics"),
                                 p('Rows ', strong('BEST'), 'indicate which model has the lowest value of criterion, or is the largest
                                   significant model by likelihood ratio test.'),
                                 DT::dataTableOutput('regr_comp_table'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('Data  <- GMAT[, 1:20]'),
                                     br(),
                                     code('stand.score  <- scale(apply(Data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Fitting models'),
                                     br(),
                                     code('fun <- function(x, a, b, c){c + (1 - c) * exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('# 2PL model for item 1'),
                                     br(),
                                     code('fit2PL <- nls(Data[, 1] ~ fun(stand.score, a, b, c = 0), algorithm = "port",
                                          start = startNLR(Data, GMAT[, "group"], model = "3PLcg")[1, 1:2])'),
                                     br(),
                                     code('# 3PL model for item 1'),
                                     br(),
                                     code('fit3PL <- nls(Data[, 1] ~ fun(stand.score, a, b, c), algorithm = "port",
                                          start = startNLR(Data, GMAT[, "group"], model = "3PLcg")[1, 1:3])'),
                                     br(),
                                     br(),
                                     code('# Comparison'),
                                     br(),
                                     code('AIC(fit2PL); AIC(fit3PL)'),
                                     br(),
                                     code('BIC(fit2PL); BIC(fit3PL)'),
                                     br(),
                                     code('LRstat <- -2 * (sapply(fit2PL, logLik) - sapply(fit3PL, logLik))'),
                                     br(),
                                     code('LRdf <- 1'),
                                     br(),
                                     code('LRpval <- 1 - pchisq(LRstat, LRdf)'),
                                     br(),
                                     code('LRpval <- p.adjust(LRpval, method = "BH")'),
                                     br()),
                                 br()
                                 ),
                        "----",
                        # * MULTINOMIAL ####
                        tabPanel("Multinomial",
                                 h3("Multinomial regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Multinomial regression'),'allows for simultaneous modelling of probability of choosing
                                   given distractors on standardized total score (Z-score).'),
                                 br(),
                                 h4("Plot with estimated curves of multinomial regression"),
                                 p('Points represent proportion of selected option with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score and who selected given option.'),
                                 sliderInput("multiSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('multi_plot'),
                                 downloadButton("DB_multi_plot", label = "Download figure"),
                                 h4("Equation"),
                                 uiOutput('multi_equation'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('multi_table'))),
                                 strong("Interpretation:"),
                                 htmlOutput("multi_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(nnet)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data.scored  <- GMAT[, 1:20]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data.scored, 1, sum))'),
                                     br(),
                                     code('data(GMATtest)'),
                                     br(),
                                     code('data  <- GMATtest[, colnames(GMATtest) != "group"]'),
                                     br(),
                                     code('data(GMATkey)'),
                                     br(),
                                     code('key  <- GMATkey'),

                                     br(),
                                     br(),
                                     code('# multinomial model for item 1'),
                                     br(),
                                     code('fit <- multinom(relevel(data[, 1], ref = paste(key[1])) ~ stand.score)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)')),
                                 br()
                                 )
                        ),
             #%%%%%%%%%%%%%%%%%%%%%
             # IRT MODELS #########
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("IRT models",
                        "Used methods",
                        # * RASCH ####
                        tabPanel("Rasch",
                                 h3("Rasch Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('In',
                                   strong('Rasch IRT model'), '(Rasch, 1960), all items are assumed to have the same slope in inflection point – the
                                   same discrimination', strong('a'), 'which is fixed to value of 1. Items can differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, b_{j} \\right) =  \\frac{e^{\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('rasch_mirt'),
                                 downloadButton("DP_rasch_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('raschiic_mirt'),
                                 downloadButton("DP_raschiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('raschtif_mirt'),
                                 downloadButton("DP_raschtif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015).
                                   SX2 is computed only when no missing data are present. In such a case consider using imputed dataset!'),
                                 tableOutput('raschcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('raschFactorCor_mirt'),
                                 plotOutput('raschFactor_mirt'),
                                 downloadButton("DP_raschFactor_mirt", label = "Download figure"),
                                 h4('Wright map'),
                                 p('Wright map (Wilson, 2005; Wright & Stone, 1979), also called item-person map, is a graphical tool
                                   to display person estimates and item parameters. The person side
                                   (left) represents histogram of estimated knowledge of students.
                                   The item side (right) displays estimates of difficulty of particular items. '),
                                 plotOutput('raschWrightMap_mirt'),
                                 downloadButton('DP_raschWM_mirt', label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('library(WrightMap)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "Rasch", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     br(),
                                     code('# Wright Map'),
                                     br(),
                                     code('b <- sapply(1:ncol(data), function(i) coef(fit)[[i]][, "d"])'),
                                     br(),
                                     code('wrightMap(fs, b, item.side = itemClassic)')),
                                 br()
                                 ),
                        # * 1 PL IRT ####
                        tabPanel("1PL",
                                 h3("One parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('In',
                                   strong('1PL IRT model'), ', all items are assumed to have the same slope in inflection point – the
                                   same discrimination', strong('a'), '. Items can differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a, b_{j} \\right) =  \\frac{e^{a\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('oneparamirt_mirt'),
                                 downloadButton("DP_oneparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('oneparamirtiic_mirt'),
                                 downloadButton("DP_oneparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('oneparamirttif_mirt'),
                                 downloadButton("DP_oneparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics
                                   (Ames & Penfield, 2015). SX2 is computed only when no missing data are present.
                                   In such a case consider using imputed dataset!'),
                                 tableOutput('oneparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('oneparamirtFactorCor_mirt'),
                                 plotOutput('oneparamirtFactor_mirt'),
                                 downloadButton("DP_oneparamirtFactor_mirt", label = "Download figure"),
                                 h4('Wright map'),
                                 p('Wright map (Wilson, 2005; Wright & Stone, 1979), also called item-person map, is a graphical tool
                                   to display person estimates and item parameters. The person side
                                   (left) represents histogram of estimated knowledge of students.
                                   The item side (right) displays estimates of difficulty of particular items. '),
                                 plotOutput('oneparamirtWrightMap_mirt'),
                                 downloadButton('DP_oneparamirtWM_mirt', label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('library(WrightMap)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "2PL", constrain = list((1:ncol(data)) + seq(0, (ncol(data) - 1)*3, 3)), SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     br(),
                                     code('# Wright Map'),
                                     br(),
                                     code('b <- sapply(1:ncol(data), function(i) coef(fit)[[i]][, "d"])'),
                                     br(),
                                     code('wrightMap(fs, b, item.side = itemClassic)'),
                                     br(),
                                     br(),
                                     br(),
                                     code('# You can also use ltm library for IRT models'),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- rasch(data)'),
                                     br(),
                                     code('# for Rasch model use'),
                                     br(),
                                     code('# fit <- rasch(data, constraint = cbind(ncol(data) + 1, 1))'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')
                                    ),
                                 br()
                                 ),
                        # * 2PL ####
                        tabPanel("2PL ",
                                 h3("Two parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('2PL IRT model'), ' allows for different slopes in inflection point – different
                                   discriminations', strong('a.'), 'Items can also differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}\\right) =  \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('twoparamirt_mirt'),
                                 downloadButton("DP_twoparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('twoparamirtiic_mirt'),
                                 downloadButton("DP_twoparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('twoparamirttif_mirt'),
                                 downloadButton("DP_twoparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015).
                                   SX2 is computed only when no missing data are present. In such a case consider using imputed dataset!'),
                                 tableOutput('twoparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('twoparamirtFactorCor_mirt'),
                                 plotOutput('twoparamirtFactor_mirt'),
                                 downloadButton("DP_twoparamirtFactor_mirt", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "2PL", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     br(),
                                     br(),
                                     code('# You can also use ltm library for IRT models'),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- ltm(data ~ z1, IRT.param = TRUE)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')),
                                 br()
                                 ),
                        # * 3PL ####
                        tabPanel("3PL ",
                                 h3("Three parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('3PL IRT model'), ' allows for different discriminations of items', strong('a,'),
                                   'different item difficulties',
                                   strong('b,'), 'and allows also for nonzero left asymptote – pseudo-guessing', strong('c.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}, c_{j} \\right) = c_{j} + \\left(1 - c_{j}\\right) \\cdot \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('threeparamirt_mirt'),
                                 downloadButton("DP_threeparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('threeparamirtiic_mirt'),
                                 downloadButton("DP_threeparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('threeparamirttif_mirt'),
                                 downloadButton("DP_threeparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015).
                                   SX2 is computed only when no missing data are present. In such a case consider using imputed dataset!'),
                                 tableOutput('threeparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('threeparamirtFactorCor_mirt'),
                                 plotOutput('threeparamirtFactor_mirt'),
                                 downloadButton("DP_threeparamirtFactor_mirt", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "3PL", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     code('# You can also use ltm library for IRT models'),
                                     br(),
                                     br(),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- tpm(data, IRT.param = TRUE)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')),
                                 br()
                                 ),
                        "----",
                        # * MODEL COMPARISON ####
                        tabPanel("Model comparison ",
                                 h3("Item Response Theory model selection"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('IRT models can be compared by several information criterions: '),
                                 tags$ul(
                                   tags$li(strong('AIC'), 'is the Akaike information criterion (Akaike, 1974), '),
                                   tags$li(strong('AICc'), 'is AIC with a correction for finite sample size, '),
                                   tags$li(strong('BIC'), 'is the Bayesian information criterion (Schwarz, 1978).'),
                                   tags$li(strong('SABIC'), 'is the Sample-sized adjusted BIC criterion, ')
                                 ),
                                 p('Another approach to compare IRT models can be likelihood ratio chi-squared test.
                                   Significance level is set to 0.05.'),
                                 h4("Table of comparison statistics"),
                                 p('Row ', strong('BEST'), 'indicates which model has the lowest value of criterion, or is the largest
                                   significant model by likelihood ratio test.'),
                                 tableOutput('irtcomparison'),
                                 tags$style(type = "text/css", "#irtcomparison tr:last-child {font-weight:bold;}"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT model'),
                                     br(),
                                     code('fit1PL <- mirt(data, model = 1, itemtype = "3PL", SE = T)'),
                                     br(),
                                     code('# 2PL IRT model'),
                                     br(),
                                     code('fit2PL <- mirt(data, model = 1, itemtype = "2PL")'),
                                     br(),
                                     code('# 3PL IRT model'),
                                     br(),
                                     code('fit3PL <- mirt(data, model = 1, itemtype = "3PL")'),
                                     br(),
                                     br(),
                                     code('# Comparison'),
                                     br(),
                                     code('anova(fit1PL, fit2PL)'),
                                     br(),
                                     code('anova(fit2PL, fit3PL)')),
                                 br()
                                 ),
                        "----",
                        # * BOCK'S NOMINAL MODEL ####
                        tabPanel("Bock's nominal model",
                                 h3("Bock's nominal Item Response Theory model"),
                                 p('The nominal response model (NRM) was introduced by Bock (1972) as a way to model
                                   responses to items with two or more nominal categories. This model is suitable for
                                   multiple-choice items with no particular ordering of distractors. The correct answer
                                   represent the highest category, in terms of the measured latent trait. '),
                                 h4('Equation'),
                                 withMathJax('For ', strong('K'), ' possible test choices is the probability of the choice ', strong('k'), ' for
                                   person ', strong('i'), ' with latent trait', strong('\\(\\theta\\)'), ' in item ', strong('j'),
                                   'given by the following equation: '),
                                 ('$$\\mathrm{P}(Y_{ij} = k|\\theta_i, a_{j1}, al_{j(l-1)}, d_{j(l-1)}, l = 1, \\dots, K) =
                                             \\frac{e^{(ak_{j(k-1)} * a_{j1} * \\theta_i + d_{j(k-1)})}}{\\sum_l e^{(al_{j(l-1)} * a_{j1} * \\theta_i + d_{j(l-1)})}}$$'),
                                 br(),
                                 h4("Item characteristic curves"),
                                 plotOutput('bock_CC'),
                                 downloadButton("DP_bock_CC", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('bock_IIC'),
                                 downloadButton("DP_bock_IIC", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('bock_TIF'),
                                 downloadButton("DP_bock_TIF", label = "Download figure"),
                                 h4("Table of parameters"),
                                 textOutput("bock_coef_warning"),
                                 tableOutput('bock_coef'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('bockFactorCorInput_mirt'),
                                 plotOutput('bock_factor'),
                                 downloadButton("DP_bock_factor", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "nominal")'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)')),
                                 br()
                                 ),
                        "----",
                        "Training",
                        # * ITEM PLOTS ####
                        tabPanel("Characteristic and information curves",
                                 h3("Characteristic and information curves"),
                                 p('Here you can explore behaviour of two item characteristic curves \\(\\mathrm{P}\\left(\\theta\\right)\\) and their item
                                   information functions \\(\\mathrm{I}\\left(\\theta\\right)\\) in 4PL IRT model. '),
                                 h4("Parameters"),
                                 p('Select parameters ', strong('a'), '(discrimination), ', strong('b'), '(difficulty), ',
                                   strong('c'), '(guessing) and ', strong('d'), '(inattention). By constraining a = 1, c = 0, d = 1 you get
                                   Rasch model. With option c = 0 and d = 1 you get 2PL model and with option d = 1 3PL model.'),
                                 p('When you set different curve parameters, you can follow a phenomenon called Differential Item Functioning (DIF). See further
                                   section for more information. '),
                                 # be careful about the order of sliders!!!
                                 # this probably has nicer solution
                                 tags$style(HTML(".js-irs-12 .irs-single, .js-irs-12 .irs-bar-edge, .js-irs-12 .irs-bar {
                                                  background: red;
                                                  border-top-color: red;
                                                  border-bottom-color: red;
                                                  border-left-color: red;
                                                  border-right-color: red}")),
                                 tags$style(HTML(".js-irs-13 .irs-single, .js-irs-13 .irs-bar-edge, .js-irs-13 .irs-bar {
                                                  background: red;
                                                  border-top-color: red;
                                                  border-bottom-color: red;
                                                  border-left-color: red;
                                                  border-right-color: red}")),
                                 tags$style(HTML(".js-irs-14 .irs-single, .js-irs-14 .irs-bar-edge, .js-irs-14 .irs-bar {
                                                  background: red;
                                                  border-top-color: red;
                                                  border-bottom-color: red;
                                                  border-left-color: red;
                                                  border-right-color: red}")),
                                 tags$style(HTML(".js-irs-15 .irs-single, .js-irs-15 .irs-bar-edge, .js-irs-15 .irs-bar {
                                                  background: red;
                                                  border-top-color: red;
                                                  border-bottom-color: red;
                                                  border-left-color: red;
                                                  border-right-color: red}")),
                                 tags$style(HTML(".js-irs-16 .irs-single, .js-irs-16 .irs-bar-edge, .js-irs-16 .irs-bar {
                                                  background: blue;
                                                  border-top-color: blue;
                                                  border-bottom-color: blue;
                                                  border-left-color: blue;
                                                  border-right-color: blue}")),
                                 tags$style(HTML(".js-irs-17 .irs-single, .js-irs-17 .irs-bar-edge, .js-irs-17 .irs-bar {
                                                  background: blue;
                                                  border-top-color: blue;
                                                  border-bottom-color: blue;
                                                  border-left-color: blue;
                                                  border-right-color: blue}")),
                                 tags$style(HTML(".js-irs-18 .irs-single, .js-irs-18 .irs-bar-edge, .js-irs-18 .irs-bar {
                                                  background: blue;
                                                  border-top-color: blue;
                                                  border-bottom-color: blue;
                                                  border-left-color: blue;
                                                  border-right-color: blue}")),
                                 tags$style(HTML(".js-irs-19 .irs-single, .js-irs-19 .irs-bar-edge, .js-irs-19 .irs-bar {
                                                  background: blue;
                                                  border-top-color: blue;
                                                  border-bottom-color: blue;
                                                  border-left-color: blue;
                                                  border-right-color: blue}")),
                                 tags$style(HTML(".js-irs-20 .irs-single, .js-irs-20 .irs-bar-edge, .js-irs-20 .irs-bar {
                                                  background: gray;
                                                  border-top-color: gray;
                                                  border-bottom-color: gray;
                                                  border-left-color: gray;
                                                  border-right-color: gray}")),
                                 fluidRow(
                                   column(12,
                                          splitLayout(
                                            cellWidths = c("16%", "1%", "8%", "16%", "1%", "8%", "16%", "1%", "8%", "16%", "1%", "8%"),
                                            sliderInput("ccIRTSlider_a1", "a - discrimination", min = -4, max = 4,
                                                        value = 1, step = 0.1),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_a1", "", value = 1)),
                                            sliderInput("ccIRTSlider_b1", "b - difficulty", min = -4, max = 4,
                                                        value = 0, step = 0.1),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_b1", "", value = 0)),
                                            sliderInput("ccIRTSlider_c1", "c - guessing", min = 0, max = 1,
                                                        value = 0, step = 0.01),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_c1", "", value = 0)),
                                            sliderInput("ccIRTSlider_d1", "d - inattention", min = 0, max = 1,
                                                        value = 1, step = 0.01),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_d1", "", value = 1))))),
                                 fluidRow(
                                   column(12,
                                          splitLayout(
                                            cellWidths = c("16%", "1%", "8%", "16%", "1%", "8%", "16%", "1%", "8%", "16%", "1%", "8%"),
                                            sliderInput("ccIRTSlider_a2", "a - discrimination", min = -4, max = 4,
                                                        value = 2, step = 0.1),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_a2", "", value = 2)),
                                            sliderInput("ccIRTSlider_b2", "b - difficulty", min = -4, max = 4,
                                                        value = 0.5, step = 0.1),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_b2", "", value = 0.5)),
                                            sliderInput("ccIRTSlider_c2", "c - guessing", min = 0, max = 1,
                                                        value = 0, step = 0.01),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("&ensp;")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_c2", "", value = 0)),
                                            sliderInput("ccIRTSlider_d2", "d - inattention", min = 0, max = 1,
                                                        value = 1, step = 0.01),
                                            div(style="display: inline-block; vertical-align: middle; width: 50%;", HTML("&ensp;")),
                                            div(style= "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_d2", "", value = 1))))),
                                 p("Select also the value of latent ability", HTML("<b>&#952;</b>"), "to see the intepretation of the
                                   item characteristic curves. "),
                                 fluidRow(
                                   column(3,
                                          splitLayout(
                                            cellWidths = c("68%", "6%", "36%"),
                                            sliderInput("ccIRTSlider_theta", div(HTML('&#952;'), "- latent ability"), min = -4, max = 4,
                                                        value = 0, step = 0.1),
                                            div(style = "display: inline-block; vertical-align: middle; width: 50%;", HTML("<br>")),
                                            div(style = "display: inline-block; vertical-align: middle; height: 100%; width: 50%",
                                                textInput("ccIRTtext_theta", "", value = 0))))),
                                 h4("Equations"),
                                 ('$$\\mathrm{P}\\left(\\theta \\vert a, b, c, d \\right) = c + \\left(d - c\\right) \\cdot \\frac{e^{a\\left(\\theta-b\\right) }}{1+e^{a\\left(\\theta-b\\right) }} $$'),
                                 ('$$\\mathrm{I}\\left(\\theta \\vert a, b, c, d \\right) = a^2 \\cdot \\left(d - c\\right) \\cdot \\frac{e^{a\\left(\\theta-b\\right) }}{\\left[1+e^{a\\left(\\theta-b\\right)}\\right]^2} $$'),
                                 uiOutput("ccIRT_interpretation"),
                                 br(),
                                 splitLayout(cellWidths = c("50%", "50%"), plotlyOutput('ccIRT_plot'), plotlyOutput('iccIRT_plot')),
                                 splitLayout(cellWidths = c("50%", "50%"), downloadButton("DB_ccIRT", label = "Download figure"), downloadButton("DB_iccIRT", label = "Download figure")),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(ggplot2)'),
                                     br(),
                                     code('# parameters'),
                                     br(),
                                     code('a1 <- 1; b1 <- 0; c1 <- 0; d1 <- 1'),
                                     br(),
                                     code('a2 <- 2; b2 <- 0.5; c2 <- 0; d2 <- 1'),
                                     br(),
                                     code('# latent ability'),
                                     br(),
                                     code('theta <- 0'),
                                     br(),
                                     code('# function for IRT characteristic curve'),
                                     br(),
                                     code('ccirt <- function(theta, a, b, c, d){
                                          return(c + (d - c)/(1 + exp(-a*(theta - b))))
                                          }'),
                                 br(),
                                     code('# plot for characteristic curves'),
                                 br(),
                                     HTML('<code>g <- ggplot(data = data.frame(x = -4:4), mapping = aes(x = x)) +<br>
                                          stat_function(fun = ccirt, args = list(a = a1, b = b1, c = c1, d = d1), aes(color = "2", linetype = "2")) +<br>
                                          stat_function(fun = ccirt, args = list(a = a2, b = b2, c = c2, d = d2), aes(color = "3", linetype = "3")) +<br>
                                          geom_segment(aes(y = ccirt(theta, a = a1, b = b1, c = c1, d = d1), yend = ccirt(theta, a = a1, b = b1, c = c1, d = d1), x = -4, xend = theta, color = "1", linetype = "1")) +<br>
                                          geom_segment(aes(y = ccirt(theta, a = a2, b = b2, c = c2, d = d2), yend = ccirt(theta, a = a2, b = b2, c = c2, d = d2), x = -4, xend = theta, color = "1", linetype = "1")) +<br>
                                          geom_segment(aes(y = 0, yend = max(ccirt(theta, a = a1, b = b1, c = c1, d = d1), ccirt(theta, a = a2, b = b2, c = c2, d = d2)), x = theta, xend = theta, color = "1", linetype = "1")) +<br>
                                          xlim(-4, 4) +<br>
                                          xlab("Ability") +<br>
                                          ylab("Probability of correct answer") +<br>
                                          theme_bw() +<br>
                                          ylim(0, 1) +<br>
                                          theme(axis.line  = element_line(colour = "black"), panel.grid.major = element_blank(), panel.grid.minor = element_blank(), legend.position = c(0.97, 0.03), legend.justification = c(0.97, 0.03)) +<br>
                                          scale_color_manual(name = "", breaks = c("1", "2", "3"), values = c("gray", "red", "blue"), labels = c(paste(expression(theta), "=", theta), paste(paste(letters[1:4], "=", c(a1, b1, c1, d1)), collapse = ", "), paste(paste(letters[1:4], "=", c(a2, b2, c2, d2)), collapse = ", "))) +<br>
                                          scale_linetype_manual(name = "", breaks = c("1", "2", "3"), values = c("dashed", "solid", "solid"), labels = c(paste(expression(theta), "=", theta), paste(paste(letters[1:4], "=", c(a1, b1, c1, d1)), collapse = ", "), paste(paste(letters[1:4], "=", c(a2, b2, c2, d2)), collapse = ", "))) +<br>
                                          ggtitle("Item characteristic curve")</code>'),
                                     br(),
                                     code('g')),
                                 br()
                                 )
                                 ),
             #%%%%%%%%%%%%%%%%%%%%%
             # DIF/FAIRNESS #######
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("DIF/Fairness",
                        # * SUMMARY ####
                        "Description",
                        tabPanel('About DIF',
                                 h3('Differential Item Functioning / Item Fairness'),
                                 p('Differential item functioning (DIF) occurs when people from different
                                   groups (commonly gender or ethnicity) with the same underlying true
                                   ability have a different probability of answering the item correctly.
                                   If item functions differently for two groups, it is potentially unfair.
                                   In general, two type of DIF can be recognized: if the item has different
                                   difficulty for given two groups with the same discrimination, ',
                                   strong('uniform'), 'DIF is present (left figure). If the item has different
                                   discrimination and possibly also different difficulty for given two groups, ',
                                   strong('non-uniform'), 'DIF is present (right figure)'),
                                 br(),
                                 fluidRow(
                                   column(6, align="center",
                                          img(src = 'fig_NLR_uniformDIF.png', width = 380)),
                                   column(6, align="center",
                                          img(src = 'fig_NLR_nonuniformDIF.png', width = 380))
                                 ),
                                 br(),
                                 br()
                        ),
                        "----",
                        "Used methods",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Total scores"),
                                 p('DIF is not about total scores! Two groups may have the same distribution of total scores, yet,
                                   some item may function differently for two groups. Also, one of the groups may have signifficantly
                                   lower total score, yet, it may happen that there is no DIF item!'),
                                 h4("Summary of total scores for groups"),
                                 tableOutput('resultsgroup'),
                                 h4("Histograms of total scores for groups"),
                                 sliderInput("inSlider2group", "Cut-score", min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 p('For selected cut-score, blue part of histogram shows students with total score
                                   above the cut-score, grey column shows students with Total Score equal
                                   to cut-score and red part of histogram shows students below the cut-score.'),
                                 splitLayout(cellWidths = c("50%", "50%"), plotOutput('histbyscoregroup0'),plotOutput('histbyscoregroup1')),
                                 splitLayout(cellWidths = c("50%", "50%"), downloadButton("DP_histbyscoregroup0", label = "Download figure"),
                                 downloadButton("DP_histbyscoregroup1", label = "Download figure")),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Summary table'),
                                     br(),
                                     code('sc_zero <- apply(data[group == 0, ], 1, sum); summary(sc_zero) # total scores of reference group'),
                                     br(),
                                     code('sc_one  <- apply(data[group == 1, ], 1, sum); summary(sc_one)  # total scores of focal group'),
                                     br(),
                                     code('# Histograms'),
                                     br(),
                                     code('hist(sc_zero, breaks = 0:20)'),
                                     br(),
                                     code('hist(sc_one, breaks = 0:20)')),
                                 br()
                                 ),
                        # * DELTA PLOTS ####
                        tabPanel("Delta plots",
                                 h3("Delta plot"),
                                 p('Delta plot (Angoff & Ford, 1973) compares the proportions of correct answers per
                                   item in the two groups. It displays non-linear transformation of these proportions using
                                   quantiles of standard normal distributions (so called delta scores) for each item for the two
                                   genders in a scatterplot called diagonal plot or delta plot (see Figure). Item is under
                                   suspicion of DIF if the delta point considerably departs from the diagonal. The detection
                                   threshold is either fixed to value 1.5 or based on bivariate normal approximation (Magis &
                                   Facon, 2012).'),

                                 radioButtons('type_threshold', 'Threshold',
                                              list("Fixed", "Normal")
                                 ),
                                 checkboxInput('puri_DP', 'Item purification', FALSE),
                                 conditionalPanel(
                                   condition = "input.puri_DP",
                                   selectInput("puri_DP_type", "Purification method",
                                             c("IPP1" = "IPP1",
                                               "IPP2" = "IPP2",
                                               "IPP3" = "IPP3"
                                               ),
                                             selected = "IPP1")),
                                 plotOutput('deltaplot'),
                                 downloadButton("DP_deltaplot", label = "Download figure"),
                                 br(),
                                 br(),
                                 verbatimTextOutput("dp_text_normal"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(deltaPlotR)'),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Delta scores with fixed threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = 1.5)'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)'),
                                     br(),
                                     br(),
                                     code('# Delta scores with normal threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = "norm", purify = F)'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)')),
                                 br()
                                 ),
                        # * MANTEL-HAENSZEL ####
                        tabPanel("Mantel-Haenszel",
                                 tabsetPanel(
                                   # Summary
                                   tabPanel("Summary",
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            h4('Summary table'),
                                            p('Here you can select ', strong('correction method'), 'for multiple comparison or ',
                                              strong('item purification.')),
                                            selectInput("correction_method_MZ_print", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"
                                                        ),
                                                        selected = "none"),
                                            checkboxInput('puri_MH', 'Item purification', FALSE),
                                            verbatimTextOutput("print_DIF_MH"),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Mantel-Haenszel test'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   tabPanel('Items',
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            h4('Contingency tables and odds ratio calculation'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("difMHSlider_item",
                                                                      "Item",
                                                                      animate = TRUE,
                                                                      min = 1,
                                                                      max = 10,
                                                                      value = 1,
                                                                      step = 1)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("difMHSlider_score",
                                                                      "Cut-score",
                                                                      min = 0,
                                                                      max = 10,
                                                                      value = 1,
                                                                      step = 1))),
                                            fluidRow(column(12, align = "center", tableOutput('table_DIF_MH'))),
                                            uiOutput('ORcalculation'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Contingency table for item 1 and score 12'),
                                                br(),
                                                code('df <- data.frame(data[, 1], group)'),
                                                br(),
                                                code('colnames(df) <- c("Answer", "Group")'),
                                                br(),
                                                code('df$Answer <- relevel(factor(df$Answer, labels = c("Incorrect", "Correct")), "Correct")'),
                                                br(),
                                                code('df$Group <- factor(df$Group, labels = c("Reference Group", "Focal Group"))'),
                                                br(),
                                                code('score <- apply(data, 1, sum)'),
                                                br(),

                                                code('df <- df[score == 12, ]'),
                                                br(),

                                                code('tab <- dcast(data.frame(xtabs(~ Group + Answer, data = df)),
                                                     Group ~ Answer,
                                                     value.var = "Freq",
                                                     margins = T,
                                                     fun = sum)'),
                                                br(),
                                                code('tab'),
                                                br(),

                                                code('# Mantel-Haenszel estimate of OR'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit$alphaMH')),
                                            br()
                                            )
                        )
                        ),
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            h4("Summary table"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_logistic',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_LR',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_logistic'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_logistic',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_LR_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("diflogSlider", "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to total score.
                                              Their size is determined by count of respondents who achieved given level of
                                              total score with respect to the group membership.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            plotOutput('plot_DIF_logistic'),
                                            downloadButton("DP_plot_DIF_logistic", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = F,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$logitPar')),
                                            br()
                                            )
                                   )
                                 ),

                        # * LOGISTIC Z ####
                        tabPanel("Logistic IRT Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) = \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}{1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4('Summary table'),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison.'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_logistic_IRT_Z',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logZSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"))),
                                            verbatimTextOutput('print_DIF_logistic_IRT_Z'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison.'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_logistic_IRT_Z',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logZItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none")),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("diflog_irtSlider", "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score with respect to the group membership.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            plotOutput('plot_DIF_logistic_IRT_Z'),
                                            downloadButton("DP_plot_DIF_logistic_IRT_Z", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic_IRT_Z'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = T,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('# Coefficients for item 1 - recalculation'),
                                                br(),
                                                code('coef_old <- fit$logitPar[1, ]'),
                                                br(),
                                                code('coef <- c()'),
                                                br(),
                                                code('# a = b1, b = -b0/b1, adif = b3, bdif = -(b1b2-b0b3)/(b1(b1+b3))'),
                                                br(),
                                                code('coef[1] <- coef_old[2]'),
                                                br(),
                                                code('coef[2] <- -(coef_old[1] / coef_old[2])'),
                                                br(),
                                                code('coef[3] <- coef_old[4]'),
                                                br(),
                                                code('coef[4] <- -(coef_old[2] * coef_old[3] + coef_old[1] * coef_old[4] ) /
                                                     (coef_old[2] * (coef_old[2] + coef_old[4]))')),
                                            br()
                                                )
                                            )
                                   ),
                        # * NONLINEAR Z ####
                        tabPanel("Nonlinear IRT Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Nonlinear regression on standardized total scores with IRT parameterization'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c '),
                                              a('(Drabinova & Martinkova, 2017). ',
                                                href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                                                target = "_blank"),
                                              'Similarly to logistic regression, also nonlinear regression allows for
                                              detection of uniform and non-uniform DIF by adding a group specific intercept',
                                              strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'),
                                              '(non-uniform DIF) into the model and by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Summary table"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_NLR',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_nlrSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_NLR_print',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_NLR'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "none")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Nonlinear regression on standardized total scores with IRT parameterization'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c '),
                                              a('(Drabinova & Martinkova, 2017). ',
                                                href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                                                target = "_blank"),
                                              'Similarly to logistic regression, also nonlinear regression allows
                                              for detection of uniform and non-uniform DIF by adding a group specific intercept', strong('bDIF'), '(uniform DIF) and group specific
                                              interaction', strong('aDIF'), '(non-uniform DIF) into the model and by testing for
                                              their significance.'),
                                            h4("Plot with estimated DIF nonlinear curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_NLR',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_nlrItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_NLR_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difnlrSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score with respect to the group membership.'),
                                            plotOutput('plot_DIF_NLR'),
                                            downloadButton("DP_plot_DIF_NLR", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_NLR'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "none")'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plot(fit, item = 1)'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$nlrPAR')),
                                            br()
                                            )
                                            )
                        ),
                        # * IRT LORD ####
                        tabPanel("IRT Lord",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            fluidRow(
                                              column(6, align="center",
                                                     img(src = 'lord_udif.png', width = 380)),
                                              column(6, align="center",
                                                     img(src = 'lord_nudif.png', width = 380))
                                            ),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_print_DIF_IRT_lord',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_lordSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput('puri_Lord', 'Item purification', FALSE))),
                                            verbatimTextOutput('print_DIF_IRT_Lord'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_IRT_lord',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_lordItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput('puri_Lord_plot', 'Item purification', FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difirt_lord_itemSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            plotOutput('plot_DIF_IRT_Lord'),
                                            downloadButton("DP_plot_DIF_IRT_Lord", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_lord'),
                                            uiOutput('irteq_lord'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Lord'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef1PL <- fit1PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Lord")'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef2PL <- fit2PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Lord")'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef3PL <- fit3PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Lord")')),
                                            br()
                                            )
                                   )
                                   ),
                        # * IRT RAJU ####
                        tabPanel("IRT Raju",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            fluidRow(
                                              column(6, align="center",
                                                     img(src = 'raju_udif.png', width = 380)),
                                              column(6, align="center",
                                                     img(src = 'raju_nudif.png', width = 380))
                                            ),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_print_DIF_IRT_raju',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_rajuSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_Raju',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_IRT_Raju'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_IRT_raju',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_rajuItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_Raju_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difirt_raju_itemSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            plotOutput('plot_DIF_IRT_Raju'),
                                            downloadButton("DP_plot_DIF_IRT_Raju", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_raju'),
                                            uiOutput('irteq_raju'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Raju'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef1PL <- fit1PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Raju")'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef2PL <- fit2PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Raju")'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef3PL <- fit3PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Raju")')),
                                            br())
                                   )
                                   ),
                        # * DDF ####
                        tabPanel("DDF",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4('Equation'),
                                            p('For ', strong('K'), ' possible test choices is the probability of the correct answer for
                                               person ', strong('i'), ' with standardized total score ', strong('Z'), ' and group
                                              membership ', strong('G'),' in item ', strong('j'), 'given by the following equation: '),
                                            ('$$\\mathrm{P}(Y_{ij} = K|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{1}{1 + \\sum_l e^{\\left( b_{il0} + b_{il1} Z + b_{il2} G + b_{il3} Z:G\\right)}}$$'),
                                            p('The probability of choosing distractor ', strong('k'), ' is then given by: '),
                                            ('$$\\mathrm{P}(Y_{ij} = k|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{e^{\\left( b_{jk0} + b_{jk1} Z_i + b_{jk2} G_i + b_{jk3} Z_i:G_i\\right)}}
                                                   {1 + \\sum_l e^{\\left( b_{jl0} + b_{jl1} Z_i + b_{jl2} G_i + b_{jl3} Z_i:G_i\\right)}}$$'),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DDF',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_print_DDF",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_DDF_print',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DDF'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "none")'),
                                                br(),
                                                code('fit')),
                                            br()
                                   ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4("Plot with estimated DDF curves"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DDF',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_plot_DDF",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_DDF_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "ddfSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of selected answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score and who selected given option with
                                              respect to the group membership.'),
                                            plotOutput('plot_DDF'),
                                            downloadButton("DP_plot_DDF", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('DDFeq'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DDF'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "none")'),
                                                br(),
                                                code('# Estimated coefficients of item 1'),
                                                br(),
                                                code('fit$mlrPAR[[1]]')),
                                            br()
                                            )
                                 )
                                 )
                        ),
                #%%%%%%%%%%%%%%%%%%%%%
                # REPORTS ############
                #%%%%%%%%%%%%%%%%%%%%%
                tabPanel("Reports",
                         h3("Download report"),
                         h4("Settings of report"),
                         p(code("ShinyItemAnalysis"), " offers an option to download a report in HTML or PDF format. PDF report
                           creation requires latest version of", a("MiKTeX", href = "https://miktex.org/howto/install-miktex",
                                                                   target = "_blank"),
                           "(or other TeX distribution). If you don't have the latest installation, please, use the HTML report."),
                         p("There is an option whether to use customize settings. By checking the", strong("Customize settings"),
                           "local settings will be offered and use for each selected section of report. Otherwise the settings
                           will be taken from pages of application. You can also include your name into report as well as the name
                           of dataset which was used. "),
                         fluidRow(
                           column(2, radioButtons("report_format", "Format of report", c("HTML" = "html", "PDF" = "pdf"))),
                           column(2, checkboxInput("customizeCheck", "Customize settings", FALSE)),
                           column(2, textInput("reportAuthor", "Author")),
                           column(2, textInput("reportDataName", "Dataset"))
                         ),
                         h4("Content of report"),
                         p("Reports by default contain summary of total scores, table of standard scores, item analysis,
                           distractors plots for each item and multinomial regression plots for each item. Other analyses
                           can be selected below. "),
                         fluidRow(
                           column(8,
                                  p(strong("Validity")),
                                  checkboxInput("corr_report", "Correlation structure", FALSE),
                                  conditionalPanel(condition = "input.customizeCheck",
                                                   conditionalPanel(condition = "input.corr_report",
                                                                    div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                                        numericInput('corr_plot_clust_report',
                                                                                     label = 'Number of clusters',
                                                                                     value = 1,
                                                                                     min = 1,
                                                                                     max = 1)),
                                                                    div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                                        selectInput('corr_plot_clustmethod_report',
                                                                                    label = 'Clustering method',
                                                                                    choices = list("None" = "none",
                                                                                                   "Ward's"  = "ward.D",
                                                                                                   "Ward's n. 2" = "ward.D2",
                                                                                                   "Single" = "single",
                                                                                                   "Complete" = "complete",
                                                                                                   "Average" = "average",
                                                                                                   "McQuitty" = "mcquitty",
                                                                                                   "Median" = "median",
                                                                                                   "Centroid" = "centroid"))))),
                                  checkboxInput("predict_report", "Predictive validity", FALSE)
                           )
                         ),
                         fluidRow(
                           conditionalPanel(condition = "input.customizeCheck",
                                            column(6,
                                                   p(strong("Difficulty/discrimination plot")),
                                                   splitLayout(sliderInput('DDplotNumGroupsSlider_report','Number of groups:',
                                                                           min   = 1,
                                                                           max   = 5,
                                                                           value = 3),
                                                               sliderInput("DDplotRangeSlider_report", "Which two groups to compare:",
                                                                           min = 1,
                                                                           max = 3,
                                                                           step = 1,
                                                                           value = c(1, 3)))))
                         ),
                         fluidRow(
                           conditionalPanel(condition = "input.customizeCheck",
                                            column(6,
                                                   p(strong("Distractors plots")),
                                                   splitLayout(radioButtons('type_combinations_distractor_report',
                                                                            'Type',
                                                                            list("Combinations", "Distractors")),
                                                               sliderInput('distractorGroupSlider','Number of groups:',
                                                                             min   = 1,
                                                                             max   = 5,
                                                                             value = 3))))
                         ),
                         fluidRow(
                           column(4,
                                  radioButtons("irt_type_report", "IRT model selection",
                                               c("None" = "none",
                                                 "Rasch" = "rasch",
                                                 "1PL" = "1pl",
                                                 "2PL" = "2pl",
                                                 "3PL" = "3pl"),
                                               selected = "1pl")
                           )
                         ),

                           fluidRow(
                             column(3,
                                    p(strong("DIF method selection")),
                                    checkboxInput("histCheck", "None - histograms by group only", FALSE),
                                    checkboxInput("deltaplotCheck", "Delta plot", FALSE),
                                    checkboxInput("logregCheck", "Logistic regression", FALSE),
                                    checkboxInput("multiCheck", "Multinomial regression", FALSE)
                             ),
                             conditionalPanel(condition = "input.customizeCheck",
                             conditionalPanel(condition = "input.deltaplotCheck",
                               column(2, p(strong("Delta plot settings")),
                                      radioButtons('type_threshold_report', 'Threshold',
                                                   list("Fixed", "Normal")
                                      ),
                                      checkboxInput('puri_DP_report', 'Item purification', FALSE),
                                      conditionalPanel(
                                        condition = "input.puri_DP_report",
                                        selectInput("puri_DP_type_report", "Purification method",
                                                    c("IPP1" = "IPP1",
                                                      "IPP2" = "IPP2",
                                                      "IPP3" = "IPP3"
                                                    ),
                                                    selected = "IPP1")
                                      )
                               )
                             ),
                             conditionalPanel(condition = "input.logregCheck",
                               column(2, p(strong("Logistic regression settings")),
                                      radioButtons('type_print_DIF_logistic_report', 'Type',
                                                   c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                     "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                     "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                   ),
                                                   'both'
                                      ),
                                      selectInput("correction_method_log_report", "Correction method",
                                                  c("BH" = "BH",
                                                    "Holm" = "holm",
                                                    "Hochberg" = "hochberg",
                                                    "Hommel" = "hommel",
                                                    "BY" = "BY",
                                                    "FDR" = "fdr",
                                                    "none" = "none"
                                                  ),
                                                  selected = "none"),
                                      checkboxInput('puri_LR_report', 'Item purification', FALSE)
                               )
                             ),
                             conditionalPanel(condition = "input.multiCheck",
                               column(2, p(strong("Multinomial regression settings")),
                                      radioButtons('type_DDF_report', 'Type',
                                                   c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                     "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                     "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                   ),
                                                   'both'
                                      ),
                                      selectInput("correction_method_DDF_report", "Correction method",
                                                  c("BH" = "BH",
                                                    "Holm" = "holm",
                                                    "Hochberg" = "hochberg",
                                                    "Hommel" = "hommel",
                                                    "BY" = "BY",
                                                    "FDR" = "fdr",
                                                    "none" = "none"),
                                                  selected = "none"),
                                      checkboxInput('puri_DDF_report', 'Item purification', FALSE)
                               )
                            )
                           )
                         ),
                         p(strong("Recommendation: "), "Report generation can be faster and more reliable when you first check
                           sections of intended contents. For example, if you wish to include a ", strong("3PL IRT"),
                           " model, you can first visit ", strong("IRT models"), "section and ", strong("3PL"), " subsection."),
                         #p(strong("Warning: "), "Download of reports takes some time. Please, be patient."),
                         fluidRow(
                           column(width = 5,
                             splitLayout(cellWidths = c("45%", "55%"),
                               actionButton("generate", "Generate report"),
                               uiOutput("download_report_button")
                             )
                           )
                         ),
                         br(),
                         br(),
                         br()
                         ),

             #%%%%%%%%%%%%%%%%%%%%%
             # REFERENCES #########
             #%%%%%%%%%%%%%%%%%%%%%
             tabPanel("References",
                      h3('References'),
                      p('Akaike, H. (1974). A New Look at the Statistical Model Identification. IEEE Transactions
                        on Automatic Control, 19(6), 716-723.', a('See online.',
                                                                  href = "http://ieeexplore.ieee.org/abstract/document/1100705/",
                                                                  target = "_blank")),
                      p('Ames, A. J., & Penfield, R. D. (2015). An NCME Instructional Module on Item-Fit
                        Statistics for Item Response Theory Models. Educational Measurement: Issues and
                        Practice, 34(3), 39-48.', a("See online.",
                                                    href = "http://onlinelibrary.wiley.com/doi/10.1111/emip.12067/full",
                                                    target = "_blank")),
                      p('Angoff, W. H., & Ford, S. F. (1973). Item-Race Interaction on a Test of
                        Scholastic Aptitude. Journal of Educational Measurement, 10(2), 95-105.',
                        a("See online.",
                          href = "https://www.jstor.org/stable/1433905?seq=1#page_scan_tab_contents",
                          target = "_blank")),
                      p('Bock, R. D. (1972). Estimating Item Parameters and Latent Ability when
                        Responses Are Scored in Two or More Nominal Categories. Psychometrika,
                        37(1), 29-51.', a("See online.",
                                          href = "http://link.springer.com/article/10.1007/BF02291411",
                                          target = "_blank")),
                      p('Cronbach, L. J. (1951). Coefficient Alpha and the Internal Structure
                        of Tests. Psychometrika, 16(3), 297-334.', a("See online.",
                                                                     href = "https://link.springer.com/article/10.1007/BF02310555",
                                                                     target = "_blank")),
                      p("Drabinova, A., & Martinkova, P. (2017). Detection of Differential Item Functioning with Non-Linear
                         Regression: Non-IRT Approach Accounting for Guessing. Journal of Educational Measurement, 54(4), 498-517.",
                        a("See online.",
                          href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                          target = "_blank")),
                      p("Lord, F. M. (1980). Applications of Item Response Theory to Practical Testing Problems.
                        Routledge."),
                      p("Magis, D., & Facon, B. (2012). Angoff's Delta Method Revisited:
                        Improving DIF Detection under Small Samples. British Journal of
                        Mathematical and Statistical Psychology, 65(2), 302-321.", a("See online.",
                                                                                     href = "https://www.ncbi.nlm.nih.gov/pubmed/22500570",
                                                                                     target = "_blank")),
                      p("Mantel, N., & Haenszel, W. (1959). Statistical Aspects of the Analysis of Data from
                        Retrospective Studies. Journal of the National Cancer Institute, 22 (4), 719-748.", a("See online.",
                                                                                                              href = "http://www.medicine.mcgill.ca/epidemiology/hanley/c634/stratified/Mantel_Haenszel_1.pdf",
                                                                                                              target = "_blank")),
                      p("Martinkova, P., Drabinova, A., & Houdek, J. (2017). ShinyItemAnalysis: Analyza prijimacich a
                         jinych znalostnich ci psychologických testu. TESTFORUM, 6(9), 16–35.",
                        a("See online.", href = "http://testforum.cz/domains/testforum.cz/index.php/testforum/article/view/TF2017-9-129", target = "_blank"),
                        "(ShinyItemAnalysis: Analyzing admission and other educational and psychological tests)"),
                      p("Martinkova, P., Drabinova, A., Liaw, Y. L., Sanders, E. A., McFarland, J. L., & Price, R. M.
                        (2017). Checking Equity: Why Differential Item Functioning Analysis Should Be a Routine Part
                        of Developing Conceptual Assessments. CBE-Life Sciences Education, 16(2). ",
                        a('See online.',
                          href = "http://www.lifescied.org/content/16/2/rm2.full.pdf+html?with-ds=yes",
                          target = "_blank")),
                      p("Martinkova, P., Stepanek, L., Drabinova, A., Houdek, J., Vejrazka, M., & Stuka, C. (2017).
                        Semi-real-time analyses of item characteristics for medical school admission tests. In: Proceedings of
                        the 2017 Federated Conference on Computer Science and Information Systems. In print."),
                      p("Swaminathan, H., & Rogers, H. J. (1990). Detecting Differential Item
                        Functioning Using Logistic Regression Procedures. Journal of Educational
                        Measurement, 27(4), 361-370.",
                        a('See online.',
                          href = "https://www.jstor.org/stable/1434855?seq=1#page_scan_tab_contents",
                          target = "_blank")),
                      p("Raju, N. S. (1988). The Area between Two Item Characteristic Curves. Psychometrika,
                        53 (4), 495-502.",
                        a('See online.',
                          href = "https://link.springer.com/article/10.1007/BF02294403",
                          target = "_blank")),
                      p("Raju, N. S. (1990). Determining the Significance of Estimated Signed and Unsigned Areas
                        between Two Item Response Functions. Applied Psychological Measurement, 14 (2), 197-207.",
                        a('See online.',
                          href = "http://journals.sagepub.com/doi/abs/10.1177/014662169001400208",
                          target = "_blank")),
                      p('Rasch, G. (1960) Probabilistic Models for Some Intelligence and Attainment Tests.
                        Copenhagen: Paedagogiske Institute.'),
                      p('Schwarz, G. (1978). Estimating the Dimension of a Model. The Annals of Statistics,
                        6(2), 461-464.', a('See online.',
                                           href = "https://projecteuclid.org/euclid.aos/1176344136",
                                           target = "_blank")),
                      p("Wilson, M. (2005). Constructing Measures: An Item Response Modeling Approach."),
                      p("Wright, B. D., & Stone, M. H. (1979). Best Test Design. Chicago: Mesa Press."),
                      br()
                      )
                      )
                      )

