% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/afv_sites.R
\name{afv_sites}
\alias{afv_sites}
\title{Calculate additive function values for sites in a Landscape Network (LSN)}
\usage{
afv_sites(
  sites,
  edges,
  afv_col,
  save_local = TRUE,
  lsn_path = NULL,
  overwrite = TRUE
)
}
\arguments{
\item{sites}{A named list of one or more \code{sf} objects with
POINT geometry that have been snapped to the LSN using
\code{\link[SSNbler]{sites_to_lsn}}.}

\item{edges}{\code{sf} object with LINESTING geometry created
using \code{\link{lines_to_lsn}}.}

\item{afv_col}{Name of the column in \code{edges} containing
the additive function value for each feature, in character
format. Created using \code{\link{afv_edges}}.}

\item{save_local}{Logical indicating whether the updated
\code{sites} should be saved to \code{lsn_path} in GeoPackage
format. File basenames are taken from the names assigned to the
\code{sites} list. Default is \code{TRUE}.}

\item{lsn_path}{Optional. Local pathname to a directory in
character format specifying where the LSN resides, which is
created using \code{link[SSNbler]{lines_to_lsn}}. Must be
specified if \code{save_local = TRUE}.}

\item{overwrite}{A logical indicating whether results should be
overwritten if \code{afv_col} already exists in \code{sites}
or sites.gpkg already exists in \code{lsn_path} and
\code{save_local = TRUE}. Default = TRUE.}
}
\value{
One or more \code{sf} object(s) with all the original data
from \code{sites}, along with a new \code{afv_col} column in each
\code{sites sf} object. A named list is returned. If
\code{save_local = TRUE}, a GeoPackage for each \code{sf} object
is saved in \code{lsn_path}. Output file names are assigned based
on the input \code{sites} attribute \code{names}.
}
\description{
Calculate additive function values for sites in a Landscape Network (LSN)
}
\details{
Spatial weights are used when fitting statistical models
with 'SSN2' to split the tail up covariance function upstream of
network confluences, which allows for the disproportionate
influence of one upstream edge over another (e.g., a large stream
channel converges with a smaller one) on downstream
values. Calculating the spatial weights is a four step process:
\enumerate{
\item calculating the segment proportional influence (PI) values for
the edges,
\item calculating the additive function values (AFVs) for
the edges,
\item calculating the AFVs for the
observed and prediction sites, and
\item calculating the spatial
weights for observed and prediction sites.
}

Steps 1) and 2) are undertaken in \code{\link[=afv_edges]{afv_edges()}}, Step 3) is
calculated in \code{afv_sites()}, and Step 4) is calculated in the
package 'SSN2' when spatial stream network models that include the
tail up covariance function are fit using \code{\link[SSN2]{ssn_lm}}
or \code{\link[SSN2]{ssn_glm}}.

The additive function value (AFV) for an observed or
prediction site is equal to the AFV of the edge the site resides
on. Therefore, \eqn{0 \le AFV \le 1}. See Peterson and Ver Hoef
(2010) for a more detailed description of AFVs, how they are
calculated, and how they are used in the tail up covariance function.
}
\examples{
#' # Get temporary directory, where the example LSN will be stored
# locally.
temp_dir <- tempdir()
# Build the LSN. When working with your own data, lsn_path will be
# a local folder of your choice rather than a temporary directory.
edges<- lines_to_lsn(
   streams = MF_streams,
   lsn_path = temp_dir,
   snap_tolerance = 1,
   check_topology = FALSE,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate observed sites into the LSN
obs<- sites_to_lsn(
   sites = MF_obs,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 100,
   overwrite = TRUE,
   verbose = FALSE
)

# Incorporate the prediction dataset, preds, into the LSN
preds<- sites_to_lsn(sites = MF_preds,
   edges = edges,
   save_local = FALSE,
   snap_tolerance = 1,
   overwrite = TRUE,
   verbose = FALSE
)

# Calculate the AFVs for the edges using a column representing
# watershed area (h2oAreaKm2) for the downstream node of each edge
# feature.
edges<- afv_edges(
   edges=edges,
   infl_col = "h2oAreaKm2",
   segpi_col = "areaPI",
   lsn_path = temp_dir,
   afv_col = "afvArea",
   overwrite = TRUE,
   save_local = FALSE
)

# Calculate AFVs for observed sites (obs) and the prediction
# dataset, preds.
site.list<- afv_sites(
   sites = list(obs = obs,
                preds = preds),
   edges=edges,
   afv_col = "afvArea",
   save_local = FALSE,
   overwrite = TRUE
)

# Get names of sites in site.list
names(site.list)

# Check AFVs stored in new column afvArea to ensure that 0 <= AFV
# <= 1 and that there are no NULL values.
summary(site.list$obs$afvArea)
summary(site.list$preds$afvArea)
}
\references{
Peterson, E.E. and Ver Hoef, J.M. (2010) A
mixed model moving average approach to geostatistical modeling in
stream networks. Ecology 91(3), 644–651.
}
