% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/checkStratification.R
\name{Spattern}
\alias{Spattern}
\alias{dim_wt_tab}
\alias{soacheck2D}
\alias{soacheck3D}
\title{functions to evaluate stratification properties of (O)SOAs and GSOAs}
\usage{
Spattern(D, s, maxwt = 4, maxdim = NULL, ...)

dim_wt_tab(pat, dimlim = NULL, wtlim = NULL, ...)

soacheck2D(D, s = 3, el = 3, t = 3, verbose = FALSE)

soacheck3D(D, s = 3, el = 3, t = 3, verbose = FALSE)
}
\arguments{
\item{D}{a matrix with factor levels or an object of class \code{SOA} or a
data.frame object with numeric columns.\cr
Functions \code{soacheck2D} and \code{soacheck3D} require levels
that are consecutively numbered (starting with 0 or 1).\cr
Function \code{Spattern} also works, if all columns of \code{D}
have the same number of unique numeric values; the function will code them using
power contrasts.}

\item{s}{the prime or prime power according to which the array is checked}

\item{maxwt}{maximum weight to be considered for the pattern (default: 4; see Details);\cr
if the specified limit is larger than \code{maxdim*el},
it is reduced accordingly (where \code{el} is such that \code{s^el} is the number of levels)}

\item{maxdim}{maximum dimension to be considered for the pattern (default: \code{NULL} implies that \code{maxdim}=\code{min(maxwt, ncol(D))};
see also Details);\cr
if the specified limit is larger than \code{m=ncol(D)}, it is reduced to \code{m}}

\item{...}{currently not used}

\item{pat}{an object of class \code{Spattern}}

\item{dimlim}{integer; limits the returned dimension rows to the
rows from 1 up to \code{dimlim}; the bottom margin continues to include all
dimensions that were used in calculating \code{pat}}

\item{wtlim}{integer; limits the returned weight columns to the columns from
1 up to \code{wtlim}; the right margin continues to include all weights
that were used in calculating \code{pat}}

\item{el}{the exponent so that the number of levels of the array is \code{s^el}
(if \code{s} is not NULL)}

\item{t}{the strength for which to look (2, 3, or 4), equal to the sum of the
exponents in the stratification dimensions; for example, \code{soacheck2D} considers \cr
sxs 2D projections with \code{t=2}, \cr
s^2xs and sxs^2 projections with \code{t=3}, \cr
and s^3xs, s^2xs^2 and sxs^3 projections with \code{t=4}.\cr
If \code{t=4} and \code{el=2}, property gamma (s^3 x s and s x s^3) is obviously
impossible and will not be part of the checks.}

\item{verbose}{logical; if \code{TRUE}, additional information is printed
(confounded pair or triple projections with A2 or A3, respectively, or table of correlations)}
}
\value{
Function \code{Spattern} returns an object of class \code{Spattern}
that is a named vector with attributes:\cr
The attribute \code{call} holds the function call
(and thus documents, e.g., limits set on dimension and/or weight).\cr
The attribute \code{dim_wt_tab} holds a table of contributions
split out by dimension (rows) and weights (columns), which has class
\code{dim_wt_tab} and the further attribute \code{Spattern-class}.\cr
Function \code{dim_wt_tab} returns the \code{dim_wt_tab} attribute of
an object of class \code{Spattern}; note that the object contains NA values
for combinations of dimension and weight that cannot occur.

Function \code{dim_wt_tab} postprocesses an \code{Spattern} object
and produces a table that holds the S pattern entries
separated by the dimension of the contributing effect column group (rows)
and the weight of the effect column micro group (columns). The margin shows row and
column sums (see Details section for caveats).
}
\description{
\code{soacheck2D} and \code{soacheck3D} evaluate 2D and 3D projections,
\code{Spattern} calculates the stratification pattern by Tian and Xu (2022),
and \code{dim_wt_tab} extracts and formats the \code{dim_wt_tab} attribute of \code{Spattern}.
}
\details{
Function \code{Spattern} calculates the stratification pattern or S pattern
as proposed in Tian and Xu (2022) (under the name space-filling pattern);
the details and the implementation in this function are also described in
Groemping (2022b).\cr
Position \code{j} in the S pattern shows the imbalance when considering \code{s^j}
strata. \code{j} is also called the (total) weight. \code{j=1} can occur for an
individual column only. \code{j=2} can be obtained either for an
\code{s^2} level version of an individual column or for the crossing of
\code{s^1} level versions of two columns, and so forth.

Obtaining the entire S pattern
can be computationally demanding. The arguments \code{maxwt} and
\code{maxdim} limit the effort (choose \code{NULL} for no limit):\cr
\code{maxwt} gives an upper limit for the weight \code{j} of the previous paragraph;
if \code{NULL}, \code{maxwt} is set to \code{maxdim*el}.\cr
\code{maxdim} limits the number of columns that are considered in combination.\cr
When using a non-null \code{maxdim}, pattern entries for \code{j} larger than \code{maxdim} can be smaller
than if one would not have limited the dimension. Otherwise, dimensionality is unlimited,
which is equivalent to specifying \code{maxdim} as the minimum of \code{maxwt} and \code{ncol(D)}.

\code{Spattern} with \code{maxdim=2} and \code{maxwt=t} can be used as an alternative
to \code{soacheck2D},\cr
and analogously \code{Spattern} with \code{maxdim=2} and \code{maxwt=t} can be used as an alternative
to \code{soacheck3D}.

An \code{Spattern} object
object can be post-processed with function \code{dim_wt_tab}. That function splits
the S pattern into contributions from effect column groups of different dimensions,
arranged with a row for each dimension and a column for each weight.
If \code{Spattern} was called with \code{maxdim=NULL} and
\code{maxwt=NULL}, the output object shows the GWLP in the right margin and the
S pattern in the bottom margin. If \code{Spattern} was called with relevant restrictions
on dimensions (\code{maxdim}, default 4) and/or weights (\code{maxwt}, default 4),
sums in the margins can be smaller than they would be for unconstrained dimension and
weights.

Functions \code{soacheck2D} and \code{soacheck3D} were available before
function \code{Spattern}; many of their use cases can now be handled with \code{Spattern}
instead. The functions are often fast to yield a \code{FALSE} outcome,
but can be very slow to yield a \code{TRUE} outcome for larger designs.\cr
The functions inspect 2D and 3D
stratification, respectively. Each column must have \code{s^el} levels.
\code{t} specifies the degree of balance the functions are asked to look for.

Function \code{soacheck2D},
\itemize{
\item with el=t=2, looks for strength 2 conditions (s^2 levels, sxs balance),
\item with el=2, t=3, looks for strength 2+ / 3- conditions (s^2 levels, s^2xs balance),
\item with el=t=3, looks for strength 2* / 3 conditions (s^3 levels, s^2xs balance).
\item with el=2, t=4, looks for the enhanced strength 2+ / 3-  property alpha (s^2 levels, s^2xs^2 balance).
\item and with el=3, t=4, looks for strength 3+ / 4 conditions (s^3 levels, s^3xs and s^2xs^2 balance).
}

Function \code{soacheck3D},
\itemize{
\item with el=2, t=3, looks for strength 3- conditions (s^2 levels, sxsxs balance),
\item with el=t=3, looks for strength 3 conditions (s^3 levels, sxsxs balance),
\item and with el=3, t=4, looks for strength 3+ / 4 conditions (s^3 levels, s^2xsxs balance).
}

If \code{verbose=TRUE}, the functions print the pairs or triples that violate
the projection requirements for 2D or 3D.
}
\examples{
nullcase <- matrix(0:7, nrow=8, ncol=4)
soacheck2D(nullcase, s=2)
soacheck3D(nullcase, s=2)
Spattern(nullcase, s=2)
Spattern(nullcase, s=2, maxdim=2)
  ## the non-zero entry at position 2 indicates that
  ## soacheck2D does not comply with t=2
(Spat <- Spattern(nullcase, s=2, maxwt=4))
  ## comparison to maxdim=2 indicates that
  ## the contribution to S_4 from dimensions
  ## larger than 2 is 1
## postprocessing Spat
dim_wt_tab(Spat)

## Shi and Tang strength 3+ construction in 7 8-level factors for 32 runs
D <- SOAs_8level(32, optimize=FALSE)

## check for strength 3+ (default el=3 is OK)
## 2D check
soacheck2D(D, s=2, t=4)
## 3D check
soacheck3D(D, s=2, t=4)
## using Spattern (much faster for many columns)
  ## does not have strength 4
  Spattern(D, s=2)
  ## but complies with strength 4 for dim up to 3
  Spattern(D, s=2, maxwt=4, maxdim=3)
  ## inspect more detail
  Spat <- (Spattern(D, s = 2, maxwt=5))
  dim_wt_tab(Spat)
}
\references{
For full detail, see \code{\link{SOAs-package}}.

Groemping (2022a)\cr
Groemping (2022b)\cr
He and Tang (2013)\cr
Shi and Tang (2020)\cr
Tian and Xu (2022)
}
