% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/halton.lattice.polygon.r
\name{halton.lattice.polygon}
\alias{halton.lattice.polygon}
\title{Halton lattice inside a \code{SpatialPolygon*} object.}
\usage{
halton.lattice.polygon(
  x,
  N = 10000,
  J = NULL,
  eta = c(1, 1),
  triangular = FALSE,
  bases = c(2, 3)
)
}
\arguments{
\item{x}{A \code{SpatialPolygons} or \code{SpatialPolygonsDataFrame} object.}

\item{N}{Approximate number of points to place in the lattice.  If \code{J} 
is specified, it takes precedence.  If \code{J} is NULL, the 
algorithm attempts to place \code{N} points in the bounding box 
using Halton boxes that are as close to square as possible.  
This \code{N} is not exact, but is a target.}

\item{J}{A 2X1 vector of base powers which determines the size and shape 
of the Halton boxes. See additional description in help for 
\code{\link{hip.polygon}} function.}

\item{eta}{A 2X1 vector of the number of points to add inside each Halton box.
e.g., if \code{eta} = \code{c(3,2)}, a small grid of 3 by 2 points is 
added inside each Halton box. \code{eta[1]} is for the
horizontal dimension, \code{eta[2]} is for the vertical dimension.}

\item{triangular}{boolean, if TRUE, construct a triangular grid. 
If FALSE, construct rectangular grid.  See help for \code{\link{hip.polygon}}.}

\item{bases}{A 2X1 vector of Halton bases.  These must be co-prime.}
}
\value{
A \code{SpatialPointsDataFrame} containing locations in the Halton lattice

 Attributes of the points are: 
\itemize{
  \item \code{latticeID}: A unique identifier for every point.  ID's are integers
  numbering points in row-major order from the south.  

  \item \code{geometryID}: The ID of the polygon in \code{x} containing each 
  point.  The 
  ID of polygons in \code{x} are \code{row.names(geometry(x))}. 
  \item Any attributes of the original polygons (in \code{x}). 
}

Additional attributes of the output object, beyond those which 
make it a \code{SpatialPointsDataFrame}, are:
\itemize{
   \item \code{J}: the \code{J} vector used to construct the lattice. 
     This is either the input \code{J} or the computed \code{J} when 
     only \code{N} is specified. 
   \item \code{eta}: the \code{eta} vector used in the lattice.
   \item \code{bases}: the bases of the van der Corput sequences used in the lattice, 
     one per dimension.
   \item \code{triangular}: Whether the lattice is triangular or square.
   \item \code{hl.bbox}: the bounding box surrounding the input \code{x} object. 
     This is saved because bounding box of the return object is not the 
     same as the bounding box of \code{x} (i.e., \code{bbox(return)} \code{!=} 
     \code{bbox(x)}).
}
}
\description{
Constructs a lattice of Halton boxes (a Halton lattice) inside a 
\code{SpatialPolygons} or \code{Spatial}\code{Polygons} \code{DataFrame} object. This is a wrapper for
\code{halton.lattice}, which does all the hard work.
}
\details{
This routine is called internally by \code{hip.polygon}, and is not 
normally called by the user.
}
\examples{

# Take and plot Halton lattice to illustrate
WA.hgrid <- halton.lattice.polygon( WA, J=c(3,2), eta=c(3,2), triangular=TRUE )
plot(WA)
points(WA.hgrid, pch=16, cex=.5, col="red" )

# Plot the Halton boxes
tmp.J <- attr(WA.hgrid,"J")
tmp.b <- attr(WA.hgrid,"bases")
tmp.bb <- attr(WA.hgrid,"hl.bbox")

for(d in 1:2){
  tmp2 <- tmp.bb[d,1] + (0:(tmp.b[d]^tmp.J[d]))*(diff(tmp.bb[d,]))/(tmp.b[d]^tmp.J[d])
  if( d == 1){
      abline(v=tmp2, col="blue")
  } else{
      abline(h=tmp2, col="blue")
  }
}

# To explore, re-run the above changing J, eta, and triangular, 
}
\seealso{
\code{\link{hip.polygon}}, \code{\link{halton.lattice}}
}
\author{
Trent McDonald
}
\keyword{design}
\keyword{survey}
