/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.formats;

import java.math.BigInteger;
import java.util.LinkedHashMap;

import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.STTextUnderlineType;
import org.docx4j.dml.TextFont;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTVerticalAlignRun;
import org.docx4j.wml.HpsMeasure;
import org.docx4j.wml.RFonts;
import org.docx4j.wml.RPr;
import org.docx4j.wml.STVerticalAlignRun;
import org.docx4j.wml.U;
import org.docx4j.wml.UnderlineEnumeration;

public class TextProperties {
	private int size;
	private boolean bold;
	private boolean italic;
	private boolean underlined;
	private String color;
	private String fontfamily;
	private String valign;
	private String shading_color;
	
	public TextProperties(int size, boolean bold, boolean italic,
			boolean underlined, String color, String fontfamily, String valign) {
		super();
		this.size = size;
		this.bold = bold;
		this.italic = italic;
		this.underlined = underlined;
		this.color = color;
		this.fontfamily = fontfamily;
		this.valign = valign;
		shading_color = null;
	}
	
	public void setShadingColor(String shadingColor){
		shading_color = shadingColor;
	}
	public TextProperties getClone(){
		TextProperties tp = new TextProperties( size,
				bold, italic,
				underlined, color, fontfamily, valign);
		if( shading_color != null) tp.setShadingColor(this.shading_color);
		return( tp );
	}
	
	public int getSize() {
		return size;
	}

	public boolean isBold() {
		return bold;
	}
	public String getValign(){
		return valign;
	}
	public boolean isSub() {
		return valign.equals("subscript");
	}
	
	public boolean isSuper() {
		return valign.equals("superscript");
	}
	
	public boolean isItalic() {
		return italic;
	}

	public boolean isUnderlined() {
		return underlined;
	}

	public String getColor() {
		return color;
	}

	public String getFontfamily() {
		return fontfamily;
	}


	public LinkedHashMap<String, String> get_html_pr() {
		LinkedHashMap<String, String> runProperties = new LinkedHashMap<String, String>();

		runProperties.put("font-size",size+"pt");
		runProperties.put("font-family",fontfamily);
				
		if( bold ){
			runProperties.put("font-weight","bold");
		}
		
		if( italic ){
			runProperties.put("font-style","italic");
		}
		
		if( underlined ){
			runProperties.put("text-decoration","underline");
		}
		runProperties.put("color",color);
		
		if( shading_color!=null){
			runProperties.put("background-color", shading_color);
		}
		
		if( valign.equals("subscript")){
			runProperties.put("vertical-align","sub");
		} else if( valign.equals("superscript")){
			runProperties.put("vertical-align","super");
		}
		
		return runProperties;
	}
	public LinkedHashMap<String, String> get_html_pr_vanilla() {
		LinkedHashMap<String, String> runProperties = new LinkedHashMap<String, String>();
				
		if( bold ){
			runProperties.put("font-weight","bold");
		}
		
		if( italic ){
			runProperties.put("font-style","italic");
		}
		
		if( underlined ){
			runProperties.put("text-decoration","underline");
		}
		runProperties.put("color",color);
		

		
		return runProperties;
	}
	public RPr get_docx_pr(){
		RPr runProperties = new RPr();
		
		HpsMeasure fsize = new HpsMeasure();
		fsize.setVal( BigInteger.valueOf( size*2 ) );
		runProperties.setSz( fsize );
		
		RFonts rfonts = new RFonts();
        rfonts.setAscii(fontfamily);
        rfonts.setCs(fontfamily);
        rfonts.setHAnsi(fontfamily);
		runProperties.setRFonts(rfonts);
		
		if( shading_color!=null){
			CTShd shading = new CTShd();
			shading.setFill(shading_color);
			runProperties.setShd(shading);
		}
		if( bold ){
			BooleanDefaultTrue b = new BooleanDefaultTrue();
			b.setVal(true);
	        runProperties.setB(b);
		}
		
		if( italic ){
			BooleanDefaultTrue b = new BooleanDefaultTrue();
			b.setVal(true);
	        runProperties.setI(b);
		}
		
		if( underlined ){
			U u = new U(); 
			u.setVal(UnderlineEnumeration.SINGLE);
			runProperties.setU(u);
		}
		
		if( valign.equals("subscript")){
			CTVerticalAlignRun vv = new CTVerticalAlignRun();
			vv.setVal(STVerticalAlignRun.SUBSCRIPT);
			runProperties.setVertAlign(vv);
		} else if( valign.equals("superscript")){
			CTVerticalAlignRun vv = new CTVerticalAlignRun();
			vv.setVal(STVerticalAlignRun.SUPERSCRIPT);
			runProperties.setVertAlign(vv);
		}

		
		org.docx4j.wml.Color col = new org.docx4j.wml.Color();
		col.setVal(color);
		runProperties.setColor(col);
		
		return runProperties;
	}

	public CTTextCharacterProperties get_pptx_pr() throws Exception{
		CTTextCharacterProperties runProperties = new CTTextCharacterProperties();
		
		runProperties.setSz(size*100);
		
		TextFont tf = new TextFont();
		tf.setTypeface(fontfamily);
		runProperties.setCs(tf);
		runProperties.setLatin(tf);
		runProperties.setEa(tf);
		
		if( shading_color!=null){
			runProperties.setHighlight(org.lysis.reporters.pptx4r.tools.Format.getShadingCol(shading_color));
		}
		
		if( bold ){
			runProperties.setB(bold);
		}
		
		if( italic ){
			runProperties.setI( italic );
		}
		if( underlined ){
			runProperties.setU(STTextUnderlineType.SNG);
		}
		
		if( valign.equals("subscript")){
			runProperties.setBaseline(-25000);
		} else if( valign.equals("superscript")){
			runProperties.setBaseline(30000);
		}
		runProperties.setSolidFill(org.lysis.reporters.pptx4r.tools.Format.getCol(color));
		
		return runProperties;
	}

}
