% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TSVM.R
\name{TSVM}
\alias{TSVM}
\title{Transductive SVM classifier using the convex concave procedure}
\usage{
TSVM(X, y, X_u, C, Cstar, kernel = kernlab::vanilladot(),
  balancing_constraint = TRUE, s = 0, x_center = TRUE, scale = FALSE,
  eps = 1e-09, max_iter = 20, verbose = FALSE)
}
\arguments{
\item{X}{matrix; Design matrix for labeled data}

\item{y}{factor or integer vector; Label vector}

\item{X_u}{matrix; Design matrix for unlabeled data}

\item{C}{numeric; Cost parameter of the SVM}

\item{Cstar}{numeric; Cost parameter of the unlabeled objects}

\item{kernel}{kernlab::kernel to use}

\item{balancing_constraint}{logical; Whether a balancing constraint should be enfored that causes the fraction of objects assigned to each label in the unlabeled data to be similar to the label fraction in the labeled data.}

\item{s}{numeric; parameter controlling the loss function of the unlabeled objects (generally values between -1 and 0)}

\item{x_center}{logical;  Should the features be centered?}

\item{scale}{If TRUE, apply a z-transform to all observations in X and X_u before running the regression}

\item{eps}{numeric; Stopping criterion for the maximinimization}

\item{max_iter}{integer; Maximum number of iterations}

\item{verbose}{logical; print debugging messages, only works for vanilladot() kernel (default: FALSE)}
}
\description{
Transductive SVM using the CCCP algorithm as proposed by Collobert et al. (2006) implemented in R using the quadprog package. The implementation does not handle large datasets very well, but can be useful for smaller datasets and visualization purposes.
}
\details{
C is the cost associated with labeled objects, while Cstar is the cost for the unlabeled objects. s control the loss function used for the unlabeled objects: it controls the size of the plateau for the symmetric ramp loss function. The balancing constraint makes sure the label assignments of the unlabeled objects are similar to the prior on the classes that was observed on the labeled data.
}
\examples{
library(RSSL)

# Simple example with a few objects
X <- matrix(c(0,0.001,1,-1),nrow=2)
X_u <- matrix(c(-1,-1,-1,0,0,0,-0.4,-0.5,-0.6,1.2,1.3,1.25),ncol=2)
y <- factor(c(-1,1))

g_sup <- SVM(X,y,scale=FALSE)
g_constraint <- TSVM(X=X,y=y,X_u=X_u,
                     C=1,Cstar=0.1,balancing_constraint = TRUE)

g_noconstraint <- TSVM(X=X,y=y,X_u=X_u,
                       C=1,Cstar=0.1,balancing_constraint = FALSE)

g_lin <- LinearTSVM(X=X,y=y,X_u=X_u,C=1,Cstar=0.1)

w1 <- g_sup@alpha \%*\% X
w2 <- g_constraint@alpha \%*\% rbind(X,X_u,X_u,colMeans(X_u))
w3 <- g_noconstraint@alpha \%*\% rbind(X,X_u,X_u)
w4 <- g_lin@w

plot(X[,1],X[,2],col=factor(y),asp=1,ylim=c(-3,3))
points(X_u[,1],X_u[,2],col="darkgrey",pch=16,cex=1)
abline(-g_sup@bias/w1[2],-w1[1]/w1[2],lty=2)
abline(((1-g_sup@bias)/w1[2]),-w1[1]/w1[2],lty=2) # +1 Margin
abline(((-1-g_sup@bias)/w1[2]),-w1[1]/w1[2],lty=2) # -1 Margin
abline(-g_constraint@bias/w2[2],-w2[1]/w2[2],lty=1,col="green")
abline(-g_noconstraint@bias/w3[2],-w3[1]/w3[2],lty=1,col="red")
abline(-w4[1]/w4[3],-w4[2]/w4[3],lty=1,lwd=3,col="blue")

# As example
data <- generateSlicedCookie(200,expected=TRUE,gap=1)
X <- model.matrix(Class~.-1,data)
y <- factor(data$Class)

problem <- split_dataset_ssl(X,y,frac_ssl=0.98)

X <- problem$X
y <- problem$y
X_u <- problem$X_u
y_e <- unlist(list(problem$y,problem$y_u))
Xe<-rbind(X,X_u)

g_sup <- SVM(X,y,scale=FALSE)
g_constraint <- TSVM(X=X,y=y,X_u=X_u,
                     C=1,Cstar=0.1,balancing_constraint = TRUE,
                     x_center = FALSE,verbose=TRUE)

g_noconstraint <- TSVM(X=X,y=y,X_u=X_u,
                       C=10,Cstar=0.001,balancing_constraint = FALSE,
                       x_center = FALSE,verbose=TRUE)

g_lin <- LinearTSVM(X=X,y=y,X_u=X_u,C=10,Cstar=0.001,
                    verbose=TRUE,x_center = FALSE)

g_oracle <- SVM(Xe,y_e,scale=FALSE)

w1 <- g_sup@alpha \%*\% X
w2 <- c(g_constraint@bias,g_constraint@alpha \%*\% rbind(X,X_u,X_u,colMeans(X_u)))
w3 <- c(g_noconstraint@bias,g_noconstraint@alpha \%*\% rbind(X,X_u,X_u))
w4 <- g_lin@w
w5 <- g_oracle@alpha \%*\% Xe
print(sum(abs(w4-w3)))

plot(X[,1],X[,2],col=factor(y),asp=1,ylim=c(-3,3))
points(X_u[,1],X_u[,2],col="darkgrey",pch=16,cex=1)
abline(-g_sup@bias/w1[2],-w1[1]/w1[2],lty=2)
abline(((1-g_sup@bias)/w1[2]),-w1[1]/w1[2],lty=2) # +1 Margin
abline(((-1-g_sup@bias)/w1[2]),-w1[1]/w1[2],lty=2) # -1 Margin

# Oracle:
abline(-g_oracle@bias/w5[2],-w5[1]/w5[2],lty=1,col="purple")

# With balancing constraint:
abline(-g_constraint@bias/w2[2],-w2[1]/w2[2],lty=1,col="green")

# Without balancing constraint:
abline(-g_noconstraint@bias/w3[2],-w3[1]/w3[2],lty=1,col="red")

# Linear TSVM implementation (no constraint):
abline(-w4[1]/w4[3],-w4[2]/w4[3],lty=1,lwd=3,col="blue") 
}
\references{
Collobert, R. et al., 2006. Large scale transductive SVMs. Journal of Machine Learning Research, 7, pp.1687-1712.
}
\seealso{
Other RSSL classifiers: \code{\link{EMLeastSquaresClassifier}},
  \code{\link{EMLinearDiscriminantClassifier}},
  \code{\link{GRFClassifier}},
  \code{\link{ICLeastSquaresClassifier}},
  \code{\link{ICLinearDiscriminantClassifier}},
  \code{\link{KernelLeastSquaresClassifier}},
  \code{\link{LaplacianKernelLeastSquaresClassifier}},
  \code{\link{LaplacianSVM}},
  \code{\link{LeastSquaresClassifier}},
  \code{\link{LinearDiscriminantClassifier}},
  \code{\link{LinearSVM}}, \code{\link{LinearTSVM}},
  \code{\link{LogisticLossClassifier}},
  \code{\link{LogisticRegression}},
  \code{\link{MCLinearDiscriminantClassifier}},
  \code{\link{MCNearestMeanClassifier}},
  \code{\link{MCPLDA}},
  \code{\link{MajorityClassClassifier}},
  \code{\link{NearestMeanClassifier}},
  \code{\link{QuadraticDiscriminantClassifier}},
  \code{\link{S4VM}}, \code{\link{SVM}},
  \code{\link{SelfLearning}},
  \code{\link{USMLeastSquaresClassifier}},
  \code{\link{WellSVM}}, \code{\link{svmlin}}
}
