parseSVG = function(file.name)
{
    svgFile = xmlParse(file.name);
    newXMLNamespace(xmlRoot(svgFile), "http://www.w3.org/2000/svg", "svg");
    pathRoot = getNodeSet(svgFile, "/svg:svg/svg:g")[[1]];
    
    defaultStyle = c("stroke" = "none",
                    "stroke-width" = "1",
                    "stroke-linecap" = "butt",
                    "stroke-linejoin" = "miter",
                    "stroke-miterlimit" = "4",
                    "stroke-opacity" = "1",
                    "fill" = "rgb(0%,0%,0%)",
                    "fill-rule" = "nonzero",
                    "fill-opacity" = "1");
    
    # Handle path style in named vector
    parseStyle = function(style)
    {
        if(is.null(style)) return(NULL);
        s = unlist(strsplit(style, ";"));
        val = strsplit(s, ":");
        result = sapply(val, function(x) x[2]);
        names(result) = sapply(val, function(x) x[1]);
        return(result);
    }
    # Update the attributes in old node that also appear in new node
    # "old" must contain "new"
    updateStyle = function(old, new)
    {
        if(is.null(new)) return(old);
        result = old;
        result[names(new)] = new;
        return(result);
    }
    # Combine the style with those of parent nodes
    updateStyleUpward = function(node)
    {
        style = xmlAttrs(node)["style"];
        if(is.na(style)) style = NULL;
        style = parseStyle(style);
        style = updateStyle(defaultStyle, style);
        parentNode = xmlParent(node);
        # Recursively search the parent
        while(!is.null(parentNode))
        {
            parentStyle = xmlAttrs(parentNode)["style"];
            if(is.null(parentStyle) || is.na(parentStyle)) parentStyle = NULL;
            parentStyle = parseStyle(parentStyle);
            style = updateStyle(style, parentStyle);
            parentNode = xmlParent(parentNode);
        }
        return(style);
    }
    # Parse nodes into structured lists
    parseNode = function(node)
    {
        if(xmlName(node) == "use")
        {
            attrs = xmlAttrs(node);
            refID = sub("#", "", attrs["href"]);
            refPathNode = getNodeSet(svgFile, sprintf("//*[@id='%s']/svg:path", refID))[[1]];
            style = updateStyleUpward(refPathNode);
            style = updateStyle(style, updateStyleUpward(node));
            d = xmlAttrs(refPathNode)["d"];
            x = xmlAttrs(node)["x"];
            y = xmlAttrs(node)["y"];
        } else if(xmlName(node) == "path") {
            style = updateStyleUpward(node);
            d = xmlAttrs(node)["d"];
            x = y = 0;
        } else stop("Wrong node!");
        xy = as.numeric(c(x, y));
        names(d) = NULL;
        names(xy) = NULL;
        return(list(style = style, d = d, xy = xy));
    }
    # Flatten nodes
    expandNode = function(node)
    {
        children = xmlChildren(node);
        res = if(!length(children)) node else children;
        return(res);
    }
    nodes = unlist(xmlSApply(pathRoot, expandNode));
    names(nodes) = NULL;
    paths = lapply(nodes, parseNode);
    return(paths);
}


##' Convert a sequence of SVG files to SWF file
##'
##' Given the file names of a sequence of SVG file, this function could
##' convert them into a Flash file (.swf).
##'
##' This function uses the XML package in R and a subset of librsvg
##' (\url{http://librsvg.sourceforge.net/}) to parse the SVG file, and
##' utilize the Ming library (\url{http://www.libming.org/}) to
##' implement the conversion. Currently the support for SVG file is
##' \strong{NOT} perfect and \strong{ONLY} support files generated by
##' \code{\link[Cairo]{CairoSVG}()} in the \pkg{Cairo} package.
##'
##' @param filenames the file names of the SVG files to be converted.
##' @param outdir the output directory
##' @param output the name of the output swf file.
##' @param interval the time interval between animation frames.
##' @return The path of the generated swf file if successful.
##' @note Currently there is no function in R that could auto-generate
##' a sequence of SVG file, so you may create the SVG file once at a time.
##' See the example.
##' @export
##' @author Yixuan Qiu <\email{yixuan.qiu@@cos.name}>
##' @examples \dontrun{if(require("Cairo")) {
##'   filenames = sprintf("Rplot%03d.svg", 1:9)
##'   for(fn in filenames) {
##'     CairoSVG(fn)
##'     plot(runif(20), ylim = c(0, 1))
##'     dev.off()
##'   }
##'   output = svg2swf(filenames)
##'   swf2html(output)
##' }
##' }
##'
svg2swf = function(filenames, outdir = getwd(), output = "movie.swf",
				   interval = 1)
{
    filesdata = lapply(filenames, parseSVG);
    
    svgFile = xmlParse(filenames[1]);
    size = xmlAttrs(xmlRoot(svgFile))["viewBox"];
    size = as.numeric(unlist(strsplit(size, " ")));

    oldwd = setwd(outdir);
    .Call("svg2swf", filesdata, as.character(output), size,
		  as.numeric(interval), PACKAGE = "R2SWF");
    setwd(oldwd);
	invisible(file.path(outdir, output));
}

