# Supported file extensions:
#
# .csv: CSV files that use a comma separator.
# .csv.bz2: CSV files that use a comma separator and are compressed using bzip2.
# .csv.zip: CSV files that use a comma separator and are compressed using zip.
# .csv.gz: CSV files that use a comma separator and are compressed using gzip.
# .tsv: CSV files that use a tab separator.
# .tsv.bz2: CSV files that use a tab separator and are compressed using bzip2.
# .tsv.zip: CSV files that use a tab separator and are compressed using zip.
# .tsv.gz: CSV files that use a tab separator and are compressed using gzip.
# .wsv: CSV files that use an arbitrary whitespace separator.
# .wsv.bz2: CSV files that use an arbitrary whitespace separator and are compressed using bzip2.
# .wsv.zip: CSV files that use an arbitrary whitespace separator and are compressed using zip.
# .wsv.gz: CSV files that use an arbitrary whitespace separator and are compressed using gzip.
# .RData: .RData binary files produced by save().
# .rda: .RData binary files produced by save().
# .url: A YAML file that contains an HTTP URL and a separator specification for a remote dataset.
# .sql: A YAML file that contains database connection information for a MySQL database.
# .sav: Binary file format generated by SPSS.
# .dta: Binary file format generated by Stata.

# Current data reader functions.
CSVReader <- function(data.file, filename, variable.name)
{
  assign(variable.name,
         read.csv(filename,
                  header = TRUE,
                  sep = ','),
         envir = .GlobalEnv)
}

TSVReader <- function(data.file, filename, variable.name)
{
  assign(variable.name,
         read.csv(filename,
                  header = TRUE,
                  sep = '\t'),
         envir = .GlobalEnv)
}

WSVReader <- function(data.file, filename, variable.name)
{
  assign(variable.name,
         read.csv(filename,
                  header = TRUE,
                  sep = ' '),
         envir = .GlobalEnv)
}

RDataReader <- function(data.file, filename, variable.name)
{
  load(filename, envir = .GlobalEnv)
}

URLReader <- function(data.file, filename, variable.name)
{
  # A .url file contains YAML describing the data source.
  # An example file is shown below.
  #
  # url: "http://www.johnmyleswhite.com/ProjectTemplate/sample_data.csv"
  # separator: ","
  
  url.info <- yaml.load_file(filename)
  
  assign(variable.name,
         read.csv(url.info[['url']],
                  header = TRUE,
                  sep = url.info[['separator']]),
         envir = .GlobalEnv)
}

SQLReader <- function(data.file, filename, variable.name)
{
  # A .sql file contains YAML describing the data source.
  # An example file is shown below.
  #
  # type: mysql
  # user: sample_user
  # password: sample_password
  # host: localhost
  # dbname: sample_database
  # table: sample_table
  
  library('RMySQL')
  
  database.info <- yaml.load_file(filename)
  
  if (database.info[['type']] != 'mysql')
  {
    warn('Only databases reachable through RMySQL are currently supported.')
    assign(variable.name,
  	       NULL,
  	       envir = .GlobalEnv)
	 	return()
  }
  
  mysql.driver <- dbDriver("MySQL")
  
  connection <- dbConnect(mysql.driver,
                          user = database.info[['user']],
                          password = database.info[['password']],
                          host = database.info[['host']],
                          dbname = database.info[['dbname']])
  
  sql <- paste("SELECT * FROM `", database.info[['table']], "`", sep = '')
  
  result.set <- dbSendQuery(connection, sql)
	
	parcel.size <- 1000
	
  data.parcel <- fetch(result.set,
                       n = parcel.size)
  
	if (nrow(data.parcel) == 0)
	{
	 	assign(variable.name,
  	       NULL,
  	       envir = .GlobalEnv)
  	return()
	}
	
	assign(variable.name,
	       data.parcel,
	       envir = .GlobalEnv)
	
  while (! dbHasCompleted(result.set))
  {
    data.parcel <- fetch(result.set, n = parcel.size)
    
    if (nrow(data.parcel) > 0)
    {
      assign(variable.name,
             rbind(get(variable.name,
                       envir = .GlobalEnv),
                   data.parcel),
             envir = .GlobalEnv)
    }
  }
}

SPSSReader <- function(data.file, filename, variable.name)
{
  assign(variable.name,
         read.spss(filename),
         envir = .GlobalEnv)
}

StataReader <- function(data.file, filename, variable.name)
{
  assign(variable.name,
         read.dta(filename),
         envir = .GlobalEnv)
}

# Use a list to map file extension detection regular expressions to the
# appropriate reader functions.
extensions.dispatch.table <- list("\\.csv$" = CSVReader,
                                  "\\.csv.bz2$" = CSVReader,
                                  "\\.csv.zip$" = CSVReader,
                                  "\\.csv.gz$" = CSVReader,
                                  "\\.tsv$" = TSVReader,
                                  "\\.tsv.bz2$" = TSVReader,
                                  "\\.tsv.zip$" = TSVReader,
                                  "\\.tsv.gz$" = TSVReader,
                                  "\\.wsv$" = WSVReader,
                                  "\\.wsv.bz2$" = WSVReader,
                                  "\\.wsv.zip$" = WSVReader,
                                  "\\.wsv.gz$" = WSVReader,
                                  "\\.Rdata$" = RDataReader,
                                  "\\.rda$" = RDataReader,
                                  "\\.url$" = URLReader,
                                  "\\.sql$" = SQLReader)

data.files <- dir('data')

for (data.file in data.files)
{
  for (extension in names(extensions.dispatch.table))
  {
    filename <- file.path('data', data.file)
        
    if (grepl(extension, data.file, ignore.case = TRUE, perl = TRUE))
    {
      variable.name <- clean.variable.name(sub(extension,
                                               '',
                                               data.file,
                                               ignore.case = TRUE,
                                               perl = TRUE))

      cat(paste("Loading data set: ", variable.name, '\n', sep = ''))
      
      do.call(extensions.dispatch.table[[extension]],
              list(data.file,
                   filename,
                   variable.name))
      
      break()
    }
  }
}
