% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relax.lm.r
\name{relaxLM}
\alias{relaxLM}
\alias{relaxLM.matrix}
\alias{relaxLM.mesh3d}
\title{relax one specific landmark configuration against a reference}
\usage{
relaxLM(lm, ...)

\method{relaxLM}{matrix}(lm, reference, SMvector, outlines = NULL,
  surp = NULL, sur.name = NULL, mesh = NULL, tol = 1e-05,
  deselect = FALSE, inc.check = TRUE, iterations = 0, fixRepro = TRUE,
  missing = NULL, bending = TRUE, stepsize = ifelse(bending, 1, 0.5),
  use.lm = NULL, silent = FALSE, ...)

\method{relaxLM}{mesh3d}(lm, reference, tol = 1e-05, deselect = FALSE,
  inc.check = TRUE, iterations = 0, fixRepro = TRUE, missing = NULL,
  bending = FALSE, stepsize = ifelse(bending, 1, 0.5), use.lm = NULL,
  silent = FALSE, ...)
}
\arguments{
\item{lm}{k x 3 or k x 2 matrix containing landmark data to be slidden - or a triangular mesh of class "mesh3d". See details}

\item{...}{additonal arguments - currently unused}

\item{reference}{k x 3 or k x 2 matrix containing landmark of the reference, or a mesh with the same amount of vertices as there are landmarks in \code{lm}.}

\item{SMvector}{A vector containing the row indices of (semi-) landmarks on the curve(s) that are
allowed to slide}

\item{outlines}{A vector (or if threre are several curves) a list of vectors
(containing the rowindices) of the (Semi-)landmarks forming the curve(s) in
the successive position on the curve - including the beginning and end
points, that are not allowed to slide.}

\item{surp}{integer vector containing the row indices of semi-landmarks positioned on surfaces.}

\item{sur.name}{character: containing the filename of the corresponding
surface.When specified, mesh has to be NULL. If \code{sur.name=NULL} and \code{mesh=NULL}, the tangent planes will be estimated from the point cloud.}

\item{mesh}{triangular mesh of class "mesh3d" loaded into the R workspace,
when specified, "sur.name" has to be NULL.}

\item{tol}{numeric: Threshold for convergence in the sliding proces. Full
Procrustes distance between actual result and previous iteration.}

\item{deselect}{Logical: if TRUE, the SMvector is interpreted as those
landmarks, that are not allowed to slide.}

\item{inc.check}{Logical: if TRUE, the program stops when convergence
criterion starts increasing and reports result from last iteration.}

\item{iterations}{integer: maximum amounts the algorithm runs - even when
'tol' is not reached. When iterations=0, the algorithm runs until
convergence.}

\item{fixRepro}{logical: if \code{TRUE}, fix landmarks will also be
projected onto the surface. If you have landmarks not on the surface, select
\code{fixRepro=FALSE}}

\item{missing}{vector of integers, specifying row indices of missing (semi-)landmarks. They will be relaxed freely in 3D and not projected onto the target (works only for 2D data).}

\item{bending}{if TRUE, bending energy will be minimized, Procrustes distance otherwise (not suggested with large shape differences)}

\item{stepsize}{integer: dampening factor for the amount of sliding.
Useful to keep semi-landmarks from sliding too far off the surface.
The displacement is calculated as  \eqn{\Upsilon = \Upsilon^0 + stepsize * UT}{Y = Y0 + stepsize * UT}.
Default is set to 1 for bending=TRUE and 0.5 for bending=FALSE.}

\item{use.lm}{indices specifying a subset of (semi-)landmarks to be used in the rotation step - only used if \code{bending=FALSE}.}

\item{silent}{logical: if TRUE, console output is suppressed.}
}
\value{
returns kx3 matrix of slidden landmarks
}
\description{
relax one specific landmark configuration against a reference (e.g. a
sample mean)
}
\details{
if \code{lm} is a surface mesh, all vertices will be treated as semilandmarks and a allowed to freely slide along the surface.
}
\examples{

require(rgl)
data(nose)
### relax shornose against longnose

# define fix landmarks
fix <- c(1:5,20:21)
# define surface patch by specifying row indices of matrices
# all except those defined as fix
surp <- c(1:dim(shortnose.lm)[1])[-fix]
 
relax <- relaxLM(shortnose.lm,
         longnose.lm, mesh=shortnose.mesh, iterations=1,
         SMvector=fix, deselect=TRUE, surp=surp)

## example minimizing Procrustes distance when displacement is not
## dampened by stepsize
relaxProcD <- relaxLM(shortnose.lm,
         longnose.lm, mesh=shortnose.mesh, iterations=1,
         SMvector=fix, deselect=TRUE, surp=c(1:623)[-fix],bending=FALSE,stepsize=1)

\dontrun{
# visualize differences red=before and green=after sliding
deformGrid3d(shortnose.lm, relax, ngrid=0)

 
# visualize differences minimizing Procrusted distances red=before and green=after sliding

deformGrid3d(shortnose.lm, relaxProcD, ngrid=0)
## no smooth displacement, now let's check the distances:
rot2ref <- rotonto(relaxProcD,longnose.lm)
angle.calc(rot2ref$X,rot2ref$Y)
# 0.2492027 Procrustes distance between reference and slided shape
# (minimizing Procrustes distance)
rot2refBend <- rotonto(relax,longnose.lm)
angle.calc(rot2refBend$X,rot2refBend$Y)
# 0.2861322 Procrustes distance between reference and slided shape
# (minimizing bending energy)

rot2refOrig <- rotonto(shortnose.lm,longnose.lm)
angle.calc(rot2refOrig$X,rot2refOrig$Y)
# 0.3014957 Procrustes distance between reference and original shape
##result: while minimizing Procrustes distance, displacement is not
##guaranteed to be smooth

# add surface
wire3d(shortnose.mesh, col="white")


## finally relax two meshes with corresponding vertices:

mediumnose.mesh <- tps3d(shortnose.mesh,shortnose.lm, (shortnose.lm+longnose.lm)/2,threads=1)
## we use Procrustes distance as criterion as bending energy is pretty slow because
## of too many coordinates (more than 3000 is very unreasonable).
relaxMesh <- relaxLM(shortnose.mesh,mediumnose.mesh,iterations=2,bending=FALSE,stepsize=0.05)
}
}
\references{
Gunz, P., P. Mitteroecker, and F. L. Bookstein. 2005.
Semilandmarks in Three Dimensions, in Modern Morphometrics in Physical
Anthropology. Edited by D. E. Slice, pp. 73-98. New York: Kluwer
Academic/Plenum Publishers.
}
\seealso{
\code{\link{slider3d}}
}
\author{
Stefan Schlager
}
