% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/L1cent.R
\name{L1cent}
\alias{L1cent}
\alias{L1cent.igraph}
\alias{L1cent.matrix}
\title{L1 Centrality}
\usage{
L1cent(g, eta)

\method{L1cent}{igraph}(g, eta = NULL)

\method{L1cent}{matrix}(g, eta = NULL)
}
\arguments{
\item{g}{An \code{igraph} graph object or a distance matrix. The graph must
be undirected and connected. Equivalently, the distance matrix must be
symmetric, and all entries must be finite.}

\item{eta}{An optional nonnegative multiplicity (weight) vector for (vertex)
weighted networks. The sum of its components must be positive. If set to
\code{NULL} (the default), all vertices will have the same positive weight
(multiplicity), i.e., \code{g} is treated as a vertex unweighted graph. The
length of the \code{eta} must be equivalent to the number of vertices.}
}
\value{
A numeric vector whose length is equivalent to the number of vertices
in the graph \code{g}. Each component of the vector is the
\ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} centrality of each
vertex in the given graph.
}
\description{
Computes \ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} centrality
for each vertex. The \ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}}
centrality is a graph centrality measure defined for the vertices of a graph. It
is (roughly) defined by (1 \eqn{-} minimum multiplicity required for a selected
vertex to become the median of the graph).
}
\details{
Suppose that \code{g} is an undirected and connected graph consisting of
\eqn{n} vertices \ifelse{html}{\out{<i>v<sub>1</sub></i>, ..., <i>v<sub>n</sub></i>}}{{\eqn{v_1,\dots,v_n}}}
whose multiplicities (weights) are \eqn{\eta_1,\dots,\eta_n > 0}, respectively.

The median of this graph is the node minimizing the weighted sum of distances
(Hakimi 1964). That is,
\ifelse{html}{\out{<i>v<sub>i</sub></i>}}{{\eqn{v_i}}} is the median node if
\deqn{
 \sum_{k=1}^{n} \eta_k d(v_i, v_k)
}
is minimized, where \eqn{d(\cdot,\cdot)} denotes the geodesic (shortest path)
distance between two vertices. See \code{\link[igraph:distances]{igraph::distances()}} for algorithms for
computing geodesic distances between vertices.

The \ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} centrality for an
arbitrary node \ifelse{html}{\out{<i>v<sub>i</sub></i>}}{{\eqn{v_i}}} is
defined as \sQuote{one minus the minimum weight that is required to make it a
median.} This concept of centrality is closely related to the data depth for
ranking multivariate data, as defined in Vardi and Zhang (2000). According to
Kang and Oh (2023), it turns out that the following formula computes the
\ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} centrality for the
vertex
\ifelse{html}{\out{<i>v<sub>i</sub></i>}}{{\eqn{v_i}}}:
\deqn{
 1-\max_{j\neq i}\left\{\frac{\sum_{k=1}^{n}\eta_k (d(v_i,v_k) - d(v_j,v_k)) }{\eta_{\cdot}d(v_i,v_j)}\right\}^{+},
}
where \eqn{\{\cdot\}^{+}=\max(\cdot,0)} and \eqn{\eta_{\cdot} =
\sum_{k=1}^n \eta_k}. Hence, the
\ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} centrality of a vertex
is in \eqn{[0,1]} by the triangle inequality, and the median vertex has
centrality 1.
}
\note{
The function is valid only for undirected and connected graphs.
}
\examples{
# igraph object and distance matrix as an input lead to the same result
vertex_weight <- igraph::V(MCUmovie)$worldwidegross
cent_igraph <- L1cent(MCUmovie, eta=vertex_weight)
cent_matrix <- L1cent(igraph::distances(MCUmovie), eta=vertex_weight)
all(cent_igraph == cent_matrix)

# Top 6 vertices with the highest L1 centrality
utils::head(sort(cent_igraph, decreasing = TRUE))
}
\references{
S. L. Hakimi. Optimum locations of switching centers and the
absolute centers and medians of a graph. \emph{Operations Research},
12(3):450--459, 1964.

S. Kang and H.-S. Oh. On a notion of graph centrality based on
\ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}} data depth.
Technical Report, 2023.

Y. Vardi and C.-H. Zhang. The multivariate
\ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}}-median and
associated data depth. \emph{Proceedings of the National Academy of Sciences},
97(4):1423--1426, 2000.
}
\seealso{
\code{\link[=L1centLOC]{L1centLOC()}}, \code{\link[=L1centNB]{L1centNB()}}, \code{\link[=L1centMDS]{L1centMDS()}}, \code{\link[=L1centEDGE]{L1centEDGE()}},
\code{\link[=Lorenz_plot]{Lorenz_plot()}} for \ifelse{html}{\out{<i>L</i><sub>1</sub>}}{{\eqn{L_1}}}
centrality-based analysis.

\code{\link[igraph:betweenness]{igraph::betweenness()}}, \code{\link[igraph:closeness]{igraph::closeness()}},
\code{\link[igraph:degree]{igraph::degree()}}, \code{\link[igraph:eigen_centrality]{igraph::eigen_centrality()}} for centrality measures.
}
