% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ic.ranks.R
\name{ic.ranks}
\alias{ic.ranks}
\title{Confidence intervals for ranks}
\usage{
ic.ranks(y, sigma = rep(1, length(y)), Method = c("Exact", "Bound", "Tukey",
  "SeqTukey"), BoundChoice = c("Upper", "Lower"), ExactAlgo = c("Level",
  "Block"), alpha = 0.05, control = list(crit = NULL, trace = TRUE, adjustL
  = FALSE, adjustU = FALSE, n_adjust = n - 1, N = 10^4))
}
\arguments{
\item{y}{a real vector of observed data.}

\item{sigma}{a vector of standard deviations. If sigma is a single value, then we consider that all centers have the same standard deviation.}

\item{Method}{a character indicating the method used to produce the confidence intervals. The "Exact" produces confidence intervals using the partitioning principle and the likelihood ratio test. The "Bound" choice produces lower- or upper-bound confidence intervals (according to the "BoundChoice") for the ranks using a fast algorithm. The "Tukey" choice produces simultaneous confidence intervals for the ranks using Tukey's HSD. The "SeqTukey" produces simultaneous confidence intervals for the ranks using a sequential-rejective algorithm.}

\item{BoundChoice}{a character entry which is only relevant if the "Bound" choice is picked in the Method parameter. The default value is "Upper" which results in the upper-bound CIs for the ranks. If "Lower" is chosen, then the lower-bound CIs are generated.}

\item{ExactAlgo}{a character entry default to "Block". This parameter is only relevant when "Exact" choice in the Method parameter is chosen. This parameter gives the choice between two C++ implemented algorithms for an exact partitioning result. See below for more details.}

\item{alpha}{the significance level of the internal tests we perform (which corresponds to the FWER control of the corresponding multiple testing procedure). CIs have simultaneous significance level of 1-alpha.}

\item{control}{is a list of control parameters.}

\item{crit}{is the critical value for Tukey's HSD. If it is kept NULL, then it is calculated internally. The use of this parameter becomes handful in case the user wishes to make several simulations. By providing it, we avoid repeating a Monte-Carlo estimation of the quantile and thus we gain in execution time. In some cases (espcially when all centers have the same standard deviation), the critical value for Tukey's HSD can be found in some statistical tables.}

\item{trace}{a logical parameter which supresses the printing of the details of the method which was chosen. The default is TRUE (shows details).}

\item{adjustL}{a logical variable (default to FALSE) indicating if an adjustment on the lower bound according to the data must be considered (if possible). This choice is only relevenat if Method is chosen as "Bound" and BoundChoice is chosen as "Lower".}

\item{adjustU}{a logical variable (default to FALSE) which gives the user the choice to adjust the upper bound CIs through the parameter "n_adjust". This choice is only relevenat if Method is chosen as "Bound" and BoundChoice is chosen as "Upper".}

\item{n_adjust}{an integer-valued entry for advanced control over the lower- or upper-bound algorithms. When the "adjustL" parameter is TRUE, the new value of n_adjust is chosen automatically as the best adjustment on the lower affine bound of the chi-square quantiles according to the data. If adjustU is TRUE, then n_adjust contains the point at which the upper affine bound is tangent on the chi-square quantiles. Possible values {1,...,n-1}. If both adjustL and adjustU variables are left FALSE, then the default choice is that the lower affine bound passes between the chi-square quantiles at 1 and n-1 degrees of freedom, and the upper affine bound is tangent on n-1.}

\item{N}{the number of iterations used in order to calculate the Studentized range quantile for Tukey's algorithms.}
}
\value{
a list of two vectors containing the lower and upper bounds of the confidence intervals for the sorted observed centers.
}
\description{
This function calculates simultaneous confidence (sets) intervals (CIs) at a pre-specified level (1-alpha) for the ranks of centers mu_1,...,mu_n which are observed through a sample y using multiple testing techniques. Several possibilities are presented through a "Method" variable. There are bascially two main choices; one which uses the partitioing principle and the likelihood ratio test and the the other is based on Tukey's pairwise comparison procedure. See choices below, and for more details see the references.
}
\details{
The vector of observations need to be sorted. Otherwise, it is done internally. The observations are supposed to be realizations of independent Guassian distributions with unknown centers mu_1,...,mu_n and known standard deviations sigma = (sigma_1,...,sigma_n).

The exact-partitioning confidence intervals are calculated using two algorithms; one which corresponds to the choice "Level" and another which corresponds to the choice "Block". Both choices use an algorithm with complexity 2^n, but the "Block" algorithm is generally the faster one. In the way it is constructed (the "Block" algorithm), the execution time depends on the data and not always on the size of the data. Thus, if you are lucky, you will get the result for 50 centers in a few seconds, and if you are not, then it might take up to 2 weeks. Both algorithms are written with a C++ code in order to optimize the performance and produce results in a reasonable time for a number of centers below 50. The "Block" algorithm requires lower and upper bounds for the confidence intervals. This is automatically calculated here using the option "Bound" from the Method parameter. Hypotheses in the "Level" algorithm are represented using the combinatorial number system whereas in the "Block" algorithm we use a sparse binary representation which runs faster but not convenient to the "Level" algorithm.

The lower- and upper-bound CIs are calculated with an algorithm whose complexity is of order n^3. The bracketing obtained from the lower and upper bounds is generally very narrow with a maximum gap of 1. Moreover, in regular situations, the lower and upper bounds coincide on at least 50 percent of the centers yielding the exact-partitioning result. Thus, the bracketing is an alternative for an exact-partitioning algorithm for medium- and large-size samples (n>50). When a calculus of the lower- and upper-bound CIs is required, the default choice is when no adjustment on neither the lower nor the upper bounds is taken into account. Thus, the lower affine bound of the chi-square is a line passing by the quantiles at 1 and n-1 degrees of freedom, whereas the upper affine bound is a line tangent on the chi-square quantiles at n-1 degrees of freedom. The adjustment on the lower bound CIs can in some contexts improve on the CIs and increase the number of centers where the lower and upper bounds coincide. The best option is to adjust for both the lower and upper bounds (separately) in case a complexity of n^3 is not considered high for the problem the you solve.

Both "Tukey" and "SeqTukey" are based on multiple comparison testing and are superior to the LR-based CIs if the centers are far apart from each others and if the standard deviations are not significantly different from each others among the centers. The sequential rejective variant of Tukey's HSD rejects at least as much as Tukey's HSD and thus produces generally shorter confidence intervals for the ranks.
}
\examples{
TrueCenters = 1:50
n = 10; alpha = 0.05; sigma = runif(n,min=0.5,max=1.5)
y = as.numeric(sapply(1:n, function(ll) rnorm(1,TrueCenters[ll],sd=sigma[ll])))
ind = sort.int(y, index.return = TRUE)$ix
y = y[ind]
sigma = sigma[ind] # The sigmas need to follow the order of the y's
res = ic.ranks(y, sigma, Method = "Exact", ExactAlgo = "Level",
   alpha = 0.05, control = list(trace = TRUE))
LowerExact = res$Lower; UpperExact = res$Upper
res = ic.ranks(y, sigma, Method = "Bound", BoundChoice = "Lower",
   control = list(adjustL = FALSE, adjustU = FALSE))
LowerL = res$Lower; UpperL = res$Upper
res = ic.ranks(y, sigma, Method = "Bound", BoundChoice = "Upper",
   control = list(adjustL = FALSE, adjustU = FALSE))
LowerU = res$Lower; UpperU = res$Upper
res = ic.ranks(y, sigma, Method = "Tukey")
LowerTuk = res$Lower; UpperTuk = res$Upper
res = ic.ranks(y, sigma, Method = "SeqTukey")
LowerTukSeq = res$Lower; UpperTukSeq = res$Upper
LowerExact 
LowerTuk 
}
\references{
Simultaneous confidence sets for ranks using the partitioning principle - Technical report (Arxiv).

An improvement of Tukey's HSD with application to ranking institutions (Arxiv).
}
\author{
Diaa Al Mohamad and Jelle J. Goeman and Erik W. van Zwet. Correspondence to d.al_mohamad@lumc.nl
}
