



#' gsnHierarchicalDendrogram
#'
#' @description Generate a dendrogram plot of a hierarchical clustered set of GSNA distances. This requires an embedded
#' hierarchical cluster object of type \code{'hclust'} associated with the default or specified distance metric. Such
#' an object may be generated by running \code{gsnPareNetGenericHierarchic()} on a \code{GSNData} object prior to
#' running this function.
#'
#' The graphical output of this function can be a horizontal or circular dendrogram. When \code{show.leaves}, \code{stat_col}
#' and optionally \code{stat_col_2}, the function will output a dendrogram image with leaves colored by the significance
#' indicated in \code{stat_col} and optionally \code{stat_col_2} (with a 1 or 2 dimensional color scale). If \code{n_col}
#' is specified, the leaf sizes will be scaled by the column indicated therein.
#'
#' The function has many optional arguments, but only a few should be necessary to get a decent plot.
#'
#' @param object An object of the class \code{GSNData}
#' @param distance (optional) A character vector of length one to indicate the desired distance metric to be used for
#' generating a hierarchical dendrogram, e.g. 'lf', 'jaccard', 'stlf', etc. Defaults to the value of objects
#' \code{default_distance}.
#' @param subnet_colors (optional) A character vector of color codes matching the desired colors for subnets. If null
#' then the colors are set automatically.
#' @param filename (optional) A file for outputting a graphical image to a file as opposed to the current graphical device.
#' Output format is automatically detected from the file suffix, but can be overridden using the out_format argument.
#' (See details.)
#' @param file (optional) Synonym of filename, but deprecated. (Generates a warning.)
#' @param out_format (optional) File format of the output, either \code{'svg'}, \code{'png'}, \code{'pdf'}, or \code{'plot'}
#' (default if filename is not specified). For more information, see Details.
#' @param width (optional) Used to specify the width of the output in inches. If not specified, defaults to the current figure width.
#' @param height (optional) Used to specify the height of the output in inches. If not specified, defaults to the current figure height.
#' @param pathways_dat (optional) data.frame containing associated pathways data. This defaults to whatever pathways
#' data has already been imported into this GSNData object in \code{object$pathways$data}.
#' @param id_col (optional) Character vector of length 1 indicating the name of the column to be used
#' as an ID key in the pathways dataframe (or modules data if that is used, see below). This column should contain
#' the same values as the names of the gene sets. This defaults to the value of the pathways id_col field.
#' @param id_nchar (optional) Integer indicating the number of characters to reserve in the dendrogram plot for the
#' ID. If unspecified, it is equal to the maximal nchar of the specified ID (\code{id_col} or \code{substitute_id_col}).
#' @param pathways_title_col (optional) Character vector of length 1 indicating the name of the column in the pathways or
#' modules data.frame to be used as a Title or descriptor in the plot. If not set the function looks for the
#' following names: "Title", "Name", "NAME", "STANDARD_NAME", and takes the first that it finds. If set to NA,
#' the title part of the label is suppressed.
#' @param substitute_id_col (optional) Character vector of length 1 indicating a column used to substitute an alternative
#' ID for the labeling gene sets in data set. If set to \code{NA}, the ID in the plot is disabled.
#' @param stat_col (optional) This is the name of the column in the pathways data.frame that contains a significance
#' value for coloring network vertices. The default value is specified by \code{object$pathways$stat_col}.
#' @param stat_col_2  (optional) This is the name of an optional second column in the pathways data.frame that
#' contains a significance value for coloring network vertices in a 2-color network. The default value is specified
#' by \code{object$pathways$stat_col_2}. When specified, a 2-color network is generated. To force a 2-color network
#' to plot as a standard 1-color network using \code{stat_col} alone, use \code{stat_col_2 = NA}.
#' @param sig_order (optional) This indicates the behavior of \code{stat_col}, whether low values (\code{'loToHi'}) or
#' high values (\code{'hiToLo'}) are most significant. The default value is specified in \code{object$pathways$sig_order}.
#' @param sig_order_2  (optional) This indicates the behavior of \code{stat_col}, whether low values (\code{'loToHi'}) or
#' high values (\code{'hiToLo'}) are most significant. The default value is specified in \code{object$pathways$sig_order}.
#' @param n_col (optional) This is the name of the column in the pathways data.frame that contains a value for gene set
#' size, or any other value intended to be the bases of leaf scaling. When specified, leaf sizes will be scaled by this
#' value. (default is the value in \code{object$pathways$n_col}). An \code{NA} value can be used to override the
#' the value in \code{object$pathways$n_col} and suppress leaf scaling.
#' @param transform_function (optional) Function to transform significance values for conversion to a color scale.
#' Normally, significance values are *p*-values, and need log transformation. If there are significance values of 0,
#' these are converted to \code{-Inf} by log-transformation, so the function \code{nzLog10()} adds a small pseudocount
#' to the values to mitigate this problem, prior to log10 transformation, but for other types of data, other transformations
#' or even 'identity' may be more suitable. (default, \code{nzLog10})
#' @param font_face (optional) The font used for plot text, including leaf labels. For best results, this should be a
#' monospaced font. If not specified, the system attempts to pick a suitable default: \code{'Andale Mono'} on Mac OS X,
#' \code{'Lucida Sans Typewriter'} for Windows, and \code{'mono'} for all other systems.
#' @param color_labels_by (optional) This parameter tells the plotting function to assign colors to dendrogram
#' leaf labels on on the basis of this argument. Currently, only 'subnets' and NULL are supported arguments.
#' @param subnetColorsFunction (optional) Function for assigning colors to subnets. Only used when
#' \code{color_labels_by == 'subnet'}. The default value is \code{gsnDendroSubnetColors_dark}.
#' @param show.leaves (optional) Logical to tell the function to display leaves representing gene sets. When \code{stat_col}
#' and optionally \code{stat_col_2} are specified, naming parameters from the \code{pathways_dat} data.table, a single or
#' two-color color scale is used to represent the value of the corresponding pathways statistics.
#' @param show.legend (optional) A logical value telling the plotting function to include legends.(default: TRUE)
#' @param leaf_colors (optional) A vector containing at least 2 colors for generating a color gradient in single channel
#' visualizations. (default: c("white","yellow","red"), see details)
#' @param leaf_colors.1 (optional) A vector containing at least 2 colors for generating a color gradient in dual channel
#' visualizations. (default: c("white", "red"), see details)
#' @param leaf_colors.2 (optional) A vector containing at least 2 colors for generating a color gradient in dual channel
#' visualizations. (default: c("white", "blue"), see details)
#' @param na.color (optional) The color used for NA values. (default: "#CCCCCC")
#' @param leaf_border_color (optional) For R's open plot symbols \code{pch }\eqn{\in}\code{ ( 21, 22, 23, 24, 25 )}, supporting fill
#' with a 'bg' color, leaf border may be specified with this option. (default: "#666666")
#' @param legend.leaf.col (optional) Leaf fill color for the legend. (default: "#CCCCCC")
#' @param combine_method (optional) For dual channel plots this is a string used to indicate how colors are combined to
#' generate a two dimensional color scale. Options are "scaled_geomean" (same as "default"), "standard" (same as "euclidean" ),
#' "negative_euclidean", "mean", and "additive". See details.
#' @param use_leaf_border (optional) When automatically choosing a leaf symbol (leaves_pch), this option determines whether
#' a solid or an open symbol is used (see details).
#' @param leaves_pch (optional) Used to specify the pch symbol used to represent dendrogram leaves. (default: 22 (open square),
#' for horizontal dendrograms and dendextend version >= '1.16.0'; 15 (solid square) for horizontal dendrograms with dendextend
#' version < '1.16.0', and for circular dendrograms, 16 (solid circle))
#' @param leaf_char_shift (optional) A parameter telling the function by how many character widths to shift the leaf labels.
#' (default: 1)
#' @param render.plot (option) Logical value indicating whether to actually render the plot, or simply return a dendrogram.
#' This may be useful if graphical parameters need to be calculated but rendering is not desired. (see value)
#' @param c1.fun (optional) Function to convert the vector of numeric values represented by \code{stat_col} to a character
#' vector corresponding to colors. For dual channel plots, these colors may be combined with a second array of colors using
#' by the method specified using the \code{combine_method} parameter. If not specified, \code{c1.fun} calculated automatically
#' as a linear function.
#' @param c2.fun (optional) Same as c1.fun but for \code{stat_col_2}.
#' @param geometry (optional) Specifies either "horizontal" or "circular" type dendrogram plots. (default: horizontal)
#' @param cex (optional) Font size in cex units. This parameter is used as a basis for setting the various other font sizes
#' including those of leaf/node labels, cluster/subnet labels, and legend text sizes.
#' @param leaf_cex (optional) The cex size of the leaf symbols. This is used when \code{n_col} is not specified, i.e. there
#' is no leaf size scaling. (default: 1.5 * lab.cex)
#' @param leaf_cex_range (optional) The range of leaf sizes used in plots, from low to high. This is used when \code{n_col}
#' is specified and leaf sizes are to be scaled. This may need to be reduced if leaves overlap or are clipped on one size.
#' (default: c(0.5, 2.1))
#' @param lab.cex (optional) The cex size of dendrogram leaf labels (default: 0.9 * cex).
#' @param tree_x_size.in (optional) For horizontal dendrograms, this is the width of the dendrogram in inches, not including
#' leaf labels, cluster brackets, or legends. (default: 2)
#' @param legend_x_size.in (optional) The width of legends in inches. (default: 2)
#' @param left_margin.in (optional) The width of the left margin in inches. Ignored if .plt.plot or .mai.plot is specified.
#' (default: 0)
#' @param right_margin.in (optional) The width of the right margin of the dendrogram in inches. Ignored if .plt.plot or
#' .mai.plot is specified. If unspecified, this is calculated automatically as \code{width - tree_x_size.in}.
#' @param top_margin.in (optional) The width of the top margin of the dendrogram in inches. Ignored if .plt.plot or
#' .mai.plot is specified. (default: if no \code{main} argument is specified, 0. If a \code{main} argument is specified,
#' then it is calculated as \code{cex.main * par('cin')[2] * mar.main})
#' @param bottom_margin.in (optional) (optional) The width of the bottom margin in inches. Ignored if .plt.plot or
#' .mai.plot is specified. (default: 0)
#' @param legend.downshift.in (optional) Argument shifting the legend downward, in inches. This is useful for adjusting
#' the alignment of the legend(s) with the top of the plot. (default: for horizontal dendrograms, 0; for circular
#' dendrograms, 0.42)
#' @param .plt.plot (optional) Specifies the plot region of the output using \code{figure} coordinates, and excluding the
#' legends. This can provide a greater degree of control for plotting, but most users will not need to adjust this. See
#' the \code{plt} argument of the par graphics function for more information.
#' @param .mai.plot (optional) A parameter specifying the margins of the plot, excluding legends as inches. This is calculated
#' automatically and for most purposes, will not need to be specified.
#' @param bkt_lmargin_chars (optional) Width in character widths of the space between the leaf labels and the brackets
#' indicating cluster/subnet groups. If the leaf labels need more space, this can be increased. (default: 4)
#' @param legend_spacing.x.in (optional) Space between plot and legend in inches. With some plot configurations, it may be
#' useful to use negative values to bring the legends closer to the plot region. (default: 2 character widths)
#' @param legend_spacing.y.in (optional) Space between legends in inches. (default: 1 character height)
#' @param legend.lab.cex (optional) Legend x and y label size in cex. If unspecified, the function tries to pick a reasonable
#' value based on available space.
#' @param legend.axis.cex (optional) Legend axis label size in cex. If unspecified, the function tries to pick a reasonable
#' value based on available space.
#' @param legend.free.cex.bool (optional) Logical allowing independent optimized sizing of legend label font sizes if TRUE.
#' (default: FALSE)
#' @param main (optional) Legend main title. (default: NULL)
#' @param cex.main (optional) Font size in cex units for the main title. (default: \code{1.35 * cex})
#' @param mar.main (optional) Tells the function to reserve this many line heights for the main title. (default: 3.2)
#' @param lines.main (optional) Tells the function to place the main title this many lines away from the plot edge.
#' (default: 1.5)
#' @param colors.n (optional) The number of colors per dimension of the color scale. For single channel plots, this will be
#' equal to the number of colors in the color scale. For 2 channel plots, the number of colors is the square of this number.
#' (default 100).
#' @param legend.bg (option) The color of the legend background. (default: \code{par('bg')})
#' @param legend.fg (option) The color of the legend foreground. (default: \code{par('fg')})
#' @param resolution Image resolution in pixels per inch, only for bitmap image output formats (currently
#' png only). (default: 72)
#' @param draw.legend.box.bool (option) Logical indicating whether bounding boxes should be drawn for the legends.
#' @param DO_BROWSER (option) Logical indicating whether browser() should be run for this function. (For debugging
#' purposes, will probably remove.)
#'
#' @return An object of type 'dendrogram', with the attribute "GSNA_plot_params" containing a list of plot parameters.
#' This list is useful for retrieving plot parameters set by the function, so that they might be optimized. Likewise,
#' the dendrogram object itself can be replotted or analyzed by other means.
#'
#' @export
#'
#' @details Outputs of type pdf, png, and svg are supported for file outputs. File type is automatically detected
#' from the file suffix, but can be overridden using the \code{out_format} argument.
#'
#' Open symbols (with border and a fill color, \code{pch }\eqn{\in}\code{ ( 21, 22, 23, 24, 25 )}) are
#' used by default on dendextend versions < '1.16.0' for horizontal dendrograms. For earlier versions, and with circular
#' dendrograms, open symbols are currently unsupported.
#'
#'
#' @seealso
#'  \code{\link{gsnPareNetGenericHierarchic}}
#'  \code{\link{gsnPlotNetwork}}
#'
#' @importFrom grDevices svg dev.off png pdf
#' @importFrom graphics par strwidth
#' @importFrom dendextend labels<-
#' @importFrom utils packageVersion
#'
# @importFrom stats plot.dendrogram labels.dendrogram
gsnHierarchicalDendrogram <- function( object,
                                       distance = NULL,
                                       subnet_colors = NULL,
                                       filename = NULL,
                                       file = NULL,          # DEPRECATING
                                       out_format = NULL,

                                       width = NULL,
                                       height = NULL,
                                       .mai.plot = NULL,
                                       cex = par('cex'),

                                       subnetColorsFunction = gsnDendroSubnetColors_dark,
                                       id_col = NULL,
                                       id_nchar = NULL,
                                       pathways_title_col = c("Title", "Name", "NAME", "STANDARD_NAME" ),
                                       substitute_id_col = NULL,

                                       font_face = NULL,     #"Andale Mono", 'Courier', 'mono', Lucida Sans Typewriter"

                                       #modules = NULL,

                                       color_labels_by = "subnet",
                                       show.leaves = FALSE,

                                       show.legend = TRUE,

                                       pathways_dat = NULL,
                                       stat_col = NULL,
                                       stat_col_2 = NULL,  # NA suppresses 2-color plotting behavior
                                       sig_order = NULL,
                                       sig_order_2 = NULL,

                                       n_col = NULL,

                                       #optimal_extreme = NULL,
                                       transform_function = nzLog10,

                                       leaf_colors = c("white","yellow","red"),
                                       leaf_colors.1 = c("#FFFFFF", "red" ),
                                       leaf_colors.2 = c("#FFFFFF", "blue" ),

                                       leaf_border_color = "#666666",
                                       legend.leaf.col = "#CCCCCC",

                                       combine_method = "scaled_geomean" , # Same as "default"
                                       use_leaf_border = TRUE,

                                       render.plot = TRUE,
                                       c1.fun = NULL,
                                       c2.fun = NULL,

                                       geometry = "horizontal", # Also, "circular",
                                       .plt.plot = NULL,    # NEW
                                       #.plt = NULL,
                                       #.plt2 = NULL,
                                       leaves_pch = NULL,  # NEW

                                       leaf_char_shift = 1, # NEW
                                       #dLeaf = NULL,  # NEW
                                       na.color = "#CCCCCC",
                                       leaf_cex = NULL,
                                       leaf_cex_range = c(0.5, 2.1),
                                       lab.cex = NULL,
                                       tree_x_size.in = 2.0, #NEW

                                       #min_legend_x_size.in = 2, # NEW
                                       legend_x_size.in = 2,
                                       left_margin.in = 0, # NEW
                                       right_margin.in = NULL, # NEW DEFAULTS TO SIZE OF LEGEND
                                       top_margin.in = NULL, # NEW
                                       bottom_margin.in = 0, # NEW

                                       legend.downshift.in = NULL,
                                       #legend.downshift.in = 0.42,

                                       bkt_lmargin_chars = 4,

                                       legend_spacing.x.in = 2 * par('cin')[1],
                                       legend_spacing.y.in = par('cin')[2],
                                       legend.lab.cex = NULL,
                                       legend.axis.cex = NULL,
                                       legend.free.cex.bool = FALSE,

                                       main = NULL,
                                       cex.main = NULL,
                                       mar.main = 3.2,          # NEW reserve this many lines for main
                                       lines.main = 1.5,        # Position Main this many lines from plot

                                       colors.n = 100,          # Color levels on legend #NEW

                                       legend.bg = par('bg'),
                                       legend.fg = par('fg'),

                                       resolution = 72, # pixels per inch
                                       draw.legend.box.bool = TRUE,
                                       DO_BROWSER = FALSE

){
  if(DO_BROWSER) browser()

  # Backup par, so that original settings are restored on exit:
  .par.orig <- par( no.readonly = TRUE )
  on.exit( add = TRUE, expr = par(.par.orig) )

  # This is used to fix the problem of labels.dendrogram and plot.dendrogram not
  # being exported from stats
  # Suggested by: Tom Shafer at https://tshafer.com/blog/2020/08/r-packages-s3-methods
  requireNamespace("stats", quietly = TRUE)

  stopifnot( "GSNData" %in% class( object ) )
  transform_function.name <- deparse( substitute( transform_function ) )

  if( is.null( distance ) ) distance <- object$default_distance
  if( is.null( distance ) ) stop( 'Need distance argument.' )

  # If file is used, give a warning.
  if( ! is.null( file ) ){
    warning( "file argument is deprecated. Use filename instead." )
    if( is.null( filename ) ) filename <- file
  }

  if( is.null( font_face ) ){ # Pick a font. Sans serif monospaced if possible.
    .system <- Sys.info()['sysname']
    if( .system == 'Darwin' ){
      font_face <-  "Andale Mono"
    } else if ( .system == 'Windows' ) {
      font_face <- "Lucida Sans Typewriter"
    } else {
      font_face <- "mono"
    }
  }

  if( is.null( cex.main ) ) cex.main <- 1.35 * cex

  # If sig_col or stat_col_2 are specified, check that sig_order / sig_order_2 is specified.
  if( ( !is.null( stat_col ) && is.null( sig_order ) ) )
    stop("If stat_col is specified, so must sig_order be. ('loToHi' or 'hiToLo')")
  if( !is.null( stat_col_2 ) )
    if( !is.na( stat_col_2 ) && is.null( sig_order_2 ) )
      stop("If stat_col_2 is specified (other than NA), so must sig_order_2 be. ('loToHi' or 'hiToLo')")

  # Check if pathways_dat has been defined.
  if( is.null(pathways_dat) ) pathways_dat <- object$pathways$data
  if( !is.null(pathways_dat) ){
    if( is.null(id_col) ) id_col <- object$pathways$id_col
    if( is.null(stat_col) ) stat_col <- object$pathways$stat_col
    if( is.null(sig_order) ) sig_order <- object$pathways$sig_order
    if( is.null(id_col) ) stop( "id_col is not defined" )
    if( is.null(stat_col) ) stop( "stat_col is not defined" )
    if( is.null(sig_order) ) stop( "sig_order is not defined" )
    # stat_col_2 and sig_order_2 for two-channel networks:
    if( is.null(stat_col_2) ){
      stat_col_2 <- object$pathways$stat_col_2
    } else if( is.na(stat_col_2) ){
      stat_col_2 <- NULL;
    }
    if( is.null(sig_order_2) ) sig_order_2 <- object$pathways$sig_order_2
    if( is.null(sig_order_2) ) sig_order_2 <- object$pathways$sig_order
    if( is.null( n_col ) ) {
      n_col <- object$pathways$n_col
    } else if( is.na( n_col ) ){
      n_col <- NULL
    }
    rownames(pathways_dat) <- pathways_dat[[id_col]]
  } else stop("Need pathways data.")
  # } else { # Use data from modules when data from pathways is not available
  #   if( !is.null( modules ) ){
  #     if( 'tmod' %in% class(modules) ){
  #       pathways_dat <- modules$MODULES
  #     } else {
  #       pathways_dat <- modules
  #     }
  #     if( ! is.null( id_col ) )
  #       rownames(pathways_dat) <- pathways_dat[[id_col]]
  #     if( is.null( pathways_title_col ) ) pathways_title_col <- 'Title'
  #} else stop("Need pathways data.")
  #}
  #if( is.null( optimal_extreme ) ) optimal_extreme <- object$distances[[distance]]$pared_optimal_extreme
  #if( is.null( optimal_extreme ) ) optimal_extreme <- object$distances[[distance]]$optimal_extreme
  #if( is.null( optimal_extreme ) ) stop( "optimal_extreme is not defined" )

  # This may be elevated to a function argument if I can get it to work.
  par.font <- par('font')    # Font style (1=plain, 2=italic, 3=bold, 4=bold-italic, etc.)

  # Scan for pathways_title_column.
  if( ! is.null(pathways_title_col) && ! all(is.na(pathways_title_col)) ){
    pathways_title_col <- pathways_title_col[pathways_title_col %in% colnames( pathways_dat )][[1]]
    if(length(pathways_title_col) == 0)
      stop("Cannot find pathways Title column.\nSet correct pathways column with pathways_title_column='NAME'",
           " or opt out with pathways_title_column=NULL.\n")
  }

  if( is.null( object$distances[[distance]]$hclust ) )
    stop("Cannot find embedded hclust object for distance '", distance, "'\\nDid you run gsnPareNetGenericHierarchic()?")

  GSN.dend <- stats::as.dendrogram( object$distances[[distance]]$hclust )
  subnets.lf <- with( object$distances[[distance]]$vertex_subnets, structure( subnet, names = vertex ) )


  # Get vertex names first... because we're going to change them.
  #vertex.names <- stats:::labels.dendrogram(GSN.dend)
  vertex.names <- labels(GSN.dend)

  # Adjust sizes of labels
  if( is.null( lab.cex )) lab.cex <- ifelse( !is.null( cex ), cex * 0.9, 0.9 )
  GSN.dend <- dendextend::set( dend = GSN.dend, "labels_cex", lab.cex )

  # Pick a value for leaf_cex if unset:
  if( is.null( leaf_cex ) ) leaf_cex <- 1.6 * lab.cex

  # Assign substitute_id_col if unset. We'll be using substitute_id_col henceforth:
  if( is.null(substitute_id_col) ) substitute_id_col <- id_col
  if( !is.na( substitute_id_col ) && !is.null(pathways_title_col) && !is.na( pathways_title_col ) ){
    if( is.null( id_nchar ) ) id_nchar <- with(pathways_dat[vertex.names,], max( nchar( get(substitute_id_col) ) ) )
    GSN.dend <- dendextend::`labels<-`( GSN.dend,
                                        value = with(pathways_dat[vertex.names,],
                                                     sprintf( paste0("%-", id_nchar, "s %s"),
                                                              string = get(substitute_id_col),
                                                              get(pathways_title_col) ) ) )
  } else if ( !is.na( substitute_id_col ) ){
    GSN.dend <- dendextend::`labels<-`(GSN.dend,
                                       value = sprintf("%s", pathways_dat[vertex.names, substitute_id_col] ) )
  } else if ( !is.na( pathways_title_col ) ){
    GSN.dend <- dendextend::`labels<-`(GSN.dend,
                                       value = sprintf("%s", pathways_dat[vertex.names,pathways_title_col] ) )
  } else
    GSN.dend <- dendextend::`labels<-`(GSN.dend, value = "" )


  # Calculate the figure height and width, if not specified.
  # Start with the width of the labels:
  max_label_width.in <- max( graphics::strwidth( s = labels(GSN.dend),
                                                 #s = stats:::labels.dendrogram(GSN.dend),
                                                 units = "inches",
                                                 cex = sapply( dendextend::get_leaves_nodePar( GSN.dend ) ,
                                                               function(x) ifelse( ! is.null( x[['lab.cex']] ), x[['lab.cex']], lab.cex )),
                                                 family = font_face,
                                                 font = par.font ), na.rm = TRUE )

  x_size_needed.in <-( tree_x_size.in +                                     # Set in arguments
                       max_label_width.in +                                 # Calculated on the basis of the labels
                       ( leaf_char_shift + 10 ) * lab.cex * par('cin')[1] + # 5 chars (cluster brackets) + 2 chars label shift + 3 extra
                       legend_x_size.in )                                   # Set in arguments

  min_fig_width.in <- 7
  x_scale_factor <- 1.1

  if( is.null( width ) ){
    width <- x_scale_factor * max( x_size_needed.in,  min_fig_width.in )
  }

  if( is.null( height ) ){
    if( geometry == "horizontal" ){
      height <- length(object$distances[[distance]]$vertices) * 0.16 +
        ifelse( !is.null( main ), mar.main * cex.main * par('cin')[2], 0 )
      #if( is.null( legend.downshift.in ) ) legend.downshift.in = 0.0
    } else if( geometry == "circular" ){
      height <- width - legend_x_size.in
      #if( is.null( legend.downshift.in ) ) legend.downshift.in = 0.42
    }
  }

  if( is.null( legend.downshift.in )) legend.downshift.in <- if( geometry == "circular" ) 0.42 else 0.0


  if( height < 0 ) stop( "Insufficient canvas size to plot: height: ", height, ", width:", width  )

  if( !is.null( n_col ) && !is.null( stat_col ) ){ # If both legends are present, need more height
    if( height < 4.5 ) height <- 4.5
  }

  legend.aspect <- 1
  legend.xlab <- NULL
  legend.ylab <- NULL

  if( ! is.null( color_labels_by ) ){
    if( color_labels_by == "subnet" ){
      # If subnet_colors is null, generate new set of colors:
      if( is.null(subnet_colors) ){
        subnet_colors <- subnetColorsFunction( subnets = object$distances[[distance]]$subnets )
      }

      # If it's too short, add black for unspecified subnet.
      if( length( unique( subnets.lf ) ) > length( subnet_colors ) ){
        subnet_colors <- structure( c( subnet_colors, rep( "#000000", length( unique( subnets.lf ) ) - length( subnet_colors ) ) ),
                                    names = unique( subnets.lf ) )
      }
      labels_colors <- subnet_colors[subnets.lf[vertex.names]]
    } else {
      stop("color_labels_by = '", color_labels_by, "' is unsupported.")
    }
  } else {
    subnet_colors <- character()
    subnet_colors[names( object$distances[[distance]]$subnets )] <- "#000000"
    labels_colors <- subnet_colors[subnets.lf[vertex.names]]
  }

  twoColorEncode.fun <- NULL
  oneColorEncode.fun <- NULL
  sizeEncode.fun <- NULL

  if( show.leaves ){
    if( ! is.null( pathways_dat ) ){
      if( !is.null( stat_col_2 ) ){
        numbers.1 <- c(loToHi=-1, hiToLo = 1)[[as.character(sig_order)]] * transform_function(pathways_dat[[stat_col]])
        numbers.2 <- c(loToHi=-1, hiToLo = 1)[[as.character(sig_order_2)]] * transform_function(pathways_dat[[stat_col_2]])

        twoColorEncode.fun <- makeTwoColorEncodeFunction( numbers.1 = numbers.1,
                                                          numbers.2 = numbers.2,
                                                          colors.1 = leaf_colors.1,
                                                          colors.2 = leaf_colors.2,
                                                          combine_method = combine_method,
                                                          na.color = na.color )

        pathways_dat$leaf_color <- twoColorEncode.fun( numbers.1 = numbers.1, numbers.2 = numbers.2, output_as = "rgb" )
        legend.xlab <- stat_col_2
        legend.ylab <- stat_col
      } else {
        numbers <- c(loToHi=-1, hiToLo = 1)[[as.character(sig_order)]] * transform_function(pathways_dat[[stat_col]] )
        oneColorEncode.fun <- makeOneColorEncodeFunction( numbers = numbers, colors = leaf_colors, na.color = na.color )
        pathways_dat$leaf_color <- oneColorEncode.fun( numbers = numbers, output_as = "rgb" )
        legend.ylab <- stat_col
      }

      if( !is.null( n_col ) ){
        gs_numbers <- pathways_dat[[n_col]]
        sizeEncode.fun <- make_lv_sizing_function( numbers = gs_numbers, size_range = leaf_cex_range )
        pathways_dat$leaf_cex <- sizeEncode.fun( x = gs_numbers )
        legend.lab.n_col <- n_col
      }
    }
    leaves_colors <- pathways_dat[vertex.names, "leaf_color"]
    if( !is.null( pathways_dat$leaf_cex ) )
      leaves_cex <- pathways_dat[vertex.names, "leaf_cex"]
    else leaves_cex <- leaf_cex


    # Decide on leaves_pch.
    if( is.null( leaves_pch ) ){
      if( geometry == "circular" ){
        leaves_pch <- 16
      } else if( utils::packageVersion("dendextend") < '1.16.0' || ! use_leaf_border ){
        leaves_pch <- 15
      } else {
        leaves_pch <- 22
      }
    } else if( leaves_pch %in% c( 21, 22, 23, 24, 25 ) &&
               utils::packageVersion("dendextend") < '1.16.0' & use_leaf_border ) {
      warning("Warning: Open circles not supported with dendextend version < 1.16.0, substituting a solid.")
      leaves_pch <- c("21" = 19, "22" = 15, "23" = 18, "24" = 17, "25" = 18 )[[as.character(leaves_pch)]]
    }

    if( leaves_pch %in% c( 21, 22, 23, 24, 25 ) ){ # Special opem symbols use background:
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_pch", leaves_pch )
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_bg", leaves_colors )
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_col", leaf_border_color )
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_cex", leaves_cex )
    } else {                                       # Regular solid symbols
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_pch", leaves_pch )
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_col", leaves_colors )
      GSN.dend <- dendextend::set( dend = GSN.dend, what = "leaves_cex", leaves_cex )
    }
  }

  # Asign label colors:
  dendextend::labels_colors(GSN.dend) <- labels_colors


  if( render.plot ){
    if( is.null(out_format) ){
      if( ! is.null( filename ) ){
        if( stringr::str_detect( string =  filename, pattern = stringr::regex( "\\.svg$", ignore_case = TRUE) ) ){
          out_format <- "svg"
        } else if( stringr::str_detect( string =  filename, pattern = stringr::regex( "\\.pdf$", ignore_case = TRUE) ) ){
          out_format <- "pdf"
        } else if( stringr::str_detect( string =  filename, pattern = stringr::regex( "\\.png$", ignore_case = TRUE) ) ){
          out_format <- "png"
        } else {
          stop( "Need to specify output type." )
        }
      } else {
        out_format <- 'plot'
      }
    } else if ( out_format %in% c("svg", "pdf", "png") & is.null(filename) ){
      stop( "filename argument needed for svg, pdf, or png" )
    }

    # Default is 'plot' to current device.
    do_nothing <- function(file = NULL, width = width, height = height){}
    out_fun <- do_nothing
    close_fun <- do_nothing

    if( out_format == "svg" ){
      out_fun <- grDevices::svg
      close_fun <- grDevices::dev.off
    } else if( out_format == "pdf" ){
      out_fun <- grDevices::pdf
      close_fun <- grDevices::dev.off
    }  else if( out_format == "png" ){
      #out_fun <- grDevices::png
      out_fun <- function( height, width, ... ) grDevices::png( height = height * resolution, width = width * resolution, res = resolution, ... )
      close_fun <- grDevices::dev.off
    }

    # Open new output device if appropriate.
    if( render.plot ) out_fun( file = filename, width = width, height = height  )

    if( geometry == "horizontal" ){
      if( is.null( .plt.plot ) ){
        if( is.null( .mai.plot ) ) {
          #if( is.null( left_margin.in ) ) left_margin.in <- 0
          if( is.null( right_margin.in ) )
            right_margin.in <- width - tree_x_size.in
          #right_margin.in <- legend_x_size.in + max_label_width.in

          if( right_margin.in < max_label_width.in + legend_x_size.in ) {
            warning( "Plot may be truncated." )
          }

          if( left_margin.in < 0 ) {
            left_margin.in <- 0
            warning( "Adjusting Left Margin. Plot may be truncated." )
          }
          if( is.null( top_margin.in ) ){
            top_margin.in <- 0 + if( is.null( main ) ) 0 else cex.main * par('cin')[2] * mar.main
          }


          .mai.plot <- c( bottom_margin.in, left_margin.in, top_margin.in, right_margin.in )
          #warning( ".mai.plot: ", paste0( collapse = ",", .mai.plot ) )
        }
        graphics::par(mai = .mai.plot, cex = cex, family = font_face )

        # Adjust the labels to have a space prior equal to leaf_char_shift X the width of a space, then plot:
        label_shift <- paste0( rep(" ", leaf_char_shift ), collapse = "" )


        # stats:::plot.dendrogram( dendextend::`labels<-`( object = GSN.dend,
        #                                                  value = paste0( label_shift, stats:::labels.dendrogram( GSN.dend ))),
        #                          cex = max( leaf_cex_range, na.rm = TRUE ),
        #                          horiz = TRUE )

        plot( dendextend::`labels<-`( object = GSN.dend,
                                      value = paste0( label_shift, labels( GSN.dend ))),
              cex = max( leaf_cex_range, na.rm = TRUE ),
              horiz = TRUE )

        # Figure out position of subnets indicator brackets, calculate .plt.bar:
        cex_multiplier <- 1

        label_shift.fig <- graphics::strwidth( s = label_shift,
                                               units = "figure",
                                               cex = lab.cex,
                                               family = font_face,
                                               font = par.font ) # Width of label_shift

        # label_widths.fig <- graphics::strwidth( s = stats:::labels.dendrogram(GSN.dend),
        #                                         units = "figure",
        #                                         cex = sapply( dendextend::get_leaves_nodePar( GSN.dend ) ,
        #                                                       function(x) ifelse( ! is.null( x[['lab.cex']] ), x[['lab.cex']], lab.cex )),
        #                                         family = font_face,
        #                                         font = par.font )
        label_widths.fig <- graphics::strwidth( s = labels(GSN.dend),
                                                units = "figure",
                                                cex = sapply( dendextend::get_leaves_nodePar( GSN.dend ) ,
                                                              function(x) ifelse( ! is.null( x[['lab.cex']] ), x[['lab.cex']], lab.cex )),
                                                family = font_face,
                                                font = par.font )

        bar_lmargin.fig <- graphics::strwidth( s = "X", units = "figure", cex = cex, family = font_face, font = par.font )

        width.bar.fig <- graphics::strwidth( s = "12345", units = "figure", cex = cex, family = font_face, font = par.font ) * cex_multiplier

        # Figure out position of subnets indicator brackets, calculate .plt.bar
        # starting with current usr and plt:
        usr <- par('usr')

        # Calculate .plt.bar
        .plt.bar <- par('plt')
        .plt.bar[1] <- par('plt')[2] + (max( label_widths.fig ) + label_shift.fig + bar_lmargin.fig * bkt_lmargin_chars ) * cex_multiplier
        if( .plt.bar[1] > 1 - ( legend_x_size.in / width ) - width.bar.fig ){
          .plt.bar[1] <- 1 - ( legend_x_size.in / width ) - width.bar.fig
          warning( "Label text is too long. Adjust font size." )
        }
        .plt.bar[2] <- .plt.bar[1] + width.bar.fig

        # Set par for plotting the subnets bar
        par( plt = .plt.bar, new = TRUE ) # This will be restored by on.exit call

        plot.window( xlim = c(0,1), ylim = c(usr[3], usr[4]), yaxs = "i" )

        # Generate Brackets to show subnets:
        if( !is.null( subnets.lf ) ){
          brackets.coords <- get_brackets_coords( subnets.lf = subnets.lf, leaf.names = vertex.names, labels_colors = labels_colors )

          # Draw brackets.
          with( brackets.coords,
                segments( x0 = c( x1, x2, x2 ),
                          y0 = c( y1 + 0.10, y1 + 0.10, y2 - 0.10 ),
                          x1 = c( x2, x2, x1 ),
                          y1 = c( y1 + 0.10, y2 - 0.10, y2 - 0.10 ),
                          col = bracket_color,
                          lwd = 2 ) )

          with( brackets.coords,
                text( x = x3, y = y3, labels = cluster, col = bracket_color, cex = cex * 1.15, font = 1 ) )
          #text( x = x3, y = y3, labels = cluster, col = bracket_color, cex = cex * 2.0, font = 2 ) )
        }
        .plt.plot <- c( 0, .plt.bar[2], .plt.bar[3], .plt.bar[4] )
      }
    } else if (geometry == "circular") {
      if( is.null( .plt.plot ) ){
        if( is.null( .mai.plot ) ) .mai.plot <- c(0,0,0, legend_x_size.in )
        graphics::par(mai = .mai.plot, cex = cex, family = font_face )
        dendro <- renderCircularDendrogram( dendro = GSN.dend,
                                            leaf.names = vertex.names,
                                            subnets.lf = subnets.lf,
                                            labels.cex = lab.cex,
                                            labels_colors = labels_colors,
                                            family = font_face,
                                            cex = cex,
                                            clear = TRUE  )
        .plt.plot <- attr( x = dendro, which = ".plt.plot", exact = TRUE )
      }
    }

    if( ! is.null(twoColorEncode.fun) || ! is.null(oneColorEncode.fun) ){
      if( isTRUE( all.equal(  transform_function, identity )) ){
        if( ! is.null( sig_order_2 ) )
          legend.xlab <- paste0( c(loToHi='-', hiToLo = '')[[as.character(sig_order_2)]], legend.xlab )
        if( ! is.null( sig_order ) )
          legend.ylab <- paste0( c(loToHi='-', hiToLo = '')[[as.character(sig_order)]], legend.ylab )
      } else {
        if( ! is.null( sig_order_2 ) )
          legend.xlab <- paste0( c(loToHi='-', hiToLo = '')[[as.character(sig_order_2)]], transform_function.name, "(", legend.xlab, ")" )
        if( ! is.null( sig_order ) )
          legend.ylab <- paste0( c(loToHi='-', hiToLo = '')[[as.character(sig_order)]], transform_function.name, "(", legend.ylab, ")" )
      }
    }

    if( show.legend ){
      plt.l <- list()
      legend.lab.cex.v <- numeric()
      legend.axis.cex.v <- numeric()
      plt.idx <- 0

      # Set up layout for legends.
      # Space betwen plot and legends
      legend_spacing.x.fig <- legend_spacing.x.in / width
      # Space between legends
      legend_spacing.y.fig <- legend_spacing.y.in / height

      legend_left_x.fig <- .plt.plot[2] + legend_spacing.x.fig          # Align right edge
      legend_top_y.fig <- .plt.plot[4] - legend.downshift.in / height   # & top edge of plot
      legend_x_size.fig <- legend_x_size.in / width

      # Go through process of generating legends without rendering to determine geometries.
      # Align top legend with top and right edge of plot .plt.plot[4] and .plt.plot[2]
      # Optimize Leaf Color Legend
      if( !is.null( twoColorEncode.fun ) ){
        legend.dat <- make2ColorLegend(numbers.1 = numbers.1,
                                       numbers.2 = numbers.2,
                                       twoColorEncode.fun = twoColorEncode.fun,
                                       n = colors.n,
                                       lab.1 = legend.ylab,
                                       lab.2 = legend.xlab,
                                       .plt.leg = c( legend_left_x.fig,
                                                     legend_left_x.fig + legend_x_size.fig,
                                                     0,
                                                     legend_top_y.fig
                                       ),
                                       cex.lab = legend.lab.cex,
                                       cex.axis = legend.axis.cex,
                                       draw.legend.box.bool = draw.legend.box.bool,
                                       h.adjust = "left",
                                       render.bool = FALSE,
                                       optimize.legend.size = TRUE
        )
        plt.idx <- plt.idx + 1
        plt.l[[plt.idx]] <- legend.dat$.plt.adj
        legend.lab.cex.v[[plt.idx]] <- legend.dat$cex.lab
        legend.axis.cex.v[[plt.idx]] <- legend.dat$cex.axis
        legend_top_y.fig <- legend.dat$.plt.adj[3] - legend_spacing.y.fig
      } else if (! is.null( oneColorEncode.fun ) ){
        legend.dat <- make1ColorLegend( numbers = numbers,
                                        oneColorEncode.fun = oneColorEncode.fun,
                                        n = colors.n,
                                        lab = legend.ylab,
                                        .plt.leg = c( legend_left_x.fig,
                                                      legend_left_x.fig + legend_x_size.fig,
                                                      0,
                                                      legend_top_y.fig
                                        ),
                                        cex.lab = legend.lab.cex,
                                        cex.axis = legend.axis.cex,
                                        draw.legend.box.bool = draw.legend.box.bool,
                                        h.adjust = "left",
                                        #v.adjust = "center",
                                        render.bool = FALSE,
                                        optimize.legend.size = TRUE )
        plt.idx <- plt.idx + 1
        plt.l[[plt.idx]] <- legend.dat$.plt.adj
        legend.lab.cex.v[[plt.idx]] <- legend.dat$cex.lab
        legend.axis.cex.v[[plt.idx]] <- legend.dat$cex.axis
        legend_top_y.fig <- legend.dat$.plt.adj[3] - legend_spacing.y.fig
      }
      # Optimize Leaf Size Legend
      if( !is.null(sizeEncode.fun) ){
        legend.dat <- makeLeafSizeLegend( numbers = gs_numbers,
                                          sizeEncode.fun = sizeEncode.fun,
                                          #.plt = .plt2,
                                          .plt.leg = c( legend_left_x.fig,
                                                        legend_left_x.fig + legend_x_size.fig,
                                                        0,
                                                        legend_top_y.fig
                                          ),
                                          legend.lab = n_col,
                                          pch = leaves_pch,
                                          cex.ticks = cex,
                                          leaf_border_color = leaf_border_color,
                                          leaf.col = legend.leaf.col,
                                          legend.fg = legend.fg,
                                          legend.bg = legend.bg,
                                          h.adjust = 'left',
                                          v.adjust = 'top',
                                          render.bool = FALSE,
                                          optimize.legend.size = TRUE )
        plt.idx <- plt.idx + 1
        plt.l[[plt.idx]] <- legend.dat$.plt.adj
        legend.lab.cex.v[[plt.idx]] <- legend.dat$cex.lab
        legend.axis.cex.v[[plt.idx]] <- legend.dat$cex.ticks
        legend_top_y.fig <- legend.dat$.plt.adj[3] - legend_spacing.y.fig
      }

      if( length( plt.l ) > 0 ){
        widest_plt_idx <- which.max( sapply( X = plt.l, FUN = function( x ) x[2] - x[1] ) )
        widest_plt <- plt.l[[widest_plt_idx]]
        if( widest_plt[2] > 1 ) widest_plt[2] <- 1
        # Adjust plt.l
        plt.l <- lapply( X = plt.l, FUN = function(x){ x[2] <- widest_plt[2]; x[1] <- widest_plt[1]; x } )

        if( is.null( legend.lab.cex ) && !legend.free.cex.bool ) legend.lab.cex <- min( legend.lab.cex.v )
        if( is.null( legend.axis.cex ) && !legend.free.cex.bool ) legend.axis.cex <- min( legend.axis.cex.v )
      }
      # Render Legend
      plt.idx <- 1

      # Render Leaf Color Legend
      if( !is.null( twoColorEncode.fun ) ){
        make2ColorLegend(numbers.1 = numbers.1,
                         numbers.2 = numbers.2,
                         twoColorEncode.fun = twoColorEncode.fun,
                         n = colors.n,
                         lab.1 = legend.ylab,
                         lab.2 = legend.xlab,
                         .plt.leg = plt.l[[plt.idx]],
                         cex.lab = legend.lab.cex,
                         cex.axis = legend.axis.cex,
                         legend.fg = legend.fg,
                         legend.bg = legend.bg,
                         draw.legend.box.bool = draw.legend.box.bool,
                         v.adjust = NULL,
                         h.adjust = NULL )
        plt.idx <- plt.idx + 1
      } else if (! is.null( oneColorEncode.fun ) ){
        make1ColorLegend( numbers = numbers,
                          oneColorEncode.fun = oneColorEncode.fun,
                          cex.lab = legend.lab.cex,
                          cex.axis = legend.axis.cex,
                          legend.fg = legend.fg,
                          legend.bg = legend.bg,
                          n = colors.n,
                          lab = legend.ylab,
                          .plt.leg = plt.l[[plt.idx]],
                          draw.legend.box.bool = draw.legend.box.bool,
                          v.adjust = "center"
        )
        plt.idx <- plt.idx + 1
      }

      # Render Leaf Size Legend
      if( !is.null(sizeEncode.fun) ){
        legend.dat <- makeLeafSizeLegend( numbers = gs_numbers,
                                          sizeEncode.fun = sizeEncode.fun,
                                          #.plt = .plt2,
                                          .plt.leg = plt.l[[plt.idx]],
                                          legend.lab = n_col,
                                          pch = leaves_pch,
                                          legend.fg = legend.fg,
                                          legend.bg = legend.bg,
                                          legend.lab.cex = legend.lab.cex,
                                          cex.ticks = legend.axis.cex,
                                          leaf_border_color = leaf_border_color,
                                          draw.legend.box.bool = draw.legend.box.bool,
                                          v.adjust = NULL,
                                          h.adjust = NULL )
        plt.idx <- plt.idx + 1
      }
    }

    if( !is.null( main ) ) title( main = main,
                                  cex.main = cex.main,
                                  line = lines.main - if( geometry == "circular" ) mar.main else 0
    )

    if( geometry == "circular" ) circlize::circos.clear()
    close_fun() -> out
  }

  # Store plotting parameters as GSNA_plot_params attribute.
  attr( x = GSN.dend, which = "GSNA_plot_params" ) <- list( out_format = out_format,
                                                            geometry = geometry,
                                                            width = width,
                                                            height = height,
                                                            cex = cex,
                                                            font_face = font_face,
                                                            lab.cex = lab.cex,
                                                            leaf_cex = leaf_cex,
                                                            leaf_cex_range = leaf_cex_range,
                                                            leaf_colors = leaf_colors,
                                                            leaf_colors.1 = leaf_colors.1,
                                                            leaf_colors.2 = leaf_colors.2,
                                                            leaf_border_color = leaf_border_color,
                                                            subnet_colors = subnet_colors,
                                                            subnetColorsFunction = subnetColorsFunction,

                                                            .plt.plot = .plt.plot,
                                                            leaves_pch = leaves_pch,

                                                            leaf_char_shift = leaf_char_shift,
                                                            na.color = na.color,

                                                            tree_x_size.in = tree_x_size.in,

                                                            legend_x_size.in = legend_x_size.in,
                                                            left_margin.in = left_margin.in,
                                                            right_margin.in = right_margin.in,
                                                            top_margin.in = top_margin.in,
                                                            bottom_margin.in = bottom_margin.in,

                                                            legend.downshift.in = legend.downshift.in,
                                                            bkt_lmargin_chars = bkt_lmargin_chars,
                                                            legend_spacing.x.in = legend_spacing.x.in,
                                                            legend_spacing.y.in = legend_spacing.y.in,
                                                            legend.lab.cex = legend.lab.cex,
                                                            legend.axis.cex = legend.axis.cex,
                                                            legend.free.cex.bool = legend.free.cex.bool,
                                                            main = main,
                                                            cex.main = cex.main,
                                                            mar.main = mar.main,
                                                            lines.main = lines.main,
                                                            colors.n = colors.n,

                                                            legend.bg = legend.bg,
                                                            legend.fg = legend.fg,

                                                            draw.legend.box.bool = draw.legend.box.bool,
                                                            show.leaves = show.leaves,
                                                            show.legend = show.legend,

                                                            legend.leaf.col = legend.leaf.col,
                                                            combine_method = combine_method,
                                                            use_leaf_border = use_leaf_border,
                                                            render.plot = render.plot,
                                                            c1.fun = c1.fun,
                                                            c2.fun = c2.fun,
                                                            transform_function = transform_function,

                                                            id_col = id_col,
                                                            id_nchar = id_nchar,
                                                            pathways_title_col = pathways_title_col,
                                                            substitute_id_col = substitute_id_col,
                                                            color_labels_by = color_labels_by,
                                                            #pathways_dat = pathways_dat,
                                                            stat_col = stat_col,
                                                            stat_col_2 = stat_col_2,
                                                            sig_order = sig_order,
                                                            sig_order_2 = sig_order_2,
                                                            n_col = n_col

  )

  invisible( GSN.dend )
} # gsnHierarchicalDendrogram



