% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drvSucc.R
\name{drvSucc}
\alias{drvSucc}
\title{drvSucc : Computes the successive derivatives of a time series}
\usage{
drvSucc(tin = NULL, serie, nDeriv, weight = NULL, tstep = NULL, winL = 9)
}
\arguments{
\item{tin}{Input date vector which length should correspond to
the input time series.}

\item{serie}{A single time series provided as a single vector.}

\item{nDeriv}{The number of derivatives to be computed from
the input time series. The resulting number of
time series obtained in output will be \code{nDeriv + 1}.}

\item{weight}{A vector providing the binary weighting function
of the input data series (0 or 1). By default, all the values
are set to 1.}

\item{tstep}{Sampling time of the input time series. Used
only if time vector \code{tin} is not provided.}

\item{winL}{Number (exclusively odd number) of points of
the local window used for computing the derivatives along
the input time series. The Savitzky-Golay filter is used for
this purpose [1,2].}
}
\value{
A list containing:

$serie The original time serie

$tin The time vector containing the dates corresponding
to the original time series

$tstep The time step (assumed to be regular)

$tout The time vector of the output series

seriesDeriv A matrix containing the original time series
(smoothed by the filtering process) in the first column
and its \code{nDeriv + 1} successive derivatives in the next ones.
Note that \code{winL} values of the original time series will be lost,
that is \code{(winL - 1)/2} at the begining and \code{(winL - 1)/2}
at the end of the time series due to a computation boundary effect).
}
\description{
Computes the successive derivatives from one single time series,
using the Savitzky-Golay algorithm (1964).
}
\examples{
#############
# Example 1 #
#############
# Generate a time series:
tin <- seq(0, 5, by = 0.01)
data <- 2 * sin(5*tin)
dev.new()
par(mfrow = c(3, 1))
# Compute its derivatives:
drv <- drvSucc(tin = tin, nDeriv = 2, serie = data, winL = 5)
#
# plot original and filtered series
plot(tin, data, type='l', col = 'black', xlab = 't', ylab = 'x(t)')
lines(drv$tout, drv$seriesDeriv[,1], lty = 3, lwd = 3, col = 'green')
#
# analytic 1st derivative
firstD <- 10 * cos(5 * tin)
# plot both
plot(tin, firstD, type = 'l', col = 'black', xlab = 't', ylab = 'dx/dt')
lines(drv$tout, drv$seriesDeriv[,2], lty = 3, lwd = 3, col = 'green')
#
# analytic 2nd derivative
scdD <- -50 * sin(5 * tin)
# plot both
plot(tin, scdD, type = 'l', col = 'black', xlab = 't', ylab = 'd2x/dt2')
lines(drv$tout, drv$seriesDeriv[,3], lty=3, lwd = 3, col = 'green')

#############
# Example 2 #
#############
# load data:
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,2]

# Compute the derivatives
drvOut <- drvSucc(tin, data, nDeriv=4)
dev.new()
par(mfrow = c(3, 1))
# original and smoothed variable:
plot(drvOut$tin, drvOut$serie,
     type='p', cex = 1, xlab = 'time', ylab = 'x(t)')
lines(drvOut$tout, drvOut$seriesDeriv[,1], type='p', col='red')
lines(drvOut$tout, drvOut$seriesDeriv[,1], type='l', col='red')
# 1st derivative:
plot(drvOut$tout, drvOut$seriesDeriv[,2],
     type='p', col='red', xlab = 'time', ylab = 'dx(t)/dt')
lines(drvOut$tout, drvOut$seriesDeriv[,2], type='l', col='red')
# 2nd derivative:
plot(drvOut$tout, drvOut$seriesDeriv[,3],
     type='p', col='red', xlab = 'time', ylab = 'd2x(t)/dt2')
lines(drvOut$tout, drvOut$seriesDeriv[,3], type='l', col='red')

}
\references{
[1] Savitzky, A.; Golay, M.J.E.,
Smoothing and Differentiation of Data by Simplified Least Squares Procedures.
Analytical Chemistry 36 (8), 1627-1639, 1964.\cr
[2] Steinier J., Termonia Y., Deltour, J.
Comments on smoothing and differentiation of data by simplified least square procedure.
Analytical Chemistry 44 (11): 1906-1909, 1972. \cr
}
\seealso{
\code{\link{gloMoId}}, \code{\link{gPoMo}}, \code{\link{poLabs}}, \code{\link{compDeriv}}
}
\author{
Sylvain Mangiarotti, Mireille Huc
}
