#' @title
#' Calculate weights for estimation of a point on CERF
#'
#' @description
#' Calculates the weights of observed outcomes which is then used to estimate
#' the posterior mean of CERF at a given exposure level.
#'
#' @param w A scalar of exposure level of interest.
#' @param w_obs A vector of observed exposure levels of all samples.
#' @param scaled_obs A matrix of two columns.
#'   - First column is the scaled GPS value of all samples
#'   (GPS * 1 / sqrt(alpha))
#'   - Second column is the scaled exposure value of all samples
#'   (w * 1/sqrt(beta))
#' @param hyperparam A vector of hyper-parameters for the GP.
#'   - First element: alpha
#'   - Second element: beta
#'   - Third element: gamma/sigma
#' @param inv_sigma_obs Inverse of the covariance matrix between observed
#' samples.
#' @param GPS_m A data.frame of GPS vectors.
#'   - Column 1: A vector of estimated GPS evaluated at the observed exposure
#'   levels.
#'   - Column 2: Estimated conditional means of the exposure given covariates
#'               for all samples (e_gps_pred).
#'   - Column 3: Estimated conditional standard deviation of the exposure given
#'               covariates for all samples (e_gps_std).
#' @param est_sd Should the posterior se be computed (default=FALSE)
#' @param kernel_fn The covariance function of GP.
#'
#' @return
#' A list of two elements, weight and standard deviation.
#'
#' @keywords internal
#'
compute_weight_gp <- function(w, w_obs, scaled_obs, hyperparam,
                              inv_sigma_obs, GPS_m, est_sd = FALSE,
                              kernel_fn = function(x) exp(-x ^ 2)){

  alpha <- hyperparam[[1]]
  beta <- hyperparam[[2]]
  g_sigma <- hyperparam[[3]]

  # Compute GPS for requested w
  e_gps_pred <- GPS_m$e_gps_pred
  e_gps_std <- GPS_m$e_gps_std

  # TODO: The following section is repeated between this function
  # and compute_sd_gp function.
  GPS_w <- stats::dnorm(w, mean = e_gps_pred, sd = e_gps_std, log = TRUE)
  scaled_w <- cbind(w * sqrt(1 / alpha), GPS_w * sqrt(1 / beta))

  # kappa
  # sigma_cross = kappa/sigma^2 : Is always n*n matrix.
  # each column of sigma_cross is ki.
  # statspat.geom::crossdist
  sigma_cross <- g_sigma * kernel_fn(crossdist(scaled_w[, 1],
                                               scaled_w[, 2],
                                               scaled_obs[, 1],
                                               scaled_obs[, 2]))

  # each row is the weights for all subject for estimate of Y_i(w)
  # each column is the weight of an observed sample (w_i, c_i)
  normalized_sigma_cross <- Rfast::colmeans(sigma_cross)
  weight <- c(arma_mm(inv_sigma_obs, normalized_sigma_cross))

  # compute scaled posterior sd
  if (est_sd) {
    sigma_w <- g_sigma*kernel_fn(outer(scaled_w[,2], scaled_w[,2], "-")^2) +
      diag(nrow(scaled_w))
    sd_scaled = sqrt(sum(sigma_w)/nrow(scaled_w)^2 -
                     sum(weight*normalized_sigma_cross))
  } else {
    sd_scaled <- NA
  }

  return(list(weight = weight, sd_scaled = sd_scaled))
}
